#!/usr/bin/env python
# -*- coding: utf-8 -*-

# slab_crlb.py
# version 1.0: 2009-11-04
# version 1.1: 2011-02-02
# Copyright (C) 2009 Daniel Sjöberg, daniel.sjoberg@eit.lth.se
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see
# <http://www.gnu.org/licenses/>.


"""
This program illustrates the Cramer-Rao Lower Bound (CRLB) for
measurements of isotropic electromagnetic properties in a slab
subjected to electromagnetic waves in a waveguide, coaxial cable, or
free space. The CRLB shows how well the isotropic permittivity and
permeability can be determined if all systematic errors are
compensated for. The parameters that can be varied are permittivity,
permeability, thickness of sample, frequency, noise level, width of
waveguide, polarization, and angle of incidence.

In order to run this program, you need python, wxpython, matplotlib,
and numpy installed on your system (it was developed on an Ubuntu 9.04
platform with python 2.6.2, wxpython 2.8.9.1, matplotlib 0.98.5.2,
numpy 1.2.1). If you are running linux, please use your system's
package manager (synaptic under Ubuntu) to install these, otherwise
the packages are easily found on the web using any search engine.

Daniel Sjöberg, 2009-11-04.
"""


import wx

###############################
# Quote from Eli Bendersky's code wx_mpl_bars.py 2008-07-30: "The
# recommended way to use wx with mpl is with the WXAgg backend."
# It seems the backend must be set before importing pylab.
import matplotlib
matplotlib.use('WXAgg')
from matplotlib.figure import Figure
from matplotlib.backends.backend_wxagg import \
    FigureCanvasWxAgg as FigCanvas, \
    NavigationToolbar2WxAgg as NavigationToolbar
###############################

from pylab import *
from numpy.lib.scimath import sqrt  # Use complex sqrt by default


class PlotPanel(wx.Panel):
    """This class holds the plot window, using matplotlib as backend."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.Npoints = 100
        self.dp1 = linspace(0, 1, self.Npoints)   # Bound parameter 1
        self.dp2 = linspace(0, 1, self.Npoints)   # Bound parameter 2
        self.xdata = linspace(0, 1, self.Npoints) # Variable to plot against

        self.dpi = 100
        self.fig = Figure((5.0, 4.0), dpi=self.dpi)
        self.canvas = FigCanvas(self, -1, self.fig)

        self.axes = self.fig.add_subplot(111)
        self.fig.subplots_adjust(hspace=0.8)
        self.toolbar = NavigationToolbar(self.canvas)

        self.LegendText = [r'$\epsilon_{\mathrm{r}}$', r'$\mu_{\mathrm{r}}$']
        self.xlabel = 'Frequency (GHz)'
        self.GridOn = True
        self.ParameterListOn = True
        self.LegendOn = True
        self.ParameterList = ''

        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.canvas, 1, wx.LEFT | wx.TOP | wx.GROW)
        vSizer.Add(self.toolbar, 0, wx.EXPAND)
        vSizer.AddSpacer(10)

        self.SetSizer(vSizer)
        vSizer.Fit(self.parent)
        self.Update()

    def Update(self):
        self.axes.clear()
        self.axes.grid(self.GridOn)
        self.axes.plot(self.xdata, self.dp1, linewidth=2)
        self.axes.plot(self.xdata, self.dp2, linewidth=2)
        if self.LegendOn:
            self.axes.legend(self.LegendText, fancybox=True, shadow=True)
        self.axes.set_title(u'Cramér-Rao Lower Bound (dB)')
        self.axes.set_xlabel(self.xlabel)
        if self.ParameterListOn:
            self.axes.text(0.97, 0.03, self.ParameterList, 
                           bbox=dict(facecolor='white', alpha=0.5),
                           horizontalalignment='right',
                           verticalalignment='bottom',
                           multialignment='left',
                           fontsize='smaller',
                           transform=self.axes.transAxes)
        self.canvas.draw()


class TextPanel(wx.Panel):
    """A simple panel where a parameter can be set."""
    def __init__(self, parent, title, value):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.label = wx.StaticText(self, -1, title)
        self.input = wx.TextCtrl(self, wx.ID_ANY, str(value),
                                 style=wx.TE_PROCESS_ENTER)

        self.input.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.input.Bind(wx.EVT_TEXT_ENTER, self.Update)

        vsizer = wx.BoxSizer(wx.VERTICAL)
        vsizer.Add(self.label, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        vsizer.Add(self.input, 0, wx.ALIGN_CENTER | wx.ALL, 0)

        self.SetSizer(vsizer)

    def GetValue(self):
        return float(self.input.GetValue())

    def Update(self, event=None):
        event.Skip()


class MySlider(wx.Panel):
    """Create a slider where min and max values can be set."""
    def __init__(self, parent, labelname, min, max, rbstyle=None):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.labelname = labelname
        self.labelformat = "%s = %.2f"
        self.width = 350
        labeltext = self.labelformat % (self.labelname, (min+max)/2.0)
        self.label = wx.StaticText(self, -1, labeltext)
        self.min = min
        self.max = max
        self.slider = wx.Slider(self, -1, 50, 0, 100, (0,0), (self.width,20))
        self.mininput = wx.TextCtrl(self, wx.ID_ANY, str(min), 
                                    style=wx.TE_PROCESS_ENTER)
        self.maxinput = wx.TextCtrl(self, wx.ID_ANY, str(max), 
                                    style=wx.TE_PROCESS_ENTER)

        self.slider.Bind(wx.EVT_SLIDER, self.Update)
        self.mininput.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.maxinput.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.mininput.Bind(wx.EVT_TEXT_ENTER, self.Update)
        self.maxinput.Bind(wx.EVT_TEXT_ENTER, self.Update)

        self.slider.SetToolTip(wx.ToolTip('Slide to set variable'))
        self.mininput.SetToolTip(wx.ToolTip('Minimum value of variable'))
        self.maxinput.SetToolTip(wx.ToolTip('Maximum value of variable'))

        vsizer = wx.BoxSizer(wx.VERTICAL)
        vsizer.Add(self.slider, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        hsizer = wx.BoxSizer(wx.HORIZONTAL)
        hsizer.SetMinSize((self.width,20))
        hsizer.Add(self.mininput, 0, wx.ALIGN_LEFT | wx.ALL, 0)
        hsizer.AddStretchSpacer()
        hsizer.Add(self.label, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        hsizer.AddStretchSpacer()
        hsizer.Add(self.maxinput, 0, wx.ALIGN_RIGHT | wx.ALL, 0)
        vsizer.Add(hsizer, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        self.SetSizer(vsizer)

    def GetValue(self):
        slidervalue = float(wx.Slider.GetValue(self.slider))
        scale = (self.max-self.min)/100.0
        value = slidervalue*scale+self.min
        return value

    def SetValue(self, value):
        slidervalue = round((value-self.min)/(self.max-self.min)*100.)
        if slidervalue < 0:
            slidervalue = 0
        elif slidervalue > 100:
            slidervalue = 100
        self.slider.SetValue(slidervalue)
        self.Update()

    def Update(self, event=None):
        self.min = float(self.mininput.GetValue())
        self.max = float(self.maxinput.GetValue())
        self.value = self.GetValue()
        labeltext = self.labelformat % (self.labelname, self.value)
        self.label.Label = labeltext
        if event:
            event.Skip()


class InputPanel(wx.Panel):
    """Create a panel with input widgets."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.ReEpsSlider = MySlider(self, 'Re(eps)', 1, 10)
        self.ImEpsSlider = MySlider(self, 'Im(eps)', 0, 1)
        self.ImEpsSlider.SetValue(0)
        self.ReMuSlider = MySlider(self, 'Re(mu)', 1, 10)
        self.ReMuSlider.SetValue(1)
        self.ImMuSlider = MySlider(self, 'Im(mu)', 0, 1)
        self.ImMuSlider.SetValue(0)
        self.dSlider = MySlider(self, 'Thickness (mm)', 1, 10)
        self.fSlider = MySlider(self, 'Frequency (GHz)', 5, 15)
        self.thetaSlider = MySlider(self, 'Angle (degrees)', 0, 90)
        
        self.ReEpsSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.ImEpsSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.ReMuSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.ImMuSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.dSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.fSlider.Bind(wx.EVT_SLIDER, self.Update)
        self.thetaSlider.Bind(wx.EVT_SLIDER, self.Update)

        self.ReEpsRB = wx.RadioButton(self, -1, '', (10,10))
        self.ImEpsRB = wx.RadioButton(self, -1, '', (10,10))
        self.ReMuRB = wx.RadioButton(self, -1, '', (10,10))
        self.ImMuRB = wx.RadioButton(self, -1, '', (10,10))
        self.dRB = wx.RadioButton(self, -1, '', (10,10))
        self.fRB = wx.RadioButton(self, -1, '', (10,10))
        self.thetaRB = wx.RadioButton(self, -1, '', (10,10))

        self.ReEpsRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.ImEpsRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.ReMuRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.ImMuRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.dRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.fRB.SetToolTip(wx.ToolTip('Set x-axis variable'))
        self.thetaRB.SetToolTip(wx.ToolTip('Set x-axis variable'))

        self.ReEpsRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, 
                          id=self.ReEpsRB.GetId())
        self.ImEpsRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, 
                          id=self.ImEpsRB.GetId())
        self.ReMuRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, 
                         id=self.ReMuRB.GetId())
        self.ImMuRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, 
                         id=self.ImMuRB.GetId())
        self.dRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, id=self.dRB.GetId())
        self.fRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, id=self.fRB.GetId())
        self.thetaRB.Bind(wx.EVT_RADIOBUTTON, self.SetXaxis, id=self.thetaRB.GetId())

        ReEpsSizer = wx.BoxSizer(wx.HORIZONTAL)
        ReEpsSizer.Add(self.ReEpsRB)
        ReEpsSizer.Add(self.ReEpsSlider)

        ImEpsSizer = wx.BoxSizer(wx.HORIZONTAL)
        ImEpsSizer.Add(self.ImEpsRB)
        ImEpsSizer.Add(self.ImEpsSlider)

        ReMuSizer = wx.BoxSizer(wx.HORIZONTAL)
        ReMuSizer.Add(self.ReMuRB)
        ReMuSizer.Add(self.ReMuSlider)

        ImMuSizer = wx.BoxSizer(wx.HORIZONTAL)
        ImMuSizer.Add(self.ImMuRB)
        ImMuSizer.Add(self.ImMuSlider)

        dSizer = wx.BoxSizer(wx.HORIZONTAL)
        dSizer.Add(self.dRB)
        dSizer.Add(self.dSlider)

        fSizer = wx.BoxSizer(wx.HORIZONTAL)
        fSizer.Add(self.fRB)
        fSizer.Add(self.fSlider)

        thetaSizer = wx.BoxSizer(wx.HORIZONTAL)
        thetaSizer.Add(self.thetaRB)
        thetaSizer.Add(self.thetaSlider)

        SizerSliders = wx.BoxSizer(wx.VERTICAL)
        SizerSliders.Add(ReEpsSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(ImEpsSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(ReMuSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(ImMuSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(dSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(fSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        SizerSliders.Add(thetaSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)

        self.SetSizer(SizerSliders)

        self.eps = self.ReEpsSlider.GetValue() - 1j*self.ImEpsSlider.GetValue()
        self.mu = self.ReMuSlider.GetValue() - 1j*self.ImMuSlider.GetValue()
        self.d = self.dSlider.GetValue()
        self.f = self.fSlider.GetValue()
        self.theta = self.thetaSlider.GetValue()
        self.fRB.SetValue(True)
        self.Xaxis = 1
        self.SetXaxis()

    def Update(self, event=None):
        self.eps = self.ReEpsSlider.GetValue() - 1j*self.ImEpsSlider.GetValue()
        self.mu = self.ReMuSlider.GetValue() - 1j*self.ImMuSlider.GetValue()
        self.d = self.dSlider.GetValue()
        self.f = self.fSlider.GetValue()
        self.theta = self.thetaSlider.GetValue()
        event.Skip()

    def SetXaxis(self, event=None):
        if self.ReEpsRB.GetValue():
            self.Xaxis = 1
        elif self.ImEpsRB.GetValue():
            self.Xaxis = 2
        elif self.ReMuRB.GetValue():
            self.Xaxis = 3
        elif self.ImMuRB.GetValue():
            self.Xaxis = 4
        elif self.dRB.GetValue():
            self.Xaxis = 5
        elif self.fRB.GetValue():
            self.Xaxis = 6
        elif self.thetaRB.GetValue():
            self.Xaxis = 7
        if event:
            event.Skip()


class NumericalMenuItem(wx.MenuItem):
    """Create a menu item which can be used to set a numerical value."""
    def __init__(self, MainWindow, parent, id, variable, value, unit):
        title = '%s: %.2f %s' % (variable, value, unit)
        wx.MenuItem.__init__(self, parent, id, text=title)
        self.MainWindow = MainWindow
        self.variable = variable
        self.value = value
        self.unit = unit

    def SetValue(self, value):
        self.value = value
        title = '%s: %.2f %s' % (self.variable, self.value, self.unit)
        self.SetText(title)

    def OnSelection(self, event=None):
        """Set the value via a dialog box."""
        title = '%s (%s)' % (self.variable, self.unit)
        dlg = wx.TextEntryDialog(self.MainWindow, title, 'Enter value')
        dlg.SetValue(str(self.value))
        dlg.ShowModal()
        value = float(dlg.GetValue())
        self.SetValue(value)
        dlg.Destroy()
        event.Skip()

    def GetValue(self):
        return(self.value)


class MyFrame(wx.Frame):
    def __init__(self, parent, title):
        """Build the GUI."""
        wx.Frame.__init__(self, parent, -1, title)
        self.SetBackgroundColour('Normal')

        MenuBar = wx.MenuBar()
        FileMenu = wx.Menu()
        item = FileMenu.Append(wx.ID_EXIT, text="&Quit")
        self.Bind(wx.EVT_MENU, self.OnQuit, item)
        MenuBar.Append(FileMenu, "&File")

        ParameterMenu = wx.Menu()
        ExperimentSetupMenu = wx.Menu()
        self.SetupWaveguide = ExperimentSetupMenu.AppendRadioItem(-1, text='&Waveguide')
        self.SetupCoaxialCable = ExperimentSetupMenu.AppendRadioItem(-1, text='&Coaxial cable')
        self.SetupPlaneWave = ExperimentSetupMenu.AppendRadioItem(-1, text='&Plane wave')
        ParameterMenu.AppendMenu(-1, 'E&xperimental setup', ExperimentSetupMenu)

        DataMenu = wx.Menu()
        self.DataBoth = DataMenu.AppendRadioItem(-1, text='&Reflection and transmission')
        self.DataReflectionOnly = DataMenu.AppendRadioItem(-1, text='&Reflection data only')
        self.DataReflectionPEC = DataMenu.AppendRadioItem(-1, text='&Reflection data only (PEC backing)')
        self.DataTransmissionOnly = DataMenu.AppendRadioItem(-1, text='&Transmission data only')
        ParameterMenu.AppendMenu(-1, 'Available data', DataMenu)

        PolarizationMenu = wx.Menu()
        self.PolarizationTE = PolarizationMenu.AppendRadioItem(-1, text='TE')
        self.PolarizationTM = PolarizationMenu.AppendRadioItem(-1, text='TM')
        ParameterMenu.AppendMenu(-1, '&Polarization', PolarizationMenu)

        NoiseMenu = wx.Menu()
        self.ReflectionNoise = NumericalMenuItem(self, ParameterMenu, -1,
                                                'Reflection', 0, 'dB')
        NoiseMenu.AppendItem(self.ReflectionNoise)
        self.Bind(wx.EVT_MENU, self.ReflectionNoise.OnSelection, self.ReflectionNoise)
        self.TransmissionNoise = NumericalMenuItem(self, ParameterMenu, -1,
                                                'Transmission', 0, 'dB')
        NoiseMenu.AppendItem(self.TransmissionNoise)
        self.Bind(wx.EVT_MENU, self.TransmissionNoise.OnSelection, self.TransmissionNoise)
        ParameterMenu.AppendMenu(-1, '&Noise level', NoiseMenu)

        self.WaveguideWidth = NumericalMenuItem(self, ParameterMenu, -1,
                                                'Waveguide width', 22.9, 'mm')
        ParameterMenu.AppendItem(self.WaveguideWidth)
        self.Bind(wx.EVT_MENU, self.WaveguideWidth.OnSelection, self.WaveguideWidth)
        MenuBar.Append(ParameterMenu, '&Parameters')

        ViewMenu = wx.Menu()
        self.ShowGrid = ViewMenu.AppendCheckItem(-1, text='Show grid')
        self.ShowGrid.Check()
        self.ShowParameters = ViewMenu.AppendCheckItem(-1, text='Show parameters')
        self.ShowParameters.Check()
        self.ShowLegend = ViewMenu.AppendCheckItem(-1, text='Show legend')
        self.ShowLegend.Check()

        BoundMenu = wx.Menu()
        self.BoundMaterialParameters = BoundMenu.AppendRadioItem(-1, text='&Material parameters')
        self.BoundWaveParameters = BoundMenu.AppendRadioItem(-1, text='&Wave parameters')
        ViewMenu.AppendMenu(-1, 'Bound', BoundMenu)

        MenuBar.Append(ViewMenu, 'View')

        HelpMenu = wx.Menu()
        item = HelpMenu.Append(wx.ID_HELP, text="&About")
        self.Bind(wx.EVT_MENU, self.OnAbout, item)
        MenuBar.Append(HelpMenu, "&Help")
        self.SetMenuBar(MenuBar)

        self.Bind(wx.EVT_MENU, self.DimInactive)

        self.CRLBPanel = CRLBPanel(self)
        self.DimInactive()

        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.CRLBPanel)
        self.SetSizerAndFit(vSizer)

    def OnAbout(self, event=None):
        """Display some information about the program."""
        wx.MessageBox(u"""This program plots the Cramér-Rao lower bounds for measuring isotropic electromagnetic properties in a rectangular waveguide, coaxial cable, or free space setting, under the assumption of single mode propagation.

This is the best possible accuracy that can be obtained given a certain noise level, and requires all systematic errors have been removed.

Please send comments and suggestions to daniel.sjoberg@eit.lth.se.

Copyright (C) 2009 Daniel Sjöberg, version 1.1.""", 'Info', wx.OK, self)

    def OnQuit(self, event=None):
        """End the GUI."""
        ret  = wx.MessageBox('Are you sure you want to quit?', 'Question', 
                             wx.YES_NO | wx.NO_DEFAULT, self)
        if ret == wx.YES:
            self.Close()
        event.Skip()
        
    def DimInactive(self, event=None):
        """Dim elements which should not be active."""
        if self.SetupPlaneWave.IsChecked():
            self.CRLBPanel.InputPanel.thetaSlider.Enable(True)
            self.CRLBPanel.InputPanel.thetaRB.Enable(True)
            self.PolarizationTE.Enable(True)
            self.PolarizationTM.Enable(True)
        else:
            self.CRLBPanel.InputPanel.thetaSlider.Enable(False)
            self.CRLBPanel.InputPanel.thetaRB.Enable(False)
            self.PolarizationTE.Check()
            self.PolarizationTE.Enable(False)
            self.PolarizationTM.Enable(False)
        if self.SetupWaveguide.IsChecked():
            self.WaveguideWidth.Enable(True)
        else:
            self.WaveguideWidth.Enable(False)
        if event:
            event.Skip()


class CRLBPanel(wx.Panel):
    def __init__(self, parent):
        """Build the top panel to hold everything."""
        wx.Panel.__init__(self, parent)

        self.parent = parent
        self.parent.Bind(wx.EVT_MENU, self.Update)

        self.PlotPanel = PlotPanel(self)
        
        self.InputPanel = InputPanel(self)
        self.InputPanel.Bind(wx.EVT_SLIDER, self.Update)
        self.InputPanel.Bind(wx.EVT_RADIOBUTTON, self.Update)
        self.InputPanel.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.InputPanel.Bind(wx.EVT_TEXT_ENTER, self.Update)
        self.InputPanel.Bind(wx.EVT_CHECKBOX, self.Update)

        self.Title = wx.StaticText(self, -1, 
                                   'Measurement of isotropic slab')
        self.Title.SetFont(wx.Font(12, -1, wx.FONTSTYLE_NORMAL, 
                                    wx.FONTWEIGHT_BOLD))

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(self.PlotPanel, 0, wx.ALIGN_CENTER)
        hSizer.Add(self.InputPanel, 0, wx.ALIGN_CENTER)
        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.Title, 0, wx.ALIGN_CENTER | wx.ALL, 10)
        vSizer.Add(hSizer, 0, wx.ALIGN_CENTER)
        self.SetSizerAndFit(vSizer)

        self.Update()

    def Update(self, event=None):
        eps = self.InputPanel.eps
        mu = self.InputPanel.mu
        d = self.InputPanel.d
        f = self.InputPanel.f
        theta = self.InputPanel.theta
        rNoise = self.parent.ReflectionNoise.GetValue()
        tNoise = self.parent.TransmissionNoise.GetValue()
        a = self.parent.WaveguideWidth.GetValue()
        Npoints = self.PlotPanel.Npoints

        self.PlotPanel.ParameterList = r"""
$\epsilon_{\mathrm{r}}$ = %.2f-j%.2f
$\mu_{\mathrm{r}}$ = %.2f-j%.2f
$d$ = %.2f mm
$f$ = %.2f GHz
$\theta$ = %.2f deg""" % (real(eps), -imag(eps), real(mu), -imag(mu), d, f, theta)

        if self.InputPanel.Xaxis == 7 and not self.parent.SetupPlaneWave.IsChecked():
            self.InputPanel.fRB.SetValue(True)
            self.InputPanel.SetXaxis()

        if self.InputPanel.Xaxis==1:
            ReEps = linspace(self.InputPanel.ReEpsSlider.min,
                             self.InputPanel.ReEpsSlider.max, Npoints)
            eps = ReEps - 1j*self.InputPanel.ImEpsSlider.GetValue()
            self.PlotPanel.xdata = ReEps
            self.PlotPanel.xlabel = r'$\mathrm{Re}(\epsilon_{\mathrm{r}})$'
        elif self.InputPanel.Xaxis==2:
            ImEps = linspace(self.InputPanel.ImEpsSlider.min,
                             self.InputPanel.ImEpsSlider.max, Npoints)
            eps = self.InputPanel.ReEpsSlider.GetValue() - 1j*ImEps
            self.PlotPanel.xdata = ImEps
            self.PlotPanel.xlabel = r'$\mathrm{Im}(\epsilon_{\mathrm{r}})$'
        elif self.InputPanel.Xaxis==3:
            ReMu = linspace(self.InputPanel.ReMuSlider.min,
                            self.InputPanel.ReMuSlider.max, Npoints)
            mu = ReMu - 1j*self.InputPanel.ImMuSlider.GetValue()
            self.PlotPanel.xdata = ReMu
            self.PlotPanel.xlabel = r'$\mathrm{Re}(\mu_{\mathrm{r}})$'
        elif self.InputPanel.Xaxis==4:
            ImMu = linspace(self.InputPanel.ImMuSlider.min,
                            self.InputPanel.ImMuSlider.max, Npoints)
            mu = self.InputPanel.ReMuSlider.GetValue() - 1j*ImMu
            self.PlotPanel.xdata = ImMu
            self.PlotPanel.xlabel = r'$\mathrm{Im}(\mu_{\mathrm{r}})$'
        elif self.InputPanel.Xaxis==5:
            d = linspace(self.InputPanel.dSlider.min,
                         self.InputPanel.dSlider.max, Npoints)
            self.PlotPanel.xdata = d
            self.PlotPanel.xlabel = 'Thickness (mm)'
        elif self.InputPanel.Xaxis==6:
            f = linspace(self.InputPanel.fSlider.min,
                         self.InputPanel.fSlider.max, Npoints)
            self.PlotPanel.xdata = f
            self.PlotPanel.xlabel = 'Frequency (GHz)'
        else:
            theta = linspace(self.InputPanel.thetaSlider.min,
                             self.InputPanel.thetaSlider.max, Npoints)
            self.PlotPanel.xdata = theta
            self.PlotPanel.xlabel = 'Angle of incidence (degrees)'
            
        if self.parent.SetupWaveguide.IsChecked():
            ExperimentSetup = 'Waveguide'
        elif self.parent.SetupCoaxialCable.IsChecked():
            ExperimentSetup = 'CoaxialCable'
        else:
            ExperimentSetup = 'PlaneWave'
        if self.parent.PolarizationTM.IsChecked():
            Polarization = 'TM'
        else:
            Polarization = 'TE'
        if self.parent.DataBoth.IsChecked():
            DataAvailable = 'Both'
        elif self.parent.DataReflectionOnly.IsChecked():
            DataAvailable = 'ReflectionOnly'
        elif self.parent.DataReflectionPEC.IsChecked():
            DataAvailable = 'ReflectionPEC'
        else:
            DataAvailable = 'TransmissionOnly'
        if self.parent.BoundMaterialParameters.IsChecked():
            WhichBound = 'MaterialParameters'
            self.PlotPanel.LegendText = [r'$\epsilon_{\mathrm{r}}$', 
                                         r'$\mu_{\mathrm{r}}$']
        else:
            WhichBound = 'WaveParameters'
            self.PlotPanel.LegendText = [r'$\beta/k_{0}$', r'$Z/\eta_{0}$']
            

        dp1, dp2 = self.ComputeCRLB(eps, mu, d, f, rNoise, tNoise, a, theta,
                                    ExperimentSetup, Polarization, 
                                    DataAvailable, WhichBound)

        self.PlotPanel.dp1 = dp1
        self.PlotPanel.dp2 = dp2
        self.PlotPanel.GridOn = self.parent.ShowGrid.IsChecked()
        self.PlotPanel.ParameterListOn = self.parent.ShowParameters.IsChecked()
        self.PlotPanel.LegendOn = self.parent.ShowLegend.IsChecked()

        self.PlotPanel.Update()
        if event:
            event.Skip()

    def ComputeCRLB(self, eps, mu, d_mm, f_GHz, rNoise_dB, tNoise_dB, 
                    a_mm, theta_degrees, ExperimentSetup, Polarization, 
                    DataAvailable, WhichBound):
        rNoise = exp(log(10)*rNoise_dB/10.)
        tNoise = exp(log(10)*tNoise_dB/10.)
        d = d_mm*1e-3
        f = f_GHz*1e9
        a = a_mm*1e-3 
        theta = theta_degrees*pi/180.

        c0 = 299792458.
        omega = 2.*pi*f
        eta0 = 1.
        k0 = omega/c0
        k = k0*sqrt(eps*mu)
        if ExperimentSetup == 'Waveguide':
            kt = pi/a
        elif ExperimentSetup == 'PlaneWave':
            kt = k0*sin(theta)
        else:   # The coaxial cable or normal incidence
            kt = 0
        beta0 = sqrt(k0**2-kt**2)
        beta = sqrt(k**2-kt**2)
        if Polarization == 'TM':
            Z0 = beta0/k0*eta0
            Z = beta/k*sqrt(mu/eps)*eta0
        else:   # TE is the default
            Z0 = k0/beta0*eta0
            Z = k/beta*sqrt(mu/eps)*eta0

        r0 = (Z-Z0)/(Z+Z0)

        expf2 = exp(-2*1j*beta*d)
        expf1 = exp(-1j*beta*d)
        A = 1 - expf2
        B = 1 - r0*r0*expf2
        C = (1 - r0*r0)*expf1
        if DataAvailable == 'ReflectionPEC':
            drdr0 = 1/(1-r0*expf2) + (r0-expf2)*expf2/((1-r0*expf2)**2)
            drdb = 2j*d*(expf2/(1-r0*expf2) - (r0-expf2)*2*expf2/((1-r0*expf2)**2))
            dtdr0 = zeros(len(drdr0))
            dtdb = zeros(len(drdr0))
        else:
            drdr0 = A/B + 2.*r0*r0*expf2*A/(B*B)
            drdb = 2j*d*(r0*expf2/B - r0*A*r0*r0*expf2/(B*B))
            dtdr0 = -2*r0*expf1/B + 2*r0*expf2*C/(B*B)
            dtdb = 1j*d*(-C/B - C*r0*r0*2*expf2/(B*B))

        dbde = k**2/(2*beta*eps)
        dbdm = k**2/(2*beta*mu)
        if Polarization == 'TM':
            dZde = -Z/eps + Z/beta*dbde
            dZdm = Z/beta*dbdm
        else:  # TE is default
            dZde = -Z/beta*dbde
            dZdm = Z/mu - Z/beta*dbdm
        dr0dZ = 2*Z0/((Z+Z0)**2)

        if WhichBound == 'MaterialParameters':
            drdp1 = drdr0*dr0dZ*dZde + drdb*dbde   # Permittivity
            drdp2 = drdr0*dr0dZ*dZdm + drdb*dbdm   # Permeability
            dtdp1 = dtdr0*dr0dZ*dZde + dtdb*dbde   # Permittivity
            dtdp2 = dtdr0*dr0dZ*dZdm + dtdb*dbdm   # Permeability
        else:
            drdp1 = k0*drdb            # Refractive index
            drdp2 = eta0*drdr0*dr0dZ   # Normalized impedance
            dtdp1 = k0*dtdb            # Refractive index
            dtdp2 = eta0*dtdr0*dr0dZ   # Normalized impedance

        if DataAvailable == 'Both':
            dp1_dB = array(drdp1)
            dp2_dB = array(drdp1)
            for k in range(0, drdp1.size, 1):
                rv = matrix([drdp1[k], drdp2[k]])
                tv = matrix([dtdp1[k], dtdp2[k]])
                FIMr = 2./(rNoise**2)*transpose(conjugate(rv))*rv
                FIMt = 2./(tNoise**2)*transpose(conjugate(tv))*tv
                FIM = FIMr + FIMt
                var = real(inv(FIM))
                dp1_dB[k] = 10*log10(sqrt(var[0,0]))
                dp2_dB[k] = 10*log10(sqrt(var[1,1]))
        elif DataAvailable == 'ReflectionOnly' or DataAvailable == 'ReflectionPEC':
            FIp1 = 2*(abs(drdp1)/rNoise)**2
            FIp2 = 2*(abs(drdp2)/rNoise)**2
            dp1_dB = 10*log10(sqrt(1/FIp1))
            dp2_dB = 10*log10(sqrt(1/FIp2))
        elif DataAvailable == 'TransmissionOnly':
            FIp1 = 2*(abs(dtdp1)/tNoise)**2
            FIp2 = 2*(abs(dtdp2)/tNoise)**2
            dp1_dB = 10*log10(sqrt(1/FIp1))
            dp2_dB = 10*log10(sqrt(1/FIp2))

        return(real(dp1_dB), real(dp2_dB))


if __name__ == '__main__':
    app = wx.PySimpleApp()
    frame = MyFrame(None, title='CRLB')
    frame.Show()
    app.MainLoop()
