#!/usr/bin/env python
# -*- coding: utf-8 -*-

# antennaboundsdemo.py, version 1.0
# Copyright (C) 2010 Daniel Sjöberg, daniel.sjoberg@eit.lth.se
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see
# <http://www.gnu.org/licenses/>.

"""
This program produces a graphical demonstration of physical bounds on
antennas. The bounds themselves are based on results by Mats
Gustafsson et al at Lund University, Sweden.

Daniel Sjoberg, 2010-01-27
"""


import wx

###############################
# Quote from Eli Bendersky's code wx_mpl_bars.py 2008-07-30: "The
# recommended way to use wx with mpl is with the WXAgg backend."
# It seems the backend must be set before importing pylab.
import matplotlib
matplotlib.use('WXAgg')
from matplotlib.figure import Figure
from matplotlib.backends.backend_wxagg import \
    FigureCanvasWxAgg as FigCanvas, \
    NavigationToolbar2WxAgg as NavigationToolbar
###############################

from pylab import *
from numpy.lib.scimath import sqrt  # Use complex sqrt by default

from AntennaQ import AntennaQ 

# Data describing the curves: (name)
curves_info = [('Rectangle', 'b-', 'rec'), 
               ('Cylinder', 'r-', 'cyl'),
               ('Spheroid', 'g-', 'sph')]
# Data describing the variables: (name, unit, scale, start value, end value)
# NB: scale and unit are not properly implemented yet!
variables_info = [('length (mm)', 'mm', 1, 0, 10),
                  ('width (mm)', 'mm', 1, 0, 10),
                  ('frequency (GHz)', 'GHz', 1e9, 8, 12)]

class PlotPanel(wx.Panel):
    """This class holds the plot window, using matplotlib as backend."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.Npoints = 100
        self.ydata = []
        for n in arange(0, len(curves_info)):
            self.ydata.append(linspace(0, 1, self.Npoints))
        self.xdata = linspace(0, 1, self.Npoints) # Variable to plot against

        self.dpi = 100
        self.fig = Figure((5.0, 4.0), dpi=self.dpi)
        self.canvas = FigCanvas(self, -1, self.fig)

        self.axes = self.fig.add_subplot(111)
        self.fig.subplots_adjust(hspace=0.8)
        self.toolbar = NavigationToolbar(self.canvas)

        self.xlabel = ''
        self.ylabel = ''
        self.GridOn = True

        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.canvas, 1, wx.LEFT | wx.TOP | wx.GROW)
        vSizer.Add(self.toolbar, 0, wx.EXPAND)
        vSizer.AddSpacer(10)

        self.SetSizer(vSizer)
        vSizer.Fit(self.parent)
        self.Update()

    def Update(self):
        self.axes.clear()
        self.axes.grid(self.GridOn)
        for n in arange(0, len(curves_info)):
            if self.parent.InputPanel.GeometriesPanel.curves[n].GetValue():
                name, linestyle, geo = curves_info[n]
                self.axes.plot(self.xdata, self.ydata[n], linestyle, 
                               label=name, linewidth=2)
#        self.axes.set_title(u'')
        self.axes.set_xlabel(self.xlabel)
        self.axes.set_ylabel(self.ylabel)
        self.axes.legend(loc='best')
        self.canvas.draw()


class TextPanel(wx.Panel):
    """A simple panel where a parameter can be set."""
    def __init__(self, parent, title, value):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.label = wx.StaticText(self, -1, title)
        self.input = wx.TextCtrl(self, wx.ID_ANY, str(value),
                                 style=wx.TE_PROCESS_ENTER)

        self.input.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.input.Bind(wx.EVT_TEXT_ENTER, self.Update)

        vsizer = wx.BoxSizer(wx.VERTICAL)
        vsizer.Add(self.label, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        vsizer.Add(self.input, 0, wx.ALIGN_CENTER | wx.ALL, 0)

        self.SetSizer(vsizer)

    def GetValue(self):
        return float(self.input.GetValue())

    def Update(self, event=None):
        event.Skip()


class MySlider(wx.Panel):
    """Create a slider where min and max values can be set."""
    def __init__(self, parent, labelname, min, max, rbstyle=None):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.labelname = labelname
        self.labelformat = "%s = %.2f"
        self.width = 350
        labeltext = self.labelformat % (self.labelname, (min+max)/2.0)
        self.label = wx.StaticText(self, -1, labeltext)
        self.min = min
        self.max = max
        self.slider = wx.Slider(self, -1, 50, 0, 100, (0,0), (self.width,20))
        self.mininput = wx.TextCtrl(self, wx.ID_ANY, str(min), 
                                    style=wx.TE_PROCESS_ENTER)
        self.maxinput = wx.TextCtrl(self, wx.ID_ANY, str(max), 
                                    style=wx.TE_PROCESS_ENTER)

        self.slider.Bind(wx.EVT_SLIDER, self.Update)
        self.mininput.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.maxinput.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.mininput.Bind(wx.EVT_TEXT_ENTER, self.Update)
        self.maxinput.Bind(wx.EVT_TEXT_ENTER, self.Update)

        self.slider.SetToolTip(wx.ToolTip('Slide to set variable'))
        self.mininput.SetToolTip(wx.ToolTip('Minimum value of variable'))
        self.maxinput.SetToolTip(wx.ToolTip('Maximum value of variable'))

        vsizer = wx.BoxSizer(wx.VERTICAL)
        vsizer.Add(self.slider, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        hsizer = wx.BoxSizer(wx.HORIZONTAL)
        hsizer.SetMinSize((self.width,20))
        hsizer.Add(self.mininput, 0, wx.ALIGN_LEFT | wx.ALL, 0)
        hsizer.AddStretchSpacer()
        hsizer.Add(self.label, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        hsizer.AddStretchSpacer()
        hsizer.Add(self.maxinput, 0, wx.ALIGN_RIGHT | wx.ALL, 0)
        vsizer.Add(hsizer, 0, wx.ALIGN_CENTER | wx.ALL, 0)
        self.SetSizer(vsizer)

    def GetValue(self):
        slidervalue = float(wx.Slider.GetValue(self.slider))
        scale = (self.max-self.min)/100.0
        value = slidervalue*scale+self.min
        return value

    def SetValue(self, value):
        slidervalue = round((value-self.min)/(self.max-self.min)*100.)
        if slidervalue < 0:
            slidervalue = 0
        elif slidervalue > 100:
            slidervalue = 100
        self.slider.SetValue(slidervalue)
        self.Update()

    def Update(self, event=None):
        self.min = float(self.mininput.GetValue())
        self.max = float(self.maxinput.GetValue())
        self.value = self.GetValue()
        labeltext = self.labelformat % (self.labelname, self.value)
        self.label.Label = labeltext
        if event:
            event.Skip()

class RectangleDrawingPanel(wx.Panel):
    """Draw a rectangle with the correct dimensions."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent, size=(30, 95))
        self.parent = parent
        self.Bind(wx.EVT_PAINT, self.OnPaint)

    def OnPaint(self, event=None):
        self.dc = wx.PaintDC(self)
        self.dc.Clear()
        self.dc.BeginDrawing()
        self.dc.SetPen(wx.Pen('BLACK', 1))
        h = self.parent.parent.VariablesPanel.variables[0]
        w = self.parent.parent.VariablesPanel.variables[1]
        a = sqrt(w**2 + h**2)/2.
        scale = 30
        if a>0:
            w = round(scale*w/(2*a))
            h = round(scale*h/(2*a))
        else:
            w = 0
            h = 0

        x0 = 0
        y0 = 0
        x1 = x0
        y1 = y0 + scale + 2
        x2 = x0
        y2 = y1 + scale + 2

        self.dc.DrawRectangle(x0, y0, w, h)

        self.dc.DrawEllipse(x1, y1-2, w, 4)
        self.dc.DrawEllipticArc(x1, y1+h-2, w, 4, 0, -180)
        self.dc.DrawLine(x1, y1, x1, y1+h)
        self.dc.DrawLine(x1+w, y1, x1+w, y1+h)

        self.dc.DrawEllipse(x2, y2, w, h)
        self.dc.DrawEllipticArc(x2, y2+round(h/2)-2, w, 4, 0, -180)
        self.dc.EndDrawing()
        del self.dc
        if event:
            event.Skip()

class GeometriesPanel(wx.Panel):
    """Create a panel to choose geometry."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.curves = []
        for n in arange(0, len(curves_info)):
            name = curves_info[n][0]
            self.curves.append(wx.CheckBox(self, -1, name))
            self.curves[n].SetValue(True)
        
        vSizer = wx.BoxSizer(wx.VERTICAL)
        title = wx.StaticText(self, -1, 'Geometries')
        title.SetFont(wx.Font(10, -1, wx.NORMAL, wx.BOLD))
        vSizer.Add(title, 0, wx.ALIGN_LEFT | wx.ALL, 5)

        vSizer2 = wx.BoxSizer(wx.VERTICAL)
        for n in arange(0, len(curves_info)):
            vSizer2.Add(self.curves[n], 0, wx.ALIGN_LEFT | wx.ALL, 5)

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(vSizer2, 0, wx.ALIGN_TOP | wx.ALL, 5)
        self.drawing = RectangleDrawingPanel(self)
        hSizer.Add(self.drawing, 0, wx.ALIGN_TOP | wx.ALL, 5)

        vSizer.Add(hSizer, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        self.SetSizer(vSizer)

class PolarizationPanel(wx.Panel):
    """Create a panel to choose geometry."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.vertical = wx.RadioButton(self, -1, 'Vertical', (10,10))
        self.horizontal = wx.RadioButton(self, -1, 'Horizontal', (10,10))
        self.vertical.SetValue(True)

        vSizer = wx.BoxSizer(wx.VERTICAL)
        title = wx.StaticText(self, -1, 'Polarization')
        title.SetFont(wx.Font(10, -1, wx.NORMAL, wx.BOLD))
        vSizer.Add(title, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.vertical, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.horizontal, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        self.SetSizer(vSizer)

class BoundsPanel(wx.Panel):
    """Create a panel to choose bound."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.DQ = wx.RadioButton(self, -1, 'D/Q', (10,10))
        self.Q = wx.RadioButton(self, -1, 'Q', (10,10))
        self.gamma = wx.RadioButton(self, -1, 'Polarizability', (10,10))
        self.DQ.SetValue(True)

        vSizer = wx.BoxSizer(wx.VERTICAL)
        title = wx.StaticText(self, -1, 'Bound')
        title.SetFont(wx.Font(10, -1, wx.NORMAL, wx.BOLD))
        vSizer.Add(title, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.DQ, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.Q, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.gamma, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        self.SetSizer(vSizer)

class ScalePanel(wx.Panel):
    """Create a panel to choose scale."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        self.linear = wx.RadioButton(self, -1, 'Linear', (10,10))
        self.dB = wx.RadioButton(self, -1, 'dB', (10,10))
        self.linear.SetValue(True)

        vSizer = wx.BoxSizer(wx.VERTICAL)
        title = wx.StaticText(self, -1, 'Scale')
        title.SetFont(wx.Font(10, -1, wx.NORMAL, wx.BOLD))
        vSizer.Add(title, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.linear, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        vSizer.Add(self.dB, 0, wx.ALIGN_LEFT | wx.ALL, 5)
        self.SetSizer(vSizer)

class VariablesPanel(wx.Panel):
    """Create a panel with input widgets."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent

        Nvars = len(variables_info)
        self.Sliders = []
        self.RBs = []
        self.variableSizers = []
        for n in arange(0, Nvars):
            name, unit, scale, start, stop = variables_info[n]
            self.Sliders.append(MySlider(self, name, start, stop))
            self.RBs.append(wx.RadioButton(self, -1, '', (10,10)))
            self.variableSizers.append(wx.BoxSizer(wx.HORIZONTAL))

            self.Sliders[n].Bind(wx.EVT_SLIDER, self.Update)
            self.RBs[n].SetToolTip(wx.ToolTip('Set x-axis variable'))
            self.RBs[n].Bind(wx.EVT_RADIOBUTTON, self.Update, 
                             id=self.RBs[n].GetId())
            self.variableSizers[n].Add(self.RBs[n])
            self.variableSizers[n].Add(self.Sliders[n])

        vSizer = wx.BoxSizer(wx.VERTICAL)
        title = wx.StaticText(self, -1, 'Variables')
        title.SetFont(wx.Font(10, -1, wx.NORMAL, wx.BOLD))
        vSizer.Add(title, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        for n in arange(0, Nvars):
            vSizer.Add(self.variableSizers[n], 0, 
                             wx.ALIGN_CENTER | wx.ALL, 5)
        self.SetSizer(vSizer)

        self.RBs[0].SetValue(True)
        self.variables = []
        for n in arange(0, Nvars):
            self.variables.append(self.Sliders[n].GetValue())

    def Update(self, event=None):
        for n in arange(0, len(variables_info)):
            self.variables[n] = self.Sliders[n].GetValue()
        if event:
            event.Skip()

class InputPanel(wx.Panel):
    """Create a panel with input widgets."""
    def __init__(self, parent):
        wx.Panel.__init__(self, parent)
        self.parent = parent
        
        self.GeometriesPanel = GeometriesPanel(self)
        self.PolarizationPanel = PolarizationPanel(self)
        self.BoundsPanel = BoundsPanel(self)
        self.ScalePanel = ScalePanel(self)
        self.VariablesPanel = VariablesPanel(self)

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(self.GeometriesPanel, 0, wx.ALIGN_TOP | wx.ALL, 5)
        hSizer.Add(self.PolarizationPanel, 0, wx.ALIGN_TOP | wx.ALL, 5)
        hSizer.Add(self.BoundsPanel, 0, wx.ALIGN_TOP | wx.ALL, 5)
        hSizer.Add(self.ScalePanel, 0, wx.ALIGN_TOP | wx.ALL, 5)

        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(hSizer, 0, wx.ALIGN_CENTER | wx.ALL, 5)
        vSizer.Add(self.VariablesPanel, 0, wx.ALIGN_CENTER | wx.ALL, 5)

        self.SetSizer(vSizer)

        self.Bind(wx.EVT_SLIDER, self.GeometriesPanel.drawing.OnPaint)

class MyFrame(wx.Frame):
    def __init__(self, parent, title):
        """Build the GUI."""
        wx.Frame.__init__(self, parent, -1, title)
        self.SetBackgroundColour('Normal')

        MenuBar = wx.MenuBar()
        FileMenu = wx.Menu()
        item = FileMenu.Append(wx.ID_EXIT, text="&Quit")
        self.Bind(wx.EVT_MENU, self.OnQuit, item)
        MenuBar.Append(FileMenu, "&File")

        HelpMenu = wx.Menu()
        item = HelpMenu.Append(wx.ID_HELP, text="&About")
        self.Bind(wx.EVT_MENU, self.OnAbout, item)
        MenuBar.Append(HelpMenu, "&Help")
        self.SetMenuBar(MenuBar)

        self.MainPanel = MainPanel(self)

        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.MainPanel)
        self.SetSizerAndFit(vSizer)

    def OnAbout(self, event=None):
        """Display some information about the program."""
        wx.MessageBox(u"""This program helps visualize various bounds on bandwidth, directivity, quality factor etc.

Bounds by Mats Gustafsson, GUI by Daniel Sjöberg, 2010-01-27.""", 'Info', wx.OK, self)

    def OnQuit(self, event=None):
        """End the GUI."""
        ret  = wx.MessageBox('Are you sure you want to quit?', 'Question', 
                             wx.YES_NO | wx.NO_DEFAULT, self)
        if ret == wx.YES:
            self.Close()
        event.Skip()
        


class MainPanel(wx.Panel):
    def __init__(self, parent):
        """Build the top panel to hold everything."""
        wx.Panel.__init__(self, parent)

        self.parent = parent
        self.parent.Bind(wx.EVT_MENU, self.Update)

        self.InputPanel = InputPanel(self)
        self.InputPanel.Bind(wx.EVT_SLIDER, self.Update)
        self.InputPanel.Bind(wx.EVT_RADIOBUTTON, self.Update)
        self.InputPanel.Bind(wx.EVT_KILL_FOCUS, self.Update)
        self.InputPanel.Bind(wx.EVT_TEXT_ENTER, self.Update)
        self.InputPanel.Bind(wx.EVT_CHECKBOX, self.Update)

        self.PlotPanel = PlotPanel(self)
        
        self.Title = wx.StaticText(self, -1, 'Interactive antenna bounds')
        self.Title.SetFont(wx.Font(12, -1, wx.FONTSTYLE_NORMAL, 
                                    wx.FONTWEIGHT_BOLD))

        hSizer = wx.BoxSizer(wx.HORIZONTAL)
        hSizer.Add(self.PlotPanel, 0, wx.ALIGN_CENTER)
        hSizer.Add(self.InputPanel, 0, wx.ALIGN_CENTER)
        vSizer = wx.BoxSizer(wx.VERTICAL)
        vSizer.Add(self.Title, 0, wx.ALIGN_CENTER | wx.ALL, 10)
        vSizer.Add(hSizer, 0, wx.ALIGN_CENTER)
        self.SetSizerAndFit(vSizer)

        self.Update()

    def Update(self, event=None):
        Npoints = self.PlotPanel.Npoints

        locvars = []
        for nn in arange(0, len(variables_info)):
            locvars.append(self.InputPanel.VariablesPanel.variables[nn])
            if self.InputPanel.VariablesPanel.RBs[nn].GetValue():
                n = nn
        locvars[n] = linspace(self.InputPanel.VariablesPanel.Sliders[n].min, 
                              self.InputPanel.VariablesPanel.Sliders[n].max, 
                              Npoints)
        self.PlotPanel.xdata = locvars[n]
        self.PlotPanel.xlabel = variables_info[n][0]

        l, d, f = locvars
        # Units hardcoded so far
        l = l*1e-3
        d = d*1e-3
        f = f*1e9

        if self.InputPanel.PolarizationPanel.horizontal.GetValue():
            pol = '_h'
        else:
            pol = '_v'
        for n in arange(0, len(curves_info)):
            if self.InputPanel.GeometriesPanel.curves[n].GetValue():
                geo = curves_info[n][2] + pol
                DQ, Q, xi, a, ka, gamma = AntennaQ(l, d, f, geo)
                if self.InputPanel.BoundsPanel.DQ.GetValue():
                    ydata = DQ
                    ylabel = 'D/Q'
                if self.InputPanel.BoundsPanel.Q.GetValue():
                    ydata = Q
                    ylabel = 'Q'
                if self.InputPanel.BoundsPanel.gamma.GetValue():
                    ydata = gamma
                    ylabel = 'Polarizability'

                if self.InputPanel.ScalePanel.dB.GetValue():
                    self.PlotPanel.ydata[n] = 10*log10(abs(ydata))
                    self.PlotPanel.ylabel = ylabel + ' (dB)'
                else:
                    self.PlotPanel.ydata[n] = ydata
                    self.PlotPanel.ylabel = ylabel
                
        self.PlotPanel.Update()
        if event:
            event.Skip()


if __name__ == '__main__':
    app = wx.PySimpleApp()
    frame = MyFrame(None, title='Antenna bounds')
    frame.Show()
    app.MainLoop()
