# AntennaQ   determines physical bounds on Q and D/Q
#
# Original code by Mats Gustafsson, 2010-01-26
# Ported to python by Daniel Sjoberg, 2010-01-27


### Examples: 
#   [DQ,Q,xi,a,ka,gamma] = AntennaQ(l,d, geo, f, eta, fig)
#   [DQ,Q,xi,a,ka,gamma] = AntennaQ(logspace(-2,2,301), 1, 'rec');
#   [DQ,Q,xi,a,ka,gamma] = AntennaQ(1, 1, 'cyl_v', 1e6);
#   [DQ,Q,xi,a,ka,gamma] = AntennaQ(0.05, 0.02, 'spheroid_h', 1e9, 0.5)
#
# This function calculates physical bounds on Q and D/Q for linearly
# polarized antennas composed by non-magnetic materials
# and circumscribed by various geometries as described in: 
# Gustafsson, Sohl, and Kristensson
# Illustrations of new physical bounds on linearly polarized antennas, 
# IEEE Trans. Antennas Propagat., Vol. 57, No. 5, pp. 1319-1327, 2009.
# http://dx.doi.org/10.1109/TAP.2009.2016683
# and
# Physical limitations on antennas of arbitrary shape, 
# Proc. Roy. Soc. A, Vol. 463, No. 2086, pp. 2589-2607, 2007. 
# http://dx.doi.org/10.1098/rspa.2007.1893
#
# The polarizability is based on analytic results for spheroids and approximations of integral equation (MoM) data with rational
# functions for cylinders and rectangles. 
#
# [DQ,Q,xi,a,ka,gamma] = AntennaQ(l,d, geo, f, eta, fig)
### Input parameters
# l: vertical length or height (in m if used with frequency in Hz) (vector)
# d: horizontal diameter or width (vector)
# geo: {'rec', 'cyl_h', 'cyl_v', 'spheroid_h', 'spheroid_v', }   
#     _h for horizontal (along d) and _v for vertical (along l) polarization
# f: resonance frequency in Hz (scalar). Default f = c0/(2*pi).
# eta: generalized (all spectrum) absorption efficiency (scalar). Default eta=1/2.
# fig: if fig>0 plots the D/Q results in figure fig.
#
### Output parameters
# DQ: bound on D/Q (directivity/Q factor) 
# Q: Q factor (or antenna Q, radiation Q)
# xi: semiaxis ratio, l/d
# a: circumscribing sphere, a=sqrt(l^1+d^2)/2;
# ka: electrical size, ka = 2*pi*f*a/co;  
# gamma: polarizability of the structure
#
###
# Copyright (C) Mats Gustafsson 2009
# http://www.eit.lth.se/staff/mats.gustafsson


from pylab import *

HUGE = Inf

#function [DQ,Q,xi,a,ka,gamma] = AntennaQ(l, d, geo, f, eta, fig)
def AntennaQ(l, d, f, geo='rec', eta=0.5):

    c0 = 299792458.  # speed of light

    # test of input format
    try:
        Nl = len(l)
    except:
        Nl = 1
    try:
        Nd = len(d)
    except:
        Nd = 1
    try:
        Nf = len(f)
    except:
        Nf = 1
    Nmax = max([Nl, Nd, Nf])
    if Nl*Nd*Nf != Nmax:
        print 'Wrong input format, see help(AntennaQ)'
        DQ=[]; Q=[]; xi=[]; gamma=[]; a=[]; ka=[]
        return(DQ, Q, xi, a, ka, gamma)
        
    k = f/c0*2*pi  # wavenumber
    if Nd==1:
        if d>0:
            xi = l/d
        else:
            xi = l*HUGE
    else:
        ind1 = find(d!=0)
        ind2 = find(d==0)
        xi = zeros(Nd)
        xi[ind1] = l/d[ind1]
        xi[ind2] = l*HUGE
    try:
        Nxi = len(xi)
    except:
        Nxi = 1
        xi = array([xi])
    if 'sph' in geo:
        a = zeros(Nxi)
        l1 = l*ones(Nxi)
        d1 = d*ones(Nxi)
        for n in arange(0, Nxi):
            a[n] = max([l1[n], d1[n]])
    else:
        a = sqrt(l**2+d**2)/2  # circumscribing sphere
    ka = k*a

    if geo=='rec' or geo=='rectangle' or geo=='rec_h' or geo=='rec_v':
        if geo=='rec_h':
#            xi = 1/xi
            ind1 = find(xi!=0)
            ind2 = find(xi==0)
            xi[ind1] = 1/xi[ind1]
            xi[ind2] = HUGE
        # for xi \leq 1
        # gamma/a^3 = p1(xi)/q1(xi)*xi^2
        q1 = array([0.21950535579637, -0.02821976577187, 0.15933970190107])
        p1 = array([-0.14810750196299, 0.17814702435941, 1])
        ind1 = find(xi<=1)  # case 1
        xi1 = xi[ind1]
        gamma1 = polyval(p1,xi1)/polyval(q1,xi1)*(xi1**2)

        # for xi > 1
        # gamma/a^3 = p2(log10(xi))/q2(log10(xi))
        q2 = array([0.61958847586327, 0.74692798948382, 0.12475302882255, 0.33985806213172])
        p2 = array([-0.01722371349426, 0.16330322959941, 0.59174040686162, 1.99990589974589, 1])
        ind2 = find(xi>1)  # case 2
        xi2 = xi[ind2]
        logxi2 = log10(xi2)
        gamma2 = polyval(p2,logxi2)/polyval(q2,logxi2)

        gamma = zeros(Nxi)
        gamma[ind1] = gamma1
        gamma[ind2] = gamma2
    elif geo=='cyl_h' or geo=='cylinder_h':
        # for xi \leq 1
        # gamma/a^3 = p1(xi)/q1(xi)
        p1 = array([-13.818376196149231,  27.339214002732259, 16/3.])
        q1 = array([0.808897740782751,  0.887124366410425, 1])
        ind1 = find(xi<=1)  # case 1
        xi1 = xi[ind1]
        gamma1 = polyval(p1,xi1)/polyval(q1,xi1)

        # for xi > 1
        # gamma/a^3 = p2(xi)/q2(xi)
        p2 = array([0.000009254785803, -0.009264759779482, 9.511735699859063])
        q2 = array([0.000000781802321, -0.000720656360239, 0.752166159330269,  -0.389464464445270, 1])
        
        ind2 = find(xi>1)  # case 2
        xi2 = xi[ind2]
        gamma2 = polyval(p2,xi2)/polyval(q2,xi2)

        gamma = zeros(Nxi)
        gamma[ind1] = gamma1
        gamma[ind2] = gamma2
    elif geo=='cyl_v' or geo=='cylinder_v':
        # for xi \leq 1
        # gamma/a^3 = p1(xi)/q1(xi)*xi^1
        q1 = array([0.148320825173147, 0.069570439128987, 0.157241205707774])
        p1 = array([-1.588057841947730, 3.789007954315846, 1])
  
        ind1 = find(xi<=1)  # case 1
        xi1 = xi[ind1]
        gamma1 = polyval(p1,xi1)/polyval(q1,xi1)*xi1

        # for xi > 1
        # gamma/a^3 = p2(log10(xi))/q2(log10(xi))
        q2 = array([0.309007911270732, -0.057434715749606, 0.116173183039883])
        p2 = array([-0.014698509654725, 0.151769565450985, -0.010982155229939, 1])
        ind2 = find(xi>1)  # case 2
        xi2 = xi[ind2]
        gamma2 = polyval(p2,log10(xi2))/polyval(q2,log10(xi2))

        gamma = zeros(Nxi)
        gamma[ind1] = gamma1
        gamma[ind2] = gamma2
    elif geo=='sph_h' or geo=='spheroid_h':
        ind1 = find(xi<1)  # case 1
        xi1 = xi[ind1]
        Va = 4*pi*xi1/3.
        e = sqrt(1-xi1**2)
        L1a = (1-e**2)/(2*e**2)*(-1+arcsin(e)/(e*sqrt(1-e**2)))
        gamma1 = Va/L1a
        
        
        ind2 = find(xi>1)  # case 2
        xi2 = 1/xi[ind2]
        Vb = 4*pi*xi2**2/3.
        e = sqrt(1-xi2**2)
        L1b = 1/(4*e**3)*(2*e-(1-e**2)*log((1+e)/(1-e)))
        gamma2 = Vb/L1b

        ind3 = find(xi==1)  # case 3
        xi3 = xi[ind3]
        gamma3 = 4*pi*xi3
        
        ind4 = find(xi<1e-6)  # case 4
        xi4 = xi[ind4]
        gamma4 = 16/3. + 0*xi4        
        
        gamma = zeros(Nxi)
        gamma[ind1] = gamma1
        gamma[ind2] = gamma2
        gamma[ind3] = gamma3
        gamma[ind4] = gamma4
    elif geo=='sph_v' or geo=='spheroid_v':
        ind1 = find(xi<1)  # case 1
        xi1 = xi[ind1]
        Va = 4*pi*xi1/3.
        e = sqrt(1-xi1**2)
        L3a = 1/e**2*(1-sqrt(1-e**2)/e*arcsin(e))
        gamma1 = Va/L3a

        ind2 = find(xi>1)  # case 2
        xi2 = 1/xi[ind2]
        Vb = 4*pi*xi2**2/3.
        e = sqrt(1-xi2**2)
        L3b = (1-e**2)/(2*e**3)*(log((1+e)/(1-e))-2*e)
        gamma2 = Vb/L3b

        ind3 = find(xi==1)  # case 1
        xi3 = xi[ind3]
        gamma3 = 4*pi*xi3
        
        gamma = zeros(Nxi)
        gamma[ind1] = gamma1
        gamma[ind2] = gamma2
        gamma[ind3] = gamma3    
    else:
        print 'Wrong input format, see help(AntennaQ)'
        DQ=[]; Q=[]; xi=[]; gamma=[]; a=[]; ka=[] 
        return(DQ, Q, xi, a, ka, gamma)

    DQ = eta*gamma*ka**3/(2*pi)
    Q = 3/DQ/2   # Q as determined from a single radiating mode with D=3/2 (assuming small antennas)
    gamma = gamma*a**3
    if Nxi==1:
        gamma = gamma*ones(Nf)

    return(DQ, Q, xi, a, ka, gamma)


