/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "tmalloc.h"
#include <pthread.h> /* mutexes */
#include <stdlib.h> /* malloc, free */
#include <string.h> /* memcpy, memset */
#include "platform_types.h"
#include "assert_utils.h"
#include "memxor.h"
#include "memrnd.h"

static int initialized = 1;
static pthread_mutex_t mallocMutex = PTHREAD_MUTEX_INITIALIZER; /* for malloc and free */
static pthread_mutex_t memoryWriteMutex = PTHREAD_MUTEX_INITIALIZER;

static int tmallocIsInitialized(void) { return initialized; }

void tmallocInitialize(void) {
  if (tmallocIsInitialized()) return;
  initialized = 1;
  pthread_mutex_init(&mallocMutex, NULL);
  pthread_mutex_init(&memoryWriteMutex, NULL);
}

void tmallocShutdown(void) {
  if (!tmallocIsInitialized())
    return;
  pthread_mutex_destroy(&mallocMutex);
  pthread_mutex_destroy(&memoryWriteMutex);
  initialized = 0;
}

void *tmalloc(size_t size) {
  void *p;
  pthread_mutex_lock(&mallocMutex);
  p = malloc(size);
  pthread_mutex_unlock(&mallocMutex);
  ASSERT(p, "Allocation failed!");
  return p;
}

void tfree(void *block) {
  pthread_mutex_lock(&mallocMutex);
  ASSERT(block, "Free on NULL pointer!");
  free(block);
  pthread_mutex_unlock(&mallocMutex);
}

void *tmemcpy(void *dst, const void *src, size_t n) {
  void *p;
  pthread_mutex_lock(&memoryWriteMutex);
  p = memcpy(dst, src, n);
  pthread_mutex_unlock(&memoryWriteMutex);
  return p;
}

void *tmemset(void *s, int c, size_t n) {
  void *p;
  pthread_mutex_lock(&memoryWriteMutex);
  p = memset(s, c, n);
  pthread_mutex_unlock(&memoryWriteMutex);
  return p;
}

void *tmemxor(void *dst, void *src, size_t n) {
  void *p;
  pthread_mutex_lock(&memoryWriteMutex);
  p = memxor(dst, src, n);
  pthread_mutex_unlock(&memoryWriteMutex);
  return p;
}

void *tmemrnd(void *dst, size_t n) {
  void *p;
  pthread_mutex_lock(&memoryWriteMutex);
  p = memrnd(dst, n);
  pthread_mutex_unlock(&memoryWriteMutex);
  return p;
}

