#ifndef MEMORY_TRACKER_H
#define MEMORY_TRACKER_H
#include <stdlib.h> /* size_t */

/* regular (not thread-safe) memory operations */
void *memoryTrackerMalloc(size_t size, const char *file, int line);
void memoryTrackerFree(void *block, const char *file, int line);
void *memoryTrackerMemcpy(void *dst, const void *src, size_t n, const char *file, int line);
void *memoryTrackerMemset(void *s, int c, size_t n, const char *file, int line);
void *memoryTrackerMemxor(void *dst, void *src, size_t n, const char *file, int line);
void *memoryTrackerMemrnd(void *dst, size_t n, const char *file, int line);

/* thread-safe variants */
void *memoryTrackerTMalloc(size_t size, const char *file, int line);
void memoryTrackerTFree(void *block, const char *file, int line);
void *memoryTrackerTMemcpy(void *dst, const void *src, size_t n, const char *file, int line);
void *memoryTrackerTMemset(void *s, int c, size_t n, const char *file, int line);
void *memoryTrackerTMemxor(void *dst, void *src, size_t n, const char *file, int line);
void *memoryTrackerTMemrnd(void *dst, size_t n, const char *file, int line);

/* call before your application closes to deallocate internal structures. */
void shutDownMemoryTracker(void);

/* callback function for reporting garbaging warnings */
#define BUFFER_GARBAGING_AT_PRE_FIELD  0 /* buffer written before beginning */
#define BUFFER_GARBAGING_AT_POST_FIELD 1 /* buffer written past end */
typedef void (*memoryGarbagingWarningCallback)(
  int bytes,
  const char *file,
  int line,
  int field,
  const char *fromFile,
  int fromLine);
void registerGarbagingWarningCallback(memoryGarbagingWarningCallback f);

/* callback function for reporting general memory status information */
typedef void (*memoryInfoOverviewCallback)(
  int totalNumAllocations,
  int maxNumSimultaneousAllocations,
  int numPendingDeallocations,
  int slotSpaceExhausted);
/* callback function for reporting each remaining (unfreed) memory block */
typedef void (*memoryInfoDetailsCallback)(
  int bytesAllocated,
  const char *fileName,
  int line);
/* callback function for reporting end of memory status report */
typedef void (*memoryInfoEnd)(void);

/* memory status report functions */
void reportMemoryStatus(memoryInfoOverviewCallback overview, memoryInfoDetailsCallback details, memoryInfoEnd end);
void reportMemoryStatusDefault(void);

#endif

