/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "press_any_key.h"
#include "black_box_first_test.h"
#include "black_box_speed_test.h"
#include "black_box_simple_maxterm.h"
#include "black_box_greedy_maxterm.h"
#include "black_box_optimal_maxterm.h"
#include "black_box_bit_set_utils.h" /* logBitSet */
#include "memory_utils.h"

void printUsageInfo(void) {
  int i;
  printf("Usage: greedy <cipher> <keyFill> <ivFill> <startingSet> <allowKeyBits> <allowIvBits> <strategy> <numParallellBits>\n");
  printf("<cipher>\n");
  for (i=0; i<kNumBlackBoxCiphers; i++) printf("  %2d %s\n", i, blackBoxCipherName((bbCipher)i));
  printf("<keyFill> 0 for zeros, 1 for ones, 2 for random\n");
  printf("<ivFill>  0 for zeros, 1 for ones, 2 for random\n");
  printf("<startingSet>  \"empty\" for empty set, \"optimal\" for small optimal set\n");
  printf("<allowKeyBits> 0 for NO, 1 for YES\n");
  printf("<allowIvBits>  0 for NO, 1 for YES\n");
  printf("<strategy>\n  1 for Add best bit\n  2 for Add best 2-bit set\n  3 for Add best 3-bit set\n");
  printf("<numParallellBits> 0 for single core job, 1 for double, 2 for quadruple,...\n");
}

void smallTestSuite(void) {
  /* simple encryption test for all algorithms */
  blackBoxFirstTestAll();

  /* benchmark for all algorithms */
  blackBoxSpeedTestAll();

  /* MDM test over the n first iv bits (produces a nonrandomness detector) */
  blackBoxSimpleMaxtermAll(12);

  /* optimal bit set for MDM test (produces optimal nonrandomness detector,
   *                               small bit sets only) */
  {
    const int numBits = 2;
    int keyBit[80], ivBit[80];
    int numKeyBits, numIvBits;
    blackBoxOptimalMaxtermDistinguisher(kDES, numBits, FILL_ZEROS, &numKeyBits, keyBit, FILL_ZEROS, &numIvBits, ivBit, 1 /* output */, 2 /* num parallell bits */);
    printf("\n%d key bits ", numKeyBits); logBitSet(NULL, 0, keyBit, numKeyBits);
    printf("\n%d  iv bits ", numIvBits); logBitSet(NULL, 0, ivBit, numIvBits);
    printf("\n\n\n");
  }

  /* greedy maxterm distinguisher for HIGHT */
  blackBoxGreedyMaxtermDistinguisher(kHIGHT, NULL /* log file */, NULL /* tex file */, 12 /* maximum bit set size */, FILL_ZEROS /* key fill */, FILL_ZEROS /* iv fill */, 1 /* size of initial optimal set */, 1 /* okToUseKeyBits */, 0 /* okToUseIvBits */, 1 /* strategy ADD-1 */, 2 /* num parallell bits */);

  /* greedy maxterm distinguisher for all algorithms */
  blackBoxGreedyMaxtermDistinguisherAll(10 /* maximum bit set size */, FILL_ZEROS /* key fill */, FILL_ZEROS /* iv fill */, 1 /* size of initial optimal set */, 1 /* okToUseKeyBits */, 1 /* okToUseIvBits */, 1 /* strategy ADD-1 */, 2 /* num parallell bits */);

  /* greedy maxterm distinguisher for Trivium with initial optimal set computed on the fly */
  blackBoxGreedyMaxtermDistinguisher(kTrivium, NULL /* log file */, NULL /* tex file */, 15, FILL_ZEROS /* key fill */, FILL_ZEROS /* iv fill */, 2 /* size of initial optimal set */, 1 /* okToUseKeyBits */, 1 /* okToUseIvBits */, 1 /* strategy ADD-1 */, 2 /* num parallell bits */);

  /* greedy maxterm distinguisher for Grain-128 with initial optimal set computed on the fly */
  blackBoxGreedyMaxtermDistinguisher(kGrain128, NULL /* log file */, NULL /* tex file */, 15, FILL_ZEROS /* key fill */, FILL_ZEROS /* iv fill */, 2 /* size of initial optimal set */, 1 /* okToUseKeyBits */, 1 /* okToUseIvBits */, 1 /* strategy ADD-1 */, 2 /* num parallell bits */);

  /* optimal bit flip set */
  {
    int i;
    for (i=1; i<3; i++)
      blackBoxOptimalBitFlipSet(kTrivium, i, 1 /* okToUseKeyBits */, 1 /* okToUseIvBits */);
  }
}

void greedyMDMtest(int argc, char **argv) {
  bbCipher cipher;
  int keyFill, ivFill, startingSet, allowKeyBits, allowIvBits, strategy, parallellBits;

  if (argc != 9) {
    printf("%d parameters passed, 8 expected\n\n", argc-1);
    printUsageInfo();
    return;
  }

  cipher = (bbCipher)atoi(argv[1]);
  if (cipher < 0 || cipher >= kNumBlackBoxCiphers) {
    printf("Cipher value must be between 0 and %d (inclusive).\n\n", kNumBlackBoxCiphers-1);
    printUsageInfo(); return;
  }

  keyFill = atoi(argv[2]);
  if (keyFill != 0 && keyFill != 1 && keyFill != 2) {
    printf("Key fill must be 0, 1 or 2.\n\n");
    printUsageInfo(); return;
  }
  switch (keyFill) {
  case 1: keyFill = FILL_ONES; break;
  case 2: keyFill = FILL_RANDOM; break;
  default: keyFill = FILL_ZEROS;
  }

  ivFill = atoi(argv[3]);
  if (ivFill != 0 && ivFill != 1 && ivFill != 2) {
    printf("IV fill must be 0, 1 or 2.\n\n");
    printUsageInfo(); return;
  }
  switch (ivFill) {
  case 1: ivFill = FILL_ONES; break;
  case 2: ivFill = FILL_RANDOM; break;
  default: ivFill = FILL_ZEROS;
  }

  startingSet = 1;
  if (!strncmp(argv[4], "opt", 3)) startingSet = -1;

  allowKeyBits = atoi(argv[5]);
  if (allowKeyBits != 0 && allowKeyBits != 1) {
    printf("The allowKeyBits parameter must be 0 or 1.\n\n");
    printUsageInfo(); return;
  }

  allowIvBits = atoi(argv[6]);
  if (allowIvBits != 0 && allowIvBits != 1) {
    printf("The allowIvBits parameter must be 0 or 1.\n\n");
    printUsageInfo(); return;
  }

  strategy = atoi(argv[7]);
  if (strategy != 0 && strategy != 1 && strategy != 2) {
    printf("Strategy must be 0, 1 or 2.\n\n");
    printUsageInfo(); return;
  }
  switch (strategy) {
  case 1: strategy = GREEDY_STRATEGY_ADD_1; break;
  case 2: strategy = GREEDY_STRATEGY_ADD_2; break;
  case 3: strategy = GREEDY_STRATEGY_ADD_3; break;
  }

  parallellBits = atoi(argv[8]);
  if (parallellBits > 8) {
    printf("Hey, go easy on your CPU, choose at most 8 parallell bits.\n\n");
    printUsageInfo(); return;
  }

  blackBoxGreedyMaxtermDistinguisher(
    cipher,
    "greedy.txt", /* console output is echoed to file */
    "greedy.tex", /* ...and a nice graph in tex is output as well */
    60 /* max bit set size */,
    keyFill,
    ivFill,
    startingSet, /* starting set [-1=stored optimal, 1=empty set] */
    allowKeyBits,
    allowIvBits,
    strategy,
    parallellBits);
}

int main(int argc, char **argv) {
//  smallTestSuite();
  greedyMDMtest(argc, argv);

  printf("\n\n");
#ifdef MEMORY_TRACKER_ENABLED
  reportMemoryStatusDefault();
  shutDownMemoryTracker();
#endif
  printf("\nDone!\n");
  pressAnyKey();
  return 0;
}

