/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_xtea.h"

#ifdef ENDIANNESS_UNDETERMINED
#undef ENDIANNESS_UNDETERMINED
#endif
#ifdef BIG_ENDIAN
#undef BIG_ENDIAN
#endif
#ifndef LITTLE_ENDIAN
#define LITTLE_ENDIAN
#endif

#define TEA_NUM_ROUNDS 32
#define XTEA_NUM_ROUNDS 32

#define TEA_BLOCK_SIZE 8
#define XTEA_BLOCK_SIZE TEA_BLOCK_SIZE

#define REVERSE_BYTE_ORDER_32(x) \
  { \
    UINT32 v_rev = (((x) << 16) | ((x) >> 16)); \
    x = ( ((v_rev & 0xff00ff00UL) >> 8) | ((v_rev & 0x00ff00ffUL) << 8) ); \
  }

#define RET(H,err) \
  { \
    H->Status = err; \
    return; \
  }

#define CHECK_TEA_PARAMS(H,cm,bm) \
    if (!(H->output)) /* No provided output buffer */ \
      RET(H,-1) /* Invalid output buffer */ \
    if ((cm != kEncrypt) && (cm != kDecrypt)) \
      RET(H,-1) /* invalid cryption mode */ \
    if ((bm != kECB) && (bm != kCBC)) \
      RET(H,-1) /* invalid block mode */ \
    if (!(H->key)) \
      RET(H,-1) /* invalid key buffer */ \
    if ((H->keyLengthInBits != 128)) \
      RET(H,-1) /* invalid key length */ \
    if ((bm == kCBC) && !(H->IV)) \
      RET(H,-1) /* invalid IV buffer */ \
    if ((bm == kCBC) && (H->IV) && (H->IVLengthInBytes != TEA_BLOCK_SIZE)) \
      RET(H,-1) /* invalid IV length */

#define CHECK_XTEA_PARAMS(H,cm,bm,r) \
    if (!(H->output)) /* No provided output buffer */ \
      RET(H,-1) /* Invalid output buffer */ \
    if ((cm != kEncrypt) && (cm != kDecrypt)) \
      RET(H,-1) /* invalid cryption mode */ \
    if ((bm != kECB) && (bm != kCBC)) \
      RET(H,-1) /* invalid block mode */ \
    if ((r < 1) || (1024 < r)) /* to catch incorrect usage */ \
      RET(H,-1) /* invalid round parameter */ \
    if (!(H->key)) \
      RET(H,-1) /* invalid key buffer */ \
    if ((H->keyLengthInBits != 128)) \
      RET(H,-1) /* invalid key length */ \
    if ((bm == kCBC) && !(H->IV)) \
      RET(H,-1) /* invalid IV buffer */ \
    if ((bm == kCBC) && (H->IV) && (H->IVLengthInBytes != XTEA_BLOCK_SIZE)) \
      RET(H,-1) /* invalid IV length */

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define REVERSE_L_R(L,R) \
   if (LITTLE_ENDIAN_PLATFORM) \
  { \
    REVERSE_BYTE_ORDER_32(L) \
    REVERSE_BYTE_ORDER_32(R) \
  }
#elif defined(LITTLE_ENDIAN)
#define REVERSE_L_R(L,R) \
  REVERSE_BYTE_ORDER_32(L) \
  REVERSE_BYTE_ORDER_32(R)
#else
#define REVERSE_L_R(L,R) /* Do nothing */
#endif

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define REVERSE_ABCD(A,B,C,D) \
   if (LITTLE_ENDIAN_PLATFORM) \
  { \
    REVERSE_BYTE_ORDER_32(A) \
    REVERSE_BYTE_ORDER_32(B) \
    REVERSE_BYTE_ORDER_32(C) \
    REVERSE_BYTE_ORDER_32(D) \
  }
#elif defined(LITTLE_ENDIAN)
#define REVERSE_ABCD(A,B,C,D) \
  REVERSE_BYTE_ORDER_32(A) \
  REVERSE_BYTE_ORDER_32(B) \
  REVERSE_BYTE_ORDER_32(C) \
  REVERSE_BYTE_ORDER_32(D)
#else
#define REVERSE_ABCD(A,B,C,D) /* Do nothing */
#endif

#define GET_NEXT_TEA_INPUT_BLOCK(L,R,x) \
    L = ((UINT32*)x)[0]; /* Get next input block */ \
    R = ((UINT32*)x)[1]; \
    REVERSE_L_R(L,R) /* Reverse byte order if little endian platform */

#define GET_NEXT_XTEA_INPUT_BLOCK(L,R,x) GET_NEXT_TEA_INPUT_BLOCK(L,R,x)

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define UPDATE_TEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  if (LITTLE_ENDIAN_PLATFORM) \
  { \
  	UINT32 temp = L; \
  	REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[0] = temp; /* crypted block to output buffer */ \
    temp = R; \
  	REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[1] = temp; \
  } \
  else \
  { \
    output32Ptr[0] = L; /* crypted block to output buffer */ \
    output32Ptr[1] = R; \
  }
#elif defined(LITTLE_ENDIAN)
#define UPDATE_TEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  { \
  	UINT32 temp = L; \
  	REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[0] = temp; /* crypted block to output buffer */ \
    temp = R; \
  	REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[1] = temp; \
  }
#else
#define UPDATE_TEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  output32Ptr[0] = L; /* crypted block to output buffer */ \
  output32Ptr[1] = R;
#endif

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define UPDATE_XTEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  if (LITTLE_ENDIAN_PLATFORM) \
  { \
    UINT32 temp = L; \
    REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[0] = temp; /* crypted block to output buffer */ \
    temp = R; \
    REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[1] = temp; \
  } \
  else \
  { \
    output32Ptr[0] = L; /* crypted block to output buffer */ \
    output32Ptr[1] = R; \
  }
#elif defined(LITTLE_ENDIAN)
#define UPDATE_XTEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  { \
    UINT32 temp = L; \
    REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[0] = temp; /* crypted block to output buffer */ \
    temp = R; \
    REVERSE_BYTE_ORDER_32(temp) \
    output32Ptr[1] = temp; \
  }
#else
#define UPDATE_XTEA_OUTPUT_BUFFER(output32Ptr,L,R) \
  output32Ptr[0] = L; /* crypted block to output buffer */ \
  output32Ptr[1] = R;
#endif

#define GET_TEA_KEY(B, a, b, c, d) \
    a = ((UINT32*)(B->key))[0]; /* get key */ \
    b = ((UINT32*)(B->key))[1]; \
    c = ((UINT32*)(B->key))[2]; \
    d = ((UINT32*)(B->key))[3]; \
    REVERSE_ABCD(a,b,c,d)

#define GET_XTEA_KEY(B, k) \
    k[0] = ((UINT32*)(B->key))[0]; /* get key */ \
    k[1] = ((UINT32*)(B->key))[1]; \
    k[2] = ((UINT32*)(B->key))[2]; \
    k[3] = ((UINT32*)(B->key))[3]; \
    REVERSE_ABCD(k[0],k[1],k[2],k[3])

#define COPY_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d = *s; }
#define XOR_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d ^= *s; }

typedef enum {
  kEncrypt, /* Encryption */
  kDecrypt  /* Decryption */
} CryptionMode;

typedef enum {
  kECB, /* Electronic-codebook */
  kCBC, /* Cipher-block chaining */
  kCFB, /* Cipher feedback */
  kOFB, /* Output feedback */
  kCTR  /* Counter mode */
} BlockMode;

typedef struct {
  CryptionMode cm; /* kEncrypt or kDecrypt */ \
  BlockMode bm; /* kECB (Electronic Code Book) or kCBC (Cipher Block Chaining) */ \
  BYTE *key; /* Cryption key */ \
  UINT32 keyLengthInBits; /* Length of cryption key in bits */ \
  BYTE *IV; /* Initialization vector must be provided if block mode kCBC is used (set to NULL when using kECB) */ \
  UINT32 IVLengthInBytes; /* IV length in bytes (set to 0 when using block mode kECB) (IV length must be same as the block length of the cipher) */ \
  BYTE *output; /* Output buffer (this is where the cryption result, the cipher text, is placed) */ \
  UINT32 outputBufferLengthInBytes; /* Length of output buffer (in bytes) */ \
  UINT32 outputLengthInBytes;
  BYTE Status;
} TEA_info;

typedef struct {
  CryptionMode cm; /* kEncrypt or kDecrypt */ \
  BlockMode bm; /* kECB (Electronic Code Book) or kCBC (Cipher Block Chaining) */ \
  BYTE *key; /* Cryption key */ \
  UINT32 keyLengthInBits; /* Length of cryption key in bits */ \
  BYTE *IV; /* Initialization vector must be provided if block mode kCBC is used (set to NULL when using kECB) */ \
  UINT32 IVLengthInBytes; /* IV length in bytes (set to 0 when using block mode kECB) (IV length must be same as the block length of the cipher) */ \
  BYTE *output; /* Output buffer (this is where the cryption result, the cipher text, is placed) */ \
  UINT32 outputBufferLengthInBytes; /* Length of output buffer (in bytes) */ \
  UINT32 outputLengthInBytes;
  BYTE Status;
  UINT32 r;
} XTEA_info;

void TEA_init(TEA_info *B)
{
  CryptionMode cm = B->cm;
  BlockMode bm = B->bm;

  CHECK_TEA_PARAMS(B,cm,bm) /* Check parameters */
  B->Status = 0;
}

void TEA_cryptBlock(TEA_info *B, BYTE *x)
{
  UINT32 L, R;
  UINT32 *output32Ptr = (UINT32*)B->output; /* Get output buffer */
  UINT32 a, b, c, d;

  GET_TEA_KEY(B,a,b,c,d)
  GET_NEXT_TEA_INPUT_BLOCK(L,R,x)
  if (B->cm == kEncrypt)
  {
    UINT32 s = 0, n = TEA_NUM_ROUNDS;

    while (n-- > 0)
    {
      s += 0x9e3779b9;
      L += (R << 4) + a ^ R + s ^ (R >> 5) + b;
      R += (L << 4) + c ^ L + s ^ (L >> 5) + d;
    }
  }
  else /* decryption */
  {
    UINT32 s = 0x9e3779b9 << 5, n = TEA_NUM_ROUNDS;

    while (n-- > 0)
    {
      R -= (L << 4) + c ^ L + s ^ (L >> 5) + d;
      L -= (R << 4) + a ^ R + s ^ (R >> 5) + b;
      s -= 0x9e3779b9;
    }
  }
  UPDATE_TEA_OUTPUT_BUFFER(output32Ptr,L,R)
  B->Status = 0;
}

#define XOR_INTO_TEA_OUTPUT_BUFFER(out,L,R) \
  { \
    UINT32 temp = L; \
    REVERSE_BYTE_ORDER_32(temp) \
    *out++ ^= temp; /* crypted block to output buffer */ \
    temp = R; \
    REVERSE_BYTE_ORDER_32(temp) \
    *out++ ^= temp; \
  }

#define COPY_INTO_TEA_OUTPUT_BUFFER(out, L, R) \
  { \
    UINT32 *p = (UINT32*)(out); \
    UINT32 temp = L; \
    REVERSE_BYTE_ORDER_32(temp) \
    p[0] = temp; /* crypted block to output buffer */ \
    temp = R; \
    REVERSE_BYTE_ORDER_32(temp) \
    p[1] = temp; \
  }

#define XOR_INTO_XTEA_OUTPUT_BUFFER(out,L,R) XOR_INTO_TEA_OUTPUT_BUFFER(out,L,R)
#define COPY_INTO_XTEA_OUTPUT_BUFFER(out, L, R) COPY_INTO_TEA_OUTPUT_BUFFER(out, L, R)

void TEA_cryptBlock_xor_withInitOutput(TEA_info *B, const BYTE *x, BYTE *ct, int numBlocks)
{
  UINT32 L, R;
  UINT32 *out = (UINT32*)B->output; /* Get output buffer */
  UINT32 a, b, c, d;

  B->Status = 0;
  GET_TEA_KEY(B,a,b,c,d)
  GET_NEXT_TEA_INPUT_BLOCK(L,R,x)
  if (B->cm == kEncrypt)
  {
    UINT32 s = 0, n = TEA_NUM_ROUNDS;

    while (n-- > 0)
    {
      s += 0x9e3779b9;
      L += (R << 4) + a ^ R + s ^ (R >> 5) + b;
      R += (L << 4) + c ^ L + s ^ (L >> 5) + d;
      XOR_INTO_TEA_OUTPUT_BUFFER(out,L,R)
      if (--numBlocks == 0 && n > 0) return;
    }
  }
  else /* decryption */
  {
    UINT32 s = 0x9e3779b9 << 5, n = TEA_NUM_ROUNDS;

    while (n-- > 0)
    {
      R -= (L << 4) + c ^ L + s ^ (L >> 5) + d;
      L -= (R << 4) + a ^ R + s ^ (R >> 5) + b;
      s -= 0x9e3779b9;
      XOR_INTO_TEA_OUTPUT_BUFFER(out,L,R)
      if (--numBlocks == 0 && n > 0) return;
    }
  }
  COPY_INTO_TEA_OUTPUT_BUFFER(ct, L, R)
}

void TEA_final(TEA_info *B) { B->Status = 0; }

void XTEA_init(XTEA_info *B)
{
  CryptionMode cm = B->cm;
  BlockMode bm = B->bm;
  UINT32 N = B->r; /* get number of rounds */

  CHECK_XTEA_PARAMS(B,cm,bm,N) /* Check parameters */
  B->Status = 0;
}

void XTEA_cryptBlock(XTEA_info *B, BYTE *x)
{
  UINT32 L, R, k[4], N = B->r; /* get number of rounds */
  UINT32 *output32Ptr = (UINT32*)B->output; /* Get output buffer */

  GET_XTEA_KEY(B, k)
  GET_NEXT_XTEA_INPUT_BLOCK(L,R,x)
  if (B->cm == kEncrypt)
  {
    UINT32 limit = 0x9e3779b9 * N, sum = 0;

    while (sum != limit)
    {
      L += ((R << 4) ^ (R >> 5)) + R ^ sum + k[sum & 3];
      sum += 0x9e3779b9;
      R += ((L << 4) ^ (L >> 5)) + L ^ sum + k[(sum >> 11) & 3];
    }
  }
  else /* decryption */
  {
    UINT32 sum = 0x9e3779b9 * N;

    while (sum)
    {
      R -= ((L << 4) ^ (L >> 5)) + L ^ sum + k[(sum >> 11) & 3];
      sum -= 0x9e3779b9;
      L -= ((R << 4) ^ (R >> 5)) + R ^ sum + k[sum & 3];
    }
  }
  UPDATE_XTEA_OUTPUT_BUFFER(output32Ptr,L,R)
  B->Status = 0;
}

void XTEA_cryptBlock_xor_withInitOutput(XTEA_info *B, const BYTE *x, BYTE *ct, int numBlocks)
{
  UINT32 L, R, k[4], N = B->r; /* get number of rounds */
  UINT32 *out = (UINT32*)B->output; /* Get output buffer */

  B->Status = 0;
  GET_XTEA_KEY(B, k)
  GET_NEXT_XTEA_INPUT_BLOCK(L,R,x)
  if (B->cm == kEncrypt)
  {
    UINT32 limit = 0x9e3779b9 * N, sum = 0;

    while (sum != limit)
    {
      L += ((R << 4) ^ (R >> 5)) + R ^ sum + k[sum & 3];
      sum += 0x9e3779b9;
      R += ((L << 4) ^ (L >> 5)) + L ^ sum + k[(sum >> 11) & 3];
      XOR_INTO_XTEA_OUTPUT_BUFFER(out,L,R)
      if (--numBlocks == 0 && sum != limit) return;
    }
  }
  else /* decryption */
  {
    UINT32 sum = 0x9e3779b9 * N;

    while (sum)
    {
      R -= ((L << 4) ^ (L >> 5)) + L ^ sum + k[(sum >> 11) & 3];
      sum -= 0x9e3779b9;
      L -= ((R << 4) ^ (R >> 5)) + R ^ sum + k[sum & 3];
      XOR_INTO_XTEA_OUTPUT_BUFFER(out,L,R)
      if (--numBlocks == 0 && sum) return;
    }
  }
  COPY_INTO_XTEA_OUTPUT_BUFFER(ct, L, R)
}

void XTEA_final(XTEA_info *B) { B->Status = 0; }

/******************************************************************************
 * Black box variants
 ******************************************************************************/
int tea_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  TEA_info ctx;
  BYTE pt[TEA_BLOCK_SIZE];
  BYTE ct[TEA_BLOCK_SIZE];
  int i;
  const int numBlocks = numOutputBytes / TEA_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kCBC;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 128;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = TEA_BLOCK_SIZE;
  ctx.output = ct;
  ctx.outputBufferLengthInBytes = numBlocks * TEA_BLOCK_SIZE;

  TEA_init(&ctx);
  if (ctx.Status) return -1;
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, i==0 ? iv : ct)
    XOR_BUF_8(pt, inBuf + i * TEA_BLOCK_SIZE)
    TEA_cryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + i * TEA_BLOCK_SIZE, ct)
  }
  TEA_final(&ctx);
  return 0;
}

int tea_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  TEA_info ctx;
  BYTE pt[TEA_BLOCK_SIZE];
  BYTE ct[TEA_BLOCK_SIZE];
  int i;
  int numBlocks = numOutputBytes / TEA_BLOCK_SIZE;
  const int numSuppressedBytes = (TEA_NUM_ROUNDS - 1) * TEA_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kCBC;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 128;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = TEA_BLOCK_SIZE;
  ctx.output = outBuf;
  ctx.outputBufferLengthInBytes = numBlocks * TEA_BLOCK_SIZE;

  TEA_init(&ctx);
  if (ctx.Status) return -1;

  /* first R blocks (R-1 suppressed + first) */
  COPY_BUF_8(pt, iv)
  XOR_BUF_8(pt, inBuf)
  TEA_cryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
  if (numBlocks <= TEA_NUM_ROUNDS) {
    TEA_final(&ctx);
    return 0;
  }
  ctx.output = ct;
  numBlocks -= TEA_NUM_ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, ct)
    XOR_BUF_8(pt, inBuf + (i + 1) * TEA_BLOCK_SIZE)
    TEA_cryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + (i + TEA_NUM_ROUNDS) * TEA_BLOCK_SIZE, ct)
  }

  TEA_final(&ctx);
  return ctx.Status;
}

int xtea_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  XTEA_info ctx;
  BYTE pt[XTEA_BLOCK_SIZE];
  BYTE ct[XTEA_BLOCK_SIZE];
  int i;
  const int numBlocks = numOutputBytes / XTEA_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kCBC;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 128;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = XTEA_BLOCK_SIZE;
  ctx.output = ct;
  ctx.outputBufferLengthInBytes = numBlocks * XTEA_BLOCK_SIZE;
  ctx.r = XTEA_NUM_ROUNDS;

  XTEA_init(&ctx);
  if (ctx.Status) return -1;
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, i==0 ? iv : ct)
    XOR_BUF_8(pt, inBuf + i * XTEA_BLOCK_SIZE)
    XTEA_cryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + i * XTEA_BLOCK_SIZE, ct)
  }
  XTEA_final(&ctx);
  return 0;
}

int xtea_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  XTEA_info ctx;
  BYTE pt[XTEA_BLOCK_SIZE];
  BYTE ct[XTEA_BLOCK_SIZE];
  int i;
  int numBlocks = numOutputBytes / XTEA_BLOCK_SIZE;
  const int numSuppressedBytes = (XTEA_NUM_ROUNDS - 1) * XTEA_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kCBC;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 128;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = XTEA_BLOCK_SIZE;
  ctx.output = outBuf;
  ctx.outputBufferLengthInBytes = numBlocks * XTEA_BLOCK_SIZE;
  ctx.r = XTEA_NUM_ROUNDS;

  XTEA_init(&ctx);
  if (ctx.Status) return -1;

  /* first R blocks (R-1 suppressed + first) */
  COPY_BUF_8(pt, iv)
  XOR_BUF_8(pt, inBuf)
  XTEA_cryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
  if (numBlocks <= XTEA_NUM_ROUNDS) {
    XTEA_final(&ctx);
    return 0;
  }
  ctx.output = ct;
  numBlocks -= XTEA_NUM_ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, ct)
    XOR_BUF_8(pt, inBuf + (i + 1) * XTEA_BLOCK_SIZE)
    XTEA_cryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + (i + XTEA_NUM_ROUNDS) * XTEA_BLOCK_SIZE, ct)
  }

  XTEA_final(&ctx);
  return ctx.Status;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
int blackBoxTEAEncryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
      return tea_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return tea_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

int blackBoxXTEAEncryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
      return xtea_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return xtea_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxTEAInfo(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 16;
  if (ivSizeInBytes) *ivSizeInBytes = TEA_BLOCK_SIZE;
  if (suppressedBytes) *suppressedBytes = (TEA_NUM_ROUNDS - 1) * TEA_BLOCK_SIZE;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = TEA_BLOCK_SIZE;
}

void getBlackBoxXTEAInfo(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 16;
  if (ivSizeInBytes) *ivSizeInBytes = XTEA_BLOCK_SIZE;
  if (suppressedBytes) *suppressedBytes = (XTEA_NUM_ROUNDS - 1) * XTEA_BLOCK_SIZE;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = XTEA_BLOCK_SIZE;
}


