/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_speed_test.h"
#include "black_box_utils.h"
#include "rdtsc.h"
#include "bitslice_utils.h"
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#if defined(CLOCKS_PER_SEC) && !defined(CLK_TCK)
#define CLK_TCK CLOCKS_PER_SEC
#endif

void blackBoxSpeedTestStandard(bbCipher cipher) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  unsigned int numOutputBytes1, numOutputBytes2;
  const unsigned int minNumOutputBytes1 = 200, minNumOutputBytes2 = 1200;
  BYTE *key, *iv, *in, *out;
  int res;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  const float numSecondsPerFunction = 0.5;
  const int numClocksPerRdtsc = 10;
  UINT64 tot1 = 0, tot2 = 0;
  int numPositive1 = 0, numPositive2 = 0;
  UINT64 i;
  UINT64 min1 = 10000000, min2 = 10000000;
  UINT64 c1, c2;
  clock_t t1, t2;

  stars(nameLen + 18);
  printf("\n* Speed testing %s *\n", name);
  stars(nameLen + 18);
  printf("\n\n");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);

  key = (BYTE*)malloc(keySize * sizeof(BYTE));
  iv = (BYTE*)malloc(ivSize * sizeof(BYTE));
  memset(key, 0, keySize);
  memset(iv, 0, ivSize);

  numOutputBytes1 = numOutputBytes2 = 0;
  while (numOutputBytes1 < minNumOutputBytes1) numOutputBytes1 += implicitBlockSize;
  while (numOutputBytes2 < minNumOutputBytes2 &&
         numOutputBytes2 < numOutputBytes1 + 1000) numOutputBytes2 += implicitBlockSize;
  out = (BYTE*)malloc(numOutputBytes2 * sizeof(BYTE));
  in = (BYTE*)malloc(numOutputBytes2 * sizeof(BYTE));
  memset(in, 0, numOutputBytes2 * sizeof(BYTE));

  /* make sure cryption works */
  res = blackBoxEncrypt(cipher, key, iv, in, numOutputBytes2, out, numOutputBytes2, 0 /* no init round output */);
  if (res) {
    printf(" Error (returned %d)\n", res);
    free(key);
    free(iv);
    free(in);
    free(out);
    return;
  }

  printf("Speed test for %s\n\n", name);
  printf("%d output bytes including key schedule", numOutputBytes1);

  i = 0;
  t1 = clock();
  do {
    c1 = rdtsc();
    blackBoxEncrypt(cipher, key, iv, in, numOutputBytes1, out, numOutputBytes1, 0 /* no init round output */);
    c2 = rdtsc() - numClocksPerRdtsc;
    if (c2 - c1 > 0) {
      numPositive1++;
      tot1 += c2 - c1;
    }
    if (c2 - c1 < min1) {
      min1 = c2 - c1;
    }
    i++;
    t2 = clock();
  } while (((t2 - t1) / CLK_TCK ) < numSecondsPerFunction);

  printf(" (%Ld trials)\n", i);
  printf("Min cycles: %Ld\n", min1);
  printf("Average cycles / byte = %.2f\n", tot1 / (double)numPositive1 / numOutputBytes1);
  printf("Average cycles / bit = %.2f\n\n", tot1 / (double)numPositive1 / numOutputBytes1 / 8);
  printf("%d output bytes including key schedule", numOutputBytes2);

  i = 0;
  t1 = clock();
  do {
    c1 = rdtsc();
    blackBoxEncrypt(cipher, key, iv, in, numOutputBytes2, out, numOutputBytes2, 0 /* no init round output */);
    c2 = rdtsc() - numClocksPerRdtsc;
    if (c2 - c1 > 0) {
      numPositive2++;
      tot2 += c2 - c1;
    }
    if (c2 - c1 < min2) {
      min2 = c2 - c1;
    }
    t2 = clock();
    i++;
  } while (((t2 - t1) / CLK_TCK ) < numSecondsPerFunction);

  free(key);
  free(iv);
  free(in);
  free(out);

  printf(" (%Ld trials)\n", i);
  printf("Min cycles: %Ld\n", min2);
  printf("Average cycles / byte = %.2f\n", tot2 / (double)numPositive2 / numOutputBytes2);
  printf("Average cycles / bit = %.2f\n\n", tot2 / (double)numPositive2 / numOutputBytes2 / 8);

  printf("Keystream generation only\n");
  printf("Cycles / byte = %.2f\n", (min2 - min1) / (float)(numOutputBytes2 - numOutputBytes1));
  printf("Cycles / bit = %.2f\n\n\n",  (min2 - min1) / (float)(numOutputBytes2 - numOutputBytes1) / 8);
}

void blackBoxSpeedTestBitsliced(bbCipher cipher) {
  int keySize, ivSize, suppressedBits, implicitBlockSize;
  const unsigned int minNumOutputBits1 = 1000, minNumOutputBits2 = 10000;
  unsigned int numOutputBits1, numOutputBits2;
  UINT64 *key, *iv, *in, *out;
  int res;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  const float numSecondsPerFunction = 0.5;
  const int numClocksPerRdtsc = 10;
  UINT64 i;
  UINT64 min1 = 1000000000, min2 = 1000000000;
  UINT64 c1, c2;
  float cyclesPerBit;
  clock_t t1, t2;

  stars(nameLen + 28);
  printf("\n* Speed testing Bitsliced %s *\n", name);
  stars(nameLen + 28);
  printf("\n\n");

  if (!blackBoxCipherProvidesBitslicedImplementation(cipher)) {
    printf("No bitsliced implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfoBitsliced(cipher, &keySize, &ivSize, &suppressedBits, &implicitBlockSize);

  key = (UINT64*)malloc(keySize * sizeof(UINT64));
  iv = (UINT64*)malloc(ivSize * sizeof(UINT64));
  memset(key, 0, keySize * sizeof(UINT64));
  memset(iv, 0, ivSize * sizeof(UINT64));

  numOutputBits1 = numOutputBits2 = 0;
  while (numOutputBits1 < minNumOutputBits1) numOutputBits1 += implicitBlockSize;
  while (numOutputBits2 < minNumOutputBits2 &&
         numOutputBits2 < numOutputBits2 + 10000) numOutputBits2 += implicitBlockSize;
  out = (UINT64*)malloc(numOutputBits2 * sizeof(UINT64));
  in = (UINT64*)malloc(numOutputBits2 * sizeof(UINT64));
  memset(in, 0, numOutputBits2 * sizeof(UINT64));

  /* make sure cryption works */
  res = blackBoxEncryptBitsliced(cipher, key, iv, in, numOutputBits2, out, numOutputBits2, 0 /* no init round output */);
  if (res) {
    printf(" Error (returned %d)\n", res);
    free(key);
    free(iv);
    free(in);
    free(out);
    return;
  }

  printf("Speed test for %s\n\n", name);
  printf("%d output bits including key schedule", numOutputBits1);

  i = 0;
  t1 = clock();
  do {
    c1 = rdtsc();
    blackBoxEncryptBitsliced(cipher, key, iv, in, numOutputBits1, out, numOutputBits1, 0 /* no init round output */);
    c2 = rdtsc() - numClocksPerRdtsc;
    if (c2 - c1 < min1) {
      min1 = c2 - c1;
    }
    i++;
    t2 = clock();
  } while (((t2 - t1) / CLK_TCK ) < numSecondsPerFunction);

  printf(" (%Ld trials)\n", i);
  printf("Min cycles: %Ld\n", min1);
  cyclesPerBit = min1 / (double)numOutputBits1 / 64;
  printf("Cycles / byte (amortized) = %.2f\n", cyclesPerBit * 8);
  printf("Cycles / bit (amortized)  = %.2f\n\n", cyclesPerBit);
  printf("%d output bits including key schedule", numOutputBits2);


  i = 0;
  t1 = clock();
  do {
    c1 = rdtsc();
    blackBoxEncryptBitsliced(cipher, key, iv, in, numOutputBits2, out, numOutputBits2, 0 /* no init round output */);
    c2 = rdtsc() - numClocksPerRdtsc;
    if (c2 - c1 < min2) {
      min2 = c2 - c1;
    }
    i++;
    t2 = clock();
  } while (((t2 - t1) / CLK_TCK ) < numSecondsPerFunction);

  free(key);
  free(iv);
  free(in);
  free(out);

  printf(" (%Ld trials)\n", i);
  printf("Min cycles: %Ld\n", min2);
  cyclesPerBit = min2 / (double)numOutputBits2 / 64;
  printf("Cycles / byte (amortized) = %.2f\n", cyclesPerBit * 8);
  printf("Cycles / bit (amortized)  = %.2f\n\n", cyclesPerBit);

  printf("Keystream generation only\n");
  cyclesPerBit = (min2 - min1) / (double)(numOutputBits2 - numOutputBits1) / 64;
  printf("Cycles / byte (amortized) = %.2f\n", cyclesPerBit * 8);
  printf("Cycles / bit (amortized)  = %.2f\n\n\n", cyclesPerBit);
}

void blackBoxSpeedTest(bbCipher cipher) {
  blackBoxSpeedTestStandard(cipher);
  blackBoxSpeedTestBitsliced(cipher);
}

void blackBoxSpeedTestAll(void) {
  int i;
  starHeader("Black Box - Speed Testing");
  for (i=0; i<kNumBlackBoxCiphers; i++)
    blackBoxSpeedTest((bbCipher)i);
}

