/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_simple_maxterm.h"
#include "black_box_utils.h"
#include "log_utils.h"
#include "bittwiddling.h"
#include <stdio.h>
#include <string.h>
#include <malloc.h>

void simpleMaxtermStandard(bbCipher cipher, int numIvBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  unsigned int numOutputBytes;
  BYTE *key, *iv, *in, *out;
  int res;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  UINT64 i, max = (UINT64)1 << numIvBits;
  time_t start = time(NULL);

  stars(nameLen + 28);
  printf("\n* Simple Maxterm Test for %s *\n", name);
  stars(nameLen + 28);
  printf("\n\n");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);

  if (ivSize * 8 < numIvBits)
    numIvBits = ivSize * 8;

  numOutputBytes = suppressedBytes + implicitBlockSize;
  out = (BYTE*)malloc(numOutputBytes * sizeof(BYTE)); memset(out, 0, numOutputBytes);
  in  = (BYTE*)malloc(numOutputBytes * sizeof(BYTE)); memset( in, 0, numOutputBytes);
  key = (BYTE*)malloc(keySize * sizeof(BYTE)); memset(key, 0, keySize);
  iv  = (BYTE*)malloc( ivSize * sizeof(BYTE)); memset( iv, 0, ivSize);

  printf("Iterating over %d IV bits ( 0 ... %d )", numIvBits, numIvBits-1);
  if (numIvBits < ivSize * 8) printf(", setting remaining IV bits to zero");
  printf("\n\nKey:"); logBuf(NULL, LOGSCREEN, key, keySize, 0, 0); printf("\n\n");

  for (i=0; i<max; i++) {
    /* set iv */
    *((UINT64*)iv) = i; /* little endian */

    /* add xor */
    res = blackBoxEncrypt(cipher, key, iv, in, numOutputBytes - suppressedBytes, out, numOutputBytes, 1);
    if (res)
      printf("Error (returned %d)!\n", res);

    if (isPowerOfTwo(i+1)) {
      int numZeroBits = numInitialZeroBits(out, numOutputBytes);
      int l = lg(i+1);
      printf("[");
      logDateRaw(NULL, LOGSCREEN);
      printf(", ");
      logTimeRaw(NULL, LOGSCREEN, start);
      printf("]  With %2d IV bits  ->  %d / %d = %2.2f%% zero bits:", l, numZeroBits, suppressedBytes * 8, numZeroBits * 100 / (float)suppressedBytes / 8);
      logBuf(NULL, LOGSCREEN, out, numOutputBytes, blackBoxCipherType(cipher) == kBlockCipher ? implicitBlockSize : 0, 0);
      printf("\n");
    }
  }

  /* output result */
  {
    int numZeroBits = numInitialZeroBits(out, numOutputBytes);
    printf("\n %d / %d = %2.2f%% zero bits:", numZeroBits, suppressedBytes * 8, numZeroBits * 100 / (float)suppressedBytes / 8);
    logBuf(NULL, LOGSCREEN, out, numOutputBytes, blackBoxCipherType(cipher) == kBlockCipher ? implicitBlockSize : 0, 0);
    printf("\n\n\n");
  }

  free(in);
  free(out);
  free(iv);
  free(key);
}

void blackBoxSimpleMaxterm(bbCipher cipher, int numIvBits) {
  simpleMaxtermStandard(cipher, numIvBits);
//  simpleMaxtermBitsliced(cipher, numIvBits);
}

void blackBoxSimpleMaxtermAll(int numIvBits) {
  int i;
  starHeader("Black Box - Simple Maxterm Test");
  for (i=0; i<kNumBlackBoxCiphers; i++) {
    if ((bbCipher)i == kEdon80) continue;
    blackBoxSimpleMaxterm((bbCipher)i, numIvBits);
  }
}

