/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_rc6.h"
#include "memory_utils.h"

#ifdef ENDIANNESS_UNDETERMINED
#undef ENDIANNESS_UNDETERMINED
#endif
#ifdef BIG_ENDIAN
#undef BIG_ENDIAN
#endif
#ifndef LITTLE_ENDIAN
#define LITTLE_ENDIAN
#endif

#define RC5_W 32
#define RC5_R 12
#define RC5_B 16
#define RC5_BLOCK_LEN (RC5_W * 2 / 8)
#define RC5_ROUNDS RC5_R

#define RC6_W 32
#define RC6_R 20
#define RC6_B 16
#define RC6_BLOCK_LEN (RC6_W * 4 / 8)
#define RC6_ROUNDS RC6_R

#define STANSOFTSEC_INCLUDE_RC5
#define STANSOFTSEC_INCLUDE_RC6
#define STANSOFTSEC_INCLUDE_RC5_32
#define STANSOFTSEC_INCLUDE_RC6_32
#define STANSOFTSEC_INCLUDE_RC5_64
#define STANSOFTSEC_INCLUDE_RC6_64

#define REVERSE_BYTE_ORDER(x) x
#define REVERSE_BYTE_ORDER_64(x) x

/* maximum number of rounds for RC5 and RC6 */
#define RC_N_MAX 255 /* maximum number of rounds */
/* You may set RC_N_MAX as small as your application allows since its size
   affects stack usage. The default value is 255. */

#if defined(STANSOFTSEC_INCLUDE_RC6_64) || defined(STANSOFTSEC_INCLUDE_RC5_64)
#define RC_KBUF_SIZE (((RC_N_MAX << 1) + 4) << 3) /* (2*RC_N_MAX + 4)*8 */
#else /* defined(STANSOFTSEC_INCLUDE_RC6_32) || defined(STANSOFTSEC_INCLUDE_RC5_32) */
#define RC_KBUF_SIZE (((RC_N_MAX << 1) + 4) << 2) /* (2*RC_N_MAX + 4)*4 */
#endif

#define MAX_BLOCK_LENGTH 32

#define ROTL(x,n) (((x)<<(n))|((x)>>(32-(n))))
#define ROTR(x,n) (((x)>>(n))|((x)<<(32-(n))))
#define ROTL64(x,n) (((x)<<((UINT32)(n)))|((x)>>((UINT32)(64-(n)))))
#define ROTR64(x,n) (((x)>>((UINT32)(n)))|((x)<<((UINT32)(64-(n)))))

#define SS_MEMCPY MEMCPY
#define SS_MEMSET MEMSET

#define MAX(a,b) (((a) > (b)) ? (a) : (b))

/* Cryption mode */
typedef enum {
  kEncrypt, /* Encryption */
  kDecrypt  /* Decryption */
} CryptionMode;

/* Block modes */
typedef enum {
  kECB, /* Electronic-codebook */
  kCBC, /* Cipher-block chaining */
  kCFB, /* Cipher feedback */
  kOFB, /* Output feedback */
  kCTR  /* Counter mode */
} BlockMode;

typedef struct {

  /************* Parameters that are common to all block ciphers ***********/
  CryptionMode cm; /* kEncrypt or kDecrypt */

  BlockMode bm; /* kECB (Electronic Code Book) or kCBC (Cipher Block Chaining) */

  BYTE *key; /* Cryption key */
  UINT32 keyLengthInBits; /* Length of cryption key in bits */

  BYTE *IV; /* Initialization vector must be provided if block mode kCBC is used (set to NULL when using kECB) */
  UINT32 IVLengthInBytes; /* IV length in bytes (set to 0 when using block mode kECB) (IV length must be same as the block length of the cipher) */

  BYTE *output; /* Output buffer (this is where the cryption result, the cipher text, is placed) */
  UINT32 outputBufferLengthInBytes; /* Length of output buffer (in bytes) */

  UINT32 outputLengthInBytes; /* Num bytes of actual output */
  int Status; /* SS_OK if computation was successful, an error code if not successful (see sserr.h) */

  UINT32 numBytesInInputBlock;
  BYTE InputBlock[MAX_BLOCK_LENGTH];
  BYTE XorBlock[MAX_BLOCK_LENGTH];

  /* RC5-specific parameters */

  /* word size parameter for the RC5 algorithm.
     w must be 32 or 64. */
  UINT32 w;

  /* round parameter for the RC5 algorithm.
     r must be 255 or less. */
  UINT32 r;

  /* key length (in bytes) parameter for the RC5 algorithm.
     b overrides keyLengthInBits.
     b must be 255 or less. */
  UINT32 b;

  /* internal RC5-specific parameters */
  BYTE K[RC_KBUF_SIZE];

} RC5_info;

typedef RC5_info RC6_info;



#if defined(STANSOFTSEC_INCLUDE_RC5) || defined(STANSOFTSEC_INCLUDE_RC6)

/* RC5 & RC6 magic constants */
#define P32 0xb7e15163
#define Q32 0x9e3779b9
#define P64 0xb7e151628aed2a6b
#define Q64 0x9e3779b97f4a7c15

#if defined(STANSOFTSEC_INCLUDE_RC5_32) && defined(STANSOFTSEC_INCLUDE_RC5_64)
#define RC5_W_NOT_OK(w) (w != 32) && (w != 64) /* word length must be 32 or 64 */
#elif defined(STANSOFTSEC_INCLUDE_RC5_64)) /* but not RC5_32 */
#define RC5_W_NOT_OK(w) w != 64 /* word length must be 64 */
#else /* 32 but not 64 */
#define RC5_W_NOT_OK(w) w != 32 /* word length must be 32 */
#endif

#if defined(STANSOFTSEC_INCLUDE_RC6_32) && defined(STANSOFTSEC_INCLUDE_RC6_64)
#define RC6_W_NOT_OK(w) (w != 32) && (w != 64) /* word length must be 32 or 64 */
#elif defined(STANSOFTSEC_INCLUDE_RC6_64)) /* but not RC6_32 */
#define RC6_W_NOT_OK(w) w != 64 /* word length must be 64 */
#else /* 32 but not 64 */
#define RC6_W_NOT_OK(w) w != 32 /* word length must be 32 */
#endif

#define RET(H,err) \
  { \
    H->Status = err; \
    return; \
  }

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if big endian platform */
#define KEY_REVERSAL(L,c,TYPE,REV) \
  if (BIG_ENDIAN_PLATFORM) \
  { \
    REV((TYPE *)L, (UINT32)c); \
  }
#elif defined(BIG_ENDIAN)
#define KEY_REVERSAL(L,c,TYPE,REV) REV((TYPE *)L, (UINT32)c);
#else
#define KEY_REVERSAL(L,c,TYPE,REV) /* Do nothing */
#endif

#define MAX_L_SIZE 256 /* based on maximum key size */

#define KEY_INITIALIZATION(H,w,r,bkey,K,TYPE,wordsh,trpt) \
  { \
    UINT32 u, c, cu, i, j, t, s; \
    TYPE Ak, Bk; \
    BYTE L_bytes[MAX_L_SIZE]; \
    TYPE *L; \
    \
    u = w >> 3; /* Num bytes per word */ \
    i = trpt * u; \
    K = (TYPE*)(((RC5_info*)H)->K); \
    SS_MEMSET(((BYTE*)K), 0, (i)); \
    L = (TYPE*)L_bytes; \
    SS_MEMSET(((BYTE*)L_bytes), 0, MAX_L_SIZE); \
    SS_MEMCPY(((BYTE*)L_bytes), ((BYTE*)H->key), bkey); \
    c = (bkey + u - 1)/u; /* Num words K (expanded key buffer) fills */ \
    cu = ((c)?(c*u):(u)); \
    (void)cu; \
    if (u == 4) \
    { \
      KEY_REVERSAL(L, cu>>2, UINT32, reverseByteOrder) \
    } \
    else /* u == 8 */ \
    { \
      KEY_REVERSAL(L, cu>>3, UINT64, reverseByteOrder64) \
    }

#define KEY_EXPANSION(K,trpt,P,Q) \
    K[0] = P; /* P == P32 for w == 32*/ \
    for (i=1; i<trpt; i++) \
      K[i] = K[i-1] + Q;

#define KEY_MIXING(K,trpt,TYPE,ROT) \
    i = j = 0; \
    Ak = Bk = 0; \
    t = MAX(c, trpt); \
    for (s=1; s<(3*t)+1; s++) \
    { \
      TYPE temp, temp2; \
      \
      temp = K[i] + Ak + Bk; \
      Ak = ROT(temp, 3); \
      K[i] = Ak; \
      if (++i >= trpt) \
        i = 0; \
      temp2 = Ak + Bk; \
      temp = L[j] + temp2; \
      Bk = ROT(temp, temp2); \
      L[j] = Bk; \
      if (++j >= c) \
        j = 0; \
    } \
    SS_MEMSET(((BYTE*)L_bytes), 0, MAX_L_SIZE); /* clear L to prevent adversary from using this information */ \
  }

#define KEY_SCHEDULE_32(H,w,r,bkey,K,trpt) \
      KEY_INITIALIZATION(H,w,r,bkey,K,UINT32,2,trpt) \
      KEY_EXPANSION(K,trpt,P32,Q32) \
      KEY_MIXING(K,trpt,UINT32,ROTL)

#define KEY_SCHEDULE_64(H,w,r,bkey,K,trpt) \
      KEY_INITIALIZATION(H,w,r,bkey,K,UINT64,3,trpt) \
      KEY_EXPANSION(K,trpt,P64,Q64) \
      KEY_MIXING(K,trpt,UINT64,ROTL64)

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if big endian platform */
#define A_B_REVERSAL(A,B,REV) if (BIG_ENDIAN_PLATFORM) { REV(A) REV(B) }
#define A_B_C_D_REVERSAL(A,B,C,D,REV) if (BIG_ENDIAN_PLATFORM) { REV(A) REV(B) REV(C) REV(D) }
#define REVERSE_RC5_OUTPUT(B,TYPE,REV) \
  if (BIG_ENDIAN_PLATFORM) \
  { \
    TYPE *outputPtr = (TYPE*)B->output; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); \
  }
#define REVERSE_RC6_OUTPUT(B,TYPE,REV) \
  if (BIG_ENDIAN_PLATFORM) \
  { \
    TYPE *outputPtr = (TYPE*)B->output; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); \
  }
#elif defined(BIG_ENDIAN)
#define A_B_REVERSAL(A,B,REV) REV(A) REV(B)
#define A_B_C_D_REVERSAL(A,B,C,D,REV) REV(A) REV(B) REV(C) REV(D)
#define REVERSE_RC5_OUTPUT(B,TYPE,REV) \
  { \
    TYPE *outputPtr = (TYPE*)B->output; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); \
  }
#define REVERSE_RC6_OUTPUT(B,TYPE,REV) \
  { \
    TYPE *outputPtr = (TYPE*)B->output; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); outputPtr++; \
    REV(*outputPtr); \
  }
#else
#define A_B_REVERSAL(A,B,REV) /* Do nothing */
#define A_B_C_D_REVERSAL(A,B,C,D,REV) /* Do nothing */
#define REVERSE_RC5_OUTPUT(B,TYPE,REV) /* do nothing */
#define REVERSE_RC6_OUTPUT(B,TYPE,REV) /* do nothing */
#endif

#define CHECK_RC5_PARAMS(H,cm,bm,w,r,bkey) \
    if (!(H->output)) /* No provided output buffer */ \
      RET(H,-1) /* Invalid output buffer */ \
    if (RC5_W_NOT_OK(w)) \
      RET(H,-1) /* invalid word length */ \
    if (r > RC_N_MAX) /* max RC_N_MAX rounds (max 255 according to the RC5 specification) */ \
      RET(H,-1) /* invalid number of rounds */ \
    if (bkey > 255) /* max 255 byte keys according to the RC5 and RC6 specifications */ \
      RET(H,-1) /* invalid key length */ \
    if ((cm != kEncrypt) && (cm != kDecrypt)) \
      RET(H,-1) /* invalid cryption mode */ \
    if ((bm != kECB) && (bm != kCBC)) \
      RET(H,-1) /* invalid block mode */ \
    if (!(H->key) && (H->keyLengthInBits != 0)) \
      RET(H,-1) /* invalid key buffer */

#define RC5_ENCRYPT_ONE_BLOCK(B,x,TYPE,ROT,REV) \
  { \
    TYPE A, C; \
    UINT32 round; \
    TYPE *KPtr = (TYPE*)B->K; \
    UINT32 r = B->r; \
    TYPE *xp = (TYPE*)x; \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
  	 \
    A = *xp++; /* get next input block */ \
    C = *xp; \
    A_B_REVERSAL(A,C,REV) \
    A += *KPtr++; \
    C += *KPtr; \
    for (round=0; round<r; round++) \
    { \
      A ^= C; \
      A = ROT(A,C) + *(++KPtr); \
      C ^= A; \
      C = ROT(C,A) + *(++KPtr); \
    } \
    *outputPtr++ = A; /* crypted block to output buffer */ \
    *outputPtr = C; \
    REVERSE_RC5_OUTPUT(B,TYPE,REV) \
  }
#define RC5_32_ENCRYPT_ONE_BLOCK(B,x) RC5_ENCRYPT_ONE_BLOCK(B,x,UINT32,ROTL,REVERSE_BYTE_ORDER)
#define RC5_64_ENCRYPT_ONE_BLOCK(B,x) RC5_ENCRYPT_ONE_BLOCK(B,x,UINT64,ROTL64,REVERSE_BYTE_ORDER_64)

#define RC5_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,TYPE,ROT,REV) \
  { \
    TYPE A, C; \
    UINT32 round; \
    TYPE *KPtr = (TYPE*)B->K; \
    UINT32 r = B->r; \
    TYPE *xp = (TYPE*)x; \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
  	 \
    A = *xp++; /* get next input block */ \
    C = *xp; \
    A_B_REVERSAL(A,C,REV) \
    A += *KPtr++; \
    C += *KPtr; \
    for (round=0; round<r-1; round++) \
    { \
      A ^= C; \
      A = ROT(A,C) + *(++KPtr); \
      C ^= A; \
      C = ROT(C,A) + *(++KPtr); \
      \
      *outputPtr++ = A; /* intermediate block to output buffer */ \
      *outputPtr++ = C; \
      if (--numBlocks == 0) return; \
    } \
    \
    A ^= C; \
    A = ROT(A,C) + *(++KPtr); \
    C ^= A; \
    C = ROT(C,A) + *(++KPtr); \
    \
    *outputPtr++ = A; /* crypted block to output buffer */ \
    *outputPtr = C; \
    \
    /* extract lastBlock */ \
    outputPtr = (TYPE*)lastBlock; \
    *outputPtr++ = A; \
    *outputPtr = C; \
  }
#define RC5_32_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks) RC5_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,UINT32,ROTL,REVERSE_BYTE_ORDER)
#define RC5_64_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks) RC5_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,UINT64,ROTL64,REVERSE_BYTE_ORDER_64)

#define RC5_DECRYPT_ONE_BLOCK(B,x,TYPE,ROT,REV) \
  { \
    UINT32 round; \
    TYPE *KPtr; \
    TYPE A, C; \
    UINT32 r = B->r; \
    UINT32 trpt = (r+1)<<1; /* 2*r + 2 for RC5 */ \
    TYPE *K = (TYPE*)B->K;   /* word pointer to round key buffer */ \
    TYPE *xp = (TYPE*)x; \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
  	 \
    A = *xp++; /* Get next input block */ \
    C = *xp; \
    A_B_REVERSAL(A,C,REV) \
    KPtr = (TYPE*)B->K + trpt; \
    for (round=r; round>0; round--) \
    { \
      C -= *(--KPtr); \
      C = ROT(C,A); \
      C ^= A; \
      A -= *(--KPtr); \
      A = ROT(A,C); \
      A ^= C; \
    } \
    *outputPtr++ = A - K[0]; /* crypted block to output buffer */ \
    *outputPtr = C - K[1]; \
    REVERSE_RC5_OUTPUT(B,TYPE,REV) \
  }
#define RC5_32_DECRYPT_ONE_BLOCK(B,x) RC5_DECRYPT_ONE_BLOCK(B,x,UINT32,ROTR,REVERSE_BYTE_ORDER)
#define RC5_64_DECRYPT_ONE_BLOCK(B,x) RC5_DECRYPT_ONE_BLOCK(B,x,UINT64,ROTR64,REVERSE_BYTE_ORDER_64)


#if defined(STANSOFTSEC_INCLUDE_RC5)
/*
  Function RC5_crypt

  Description: RC5 cryption function

  Input:
    'B->cm' is the cryption mode (kEncrypt or kDecrypt)
    'B->bm' is the block mode (kECB or kCBC)
    'B->w' is the word length (32 or 64)
    'B->r' is the number of rounds (usually 12 with w=32 and 16 with w=64)
    'B->b' is the key length in bytes
    'B->key' is the RC5 key ('B->b' bytes) to be used in the encryption
    'B->input' is the data to be crypted (the plaintext when encrypting)
    'B->inputLengthInBytes' is the length of the data to be crypted
    'B->IV' is the IV (initialization vector) to be used (not used when B->bm is kECB, set to NULL)
    'B->IVLengthInBytes' is the length of the IV (not used when B->bm is kECB, set to 0)
    'B->output' is a buffer where the ciphertext is placed
    'B->outputBufferLengthInBytes' is the length of the output buffer in bytes

  Output:
    'B->Status' is the Status of the calculation (SS_OK if computed)
    'B->output' contains the RC5 decryption of 'x'
    'B->outputLengthInBytes' is the length of the output in bytes
*/

void RC5_init(RC5_info *B)
{
  CryptionMode cm = B->cm;
  BlockMode bm = B->bm;
  UINT32 w = B->w, r = B->r, bkey = B->b;

  CHECK_RC5_PARAMS(B,cm,bm,w,r,bkey)
  if (w == 32)
  {
#ifdef STANSOFTSEC_INCLUDE_RC5_32
    UINT32 *K;                 /* word pointer to round key buffer */
    UINT32 trpt = (r+1)<<1;    /* 2*r + 2 for RC5 */
    KEY_SCHEDULE_32(B,32,r,bkey,K,trpt) /* K pointer to expanded key buffer in BI */
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  else /* w == 64 */
  {
#ifdef STANSOFTSEC_INCLUDE_RC5_64
    UINT64 *K;                 /* word pointer to round key buffer */
    UINT32 trpt = (r+1)<<1;    /* 2*r + 2 for RC5 */
    KEY_SCHEDULE_64(B,64,r,bkey,K,trpt) /* K pointer to expanded key buffer in BI */
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  B->Status = 0;
}

void RC5_cryptBlock(RC5_info *B, const BYTE *x)
{
  if (B->w == 32)
  {
#ifdef STANSOFTSEC_INCLUDE_RC5_32
    if (B->cm == kEncrypt)
      RC5_32_ENCRYPT_ONE_BLOCK(B,x)
    else /* cm == kDecrypt */
      RC5_32_DECRYPT_ONE_BLOCK(B,x)
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  else /* w == 64 */
  {
#ifdef STANSOFTSEC_INCLUDE_RC5_64
    if (B->cm == kEncrypt)
      RC5_64_ENCRYPT_ONE_BLOCK(B,x)
    else /* cm == kDecrypt */
      RC5_64_DECRYPT_ONE_BLOCK(B,x)
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  B->Status = 0;
}

void RC5_cryptBlock_xor_withInitOutput(RC5_info *B, const BYTE *x, BYTE *lastBlock, int numBlocks)
{
  if (B->cm != kEncrypt) {
    B->Status = -1;
    return;
  }

  if (B->w == 32) {
    RC5_32_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks)
  } else { /* w == 64 */
    RC5_64_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks)
  }
  B->Status = 0;
}

void RC5_final(RC5_info *B) { B->Status = 0; }
#endif /* STANSOFTSEC_INCLUDE_RC5 */


#define CHECK_RC6_PARAMS(H,cm,bm,w,r,bkey) \
    if (!(H->output)) /* No provided output buffer */ \
      RET(H,-1) /* Invalid output buffer */ \
    if (RC6_W_NOT_OK(w)) \
      RET(H,-1) /* invalid word length */ \
    if (r > RC_N_MAX) /* max RC_N_MAX rounds (max 255 according to the RC5 specification) */ \
      RET(H,-1) /* invalid number of rounds */ \
    if (bkey > 255) /* max 255 byte keys according to the RC5 and RC6 specifications */ \
      RET(H,-1) /* invalid key length */ \
    if ((cm != kEncrypt) && (cm != kDecrypt)) \
      RET(H,-1) /* invalid cryption mode */ \
    if ((bm != kECB) && (bm != kCBC)) \
      RET(H,-1) /* invalid block mode */ \
    if (!(H->key) && (H->keyLengthInBits != 0)) \
      RET(H,-1) /* invalid key buffer */

#define RC6_ENCRYPT_ONE_BLOCK(B,x,lgw,TYPE,ROT,REV) \
  { \
    UINT32 round; \
    TYPE temp, t, u; \
    TYPE *KPtr; \
    UINT32 r = B->r; \
    TYPE *xp = (TYPE*)x;     /* word pointer to input data */ \
    TYPE *K = (TYPE*)B->K;    /* word pointer to round key buffer */ \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
    TYPE A, H, C, D; \
    \
    A = *xp++; H = *xp++; C = *xp++; D = *xp; /* get next input block */ \
    A_B_C_D_REVERSAL(A,H,C,D,REV) \
    KPtr = K; \
    H += *KPtr++; \
    D += *KPtr; \
    for (round=0; round<r; round++) \
    { \
   	  temp = (H)*(((H)<<1)+1); \
      t = ROT(temp,lgw); /* lg w */ \
      temp = (D)*(((D)<<1)+1); \
      u = ROT(temp,lgw); /* lg w */ \
      A ^= t; \
      A = ROT(A,u) + *(++KPtr); \
      C ^= u; \
      C = ROT(C,t) + *(++KPtr); \
      temp = A; A = H; H = C; C = D; D = temp; \
    } \
    A += *(++KPtr); \
    C += *(++KPtr); \
    *outputPtr++ = A; /* crypted block to output buffer */ \
    *outputPtr++ = H; \
    *outputPtr++ = C; \
    *outputPtr   = D; \
    REVERSE_RC6_OUTPUT(B,TYPE,REV) \
  }
#define RC6_32_ENCRYPT_ONE_BLOCK(B,x) RC6_ENCRYPT_ONE_BLOCK(B,x,5,UINT32,ROTL,REVERSE_BYTE_ORDER)
#define RC6_64_ENCRYPT_ONE_BLOCK(B,x) RC6_ENCRYPT_ONE_BLOCK(B,x,6,UINT64,ROTL64,REVERSE_BYTE_ORDER_64)

#define RC6_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,lgw,TYPE,ROT,REV) \
  { \
    UINT32 round; \
    TYPE temp, t, u; \
    TYPE *KPtr; \
    UINT32 r = B->r; \
    TYPE *xp = (TYPE*)x;     /* word pointer to input data */ \
    TYPE *K = (TYPE*)B->K;    /* word pointer to round key buffer */ \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
    TYPE A, H, C, D; \
    \
    A = *xp++; H = *xp++; C = *xp++; D = *xp; /* get next input block */ \
    A_B_C_D_REVERSAL(A,H,C,D,REV) \
    KPtr = K; \
    H += *KPtr++; \
    D += *KPtr; \
    for (round=0; round<r-1; round++) \
    { \
   	  temp = (H)*(((H)<<1)+1); \
      t = ROT(temp,lgw); /* lg w */ \
      temp = (D)*(((D)<<1)+1); \
      u = ROT(temp,lgw); /* lg w */ \
      A ^= t; \
      A = ROT(A,u) + *(++KPtr); \
      C ^= u; \
      C = ROT(C,t) + *(++KPtr); \
      temp = A; A = H; H = C; C = D; D = temp; \
      \
      *outputPtr++ = A; /* crypted block to output buffer */ \
      *outputPtr++ = H; \
      *outputPtr++ = C; \
      *outputPtr++ = D; \
      if (--numBlocks == 0) return; \
    } \
    \
    temp = (H)*(((H)<<1)+1); \
    t = ROT(temp,lgw); /* lg w */ \
    temp = (D)*(((D)<<1)+1); \
    u = ROT(temp,lgw); /* lg w */ \
    A ^= t; \
    A = ROT(A,u) + *(++KPtr); \
    C ^= u; \
    C = ROT(C,t) + *(++KPtr); \
    temp = A; A = H; H = C; C = D; D = temp; \
    \
    A += *(++KPtr); \
    C += *(++KPtr); \
    *outputPtr++ = A; /* crypted block to output buffer */ \
    *outputPtr++ = H; \
    *outputPtr++ = C; \
    *outputPtr   = D; \
    \
    /* extract lastBlock */ \
    outputPtr = (TYPE*)lastBlock; \
    *outputPtr++ = A; \
    *outputPtr++ = H; \
    *outputPtr++ = C; \
    *outputPtr   = D; \
  }
#define RC6_32_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks) RC6_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,5,UINT32,ROTL,REVERSE_BYTE_ORDER)
#define RC6_64_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks) RC6_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks,6,UINT64,ROTL64,REVERSE_BYTE_ORDER_64)

#define RC6_DECRYPT_ONE_BLOCK(B,x,lgw,TYPE,ROT,REV) \
  { \
    UINT32 round; \
    TYPE temp, t, u; \
    TYPE *KPtr; \
    TYPE A, H, C, D; \
    UINT32 r = B->r; \
    UINT32 trpt = (r+2)<<1; /* 2*r + 4 for RC6 */ \
    TYPE *xp = (TYPE*)x;     /* word pointer to input data */ \
    TYPE *K = (TYPE*)B->K;    /* word pointer to round key buffer */ \
    TYPE *outputPtr = (TYPE*)B->output; /* word pointer to output buffer */ \
    \
    A = *xp++; H = *xp++; C = *xp++; D = *xp; /* Get next input block */ \
    A_B_C_D_REVERSAL(A,H,C,D,REV) \
    KPtr = K + trpt; \
    C -= *(--KPtr); \
    A -= *(--KPtr); \
    for (round=r; round>0; round--) \
    { \
      temp = A; A = D; D = C; C = H; H = temp; \
      temp = (H)*(((H)<<1)+1); \
      t = ROT(temp,32-lgw); /* lg w */ \
      temp = (D)*(((D)<<1)+1); \
      u = ROT(temp,32-lgw); /* lg w */ \
      C -= *(--KPtr); \
      C = ROT(C,t) ^ u; \
      A -= *(--KPtr); \
      A = ROT(A,u) ^ t; \
    } \
    D -= *(--KPtr); \
    H -= *(--KPtr); \
    *outputPtr++ = A; /* crypted block to output buffer */ \
    *outputPtr++ = H; \
    *outputPtr++ = C; \
    *outputPtr   = D; \
    REVERSE_RC6_OUTPUT(B,TYPE,REV) \
  }
#define RC6_32_DECRYPT_ONE_BLOCK(B,x) RC6_DECRYPT_ONE_BLOCK(B,x,5,UINT32,ROTR,REVERSE_BYTE_ORDER)
#define RC6_64_DECRYPT_ONE_BLOCK(B,x) RC6_DECRYPT_ONE_BLOCK(B,x,6,UINT64,ROTR64,REVERSE_BYTE_ORDER_64)

#if defined(STANSOFTSEC_INCLUDE_RC6)
/*
  Function RC6_crypt

  Description: RC6 cryption function

  Input:
    'B->cm' is the cryption mode (kEncrypt or kDecrypt)
    'B->bm' is the block mode (kECB or kCBC)
    'B->w' is the word length (32 or 64)
    'B->r' is the number of rounds (usually 20 with w=32)
    'B->b' is the key length in bytes
    'B->key' is the RC6 key ('B->b' bytes) to be used in the encryption
    'B->input' is the data to be crypted (the plaintext when encrypting)
    'B->inputLengthInBytes' is the length of the data to be crypted
    'B->IV' is the IV (initialization vector) to be used (not used when B->bm is kECB, set to NULL)
    'B->IVLengthInBytes' is the length of the IV (not used when B->bm is kECB, set to 0)
    'B->output' is a buffer where the ciphertext is placed
    'B->outputBufferLengthInBytes' is the length of the output buffer in bytes

  Output:
    'B->Status' is the Status of the calculation (SS_OK if computed)
    'B->output' contains the RC6 decryption of 'x'
    'B->outputLengthInBytes' is the length of the output in bytes
*/

void RC6_init(RC6_info *B)
{
  CryptionMode cm = B->cm;
  BlockMode bm = B->bm;
  UINT32 w = B->w, r = B->r, bkey = B->b;

  CHECK_RC6_PARAMS(B,cm,bm,w,r,bkey)
  if (w == 32)
  {
#ifdef STANSOFTSEC_INCLUDE_RC6_32
    UINT32 *K;                 /* word pointer to round key buffer */
    UINT32 trpt = (r+2)<<1;    /* 2*r + 4 for RC6 */
    KEY_SCHEDULE_32(B,32,r,bkey,K,trpt) /* K allocated in key schedule */
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  else /* w == 64 */
  {
#ifdef STANSOFTSEC_INCLUDE_RC6_64
    UINT64 *K;              /* word pointer to round key buffer */
    UINT32 trpt = (r+2)<<1; /* 2*r + 4 for RC6 */
    KEY_SCHEDULE_64(B,64,r,bkey,K,trpt)
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  B->Status = 0;
}

void RC6_cryptBlock(RC6_info *B, const BYTE *x)
{
  if (B->w == 32)
  {
#ifdef STANSOFTSEC_INCLUDE_RC6_32
    if (B->cm == kEncrypt)
      RC6_32_ENCRYPT_ONE_BLOCK(B,x)
    else /* cm == kDecrypt */
      RC6_32_DECRYPT_ONE_BLOCK(B,x)
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  else /* w == 64 */
  {
#ifdef STANSOFTSEC_INCLUDE_RC6_64
    if (B->cm == kEncrypt)
      RC6_64_ENCRYPT_ONE_BLOCK(B,x)
    else /* cm == kDecrypt */
      RC6_64_DECRYPT_ONE_BLOCK(B,x)
#else
    B->Status = SS_ERROR_UNSUPPORTED_ALGORITHM;
    return;
#endif
  }
  B->Status = 0;
}

void RC6_cryptBlock_xor_withInitOutput(RC6_info *B, const BYTE *x, BYTE *lastBlock, int numBlocks) {

  if (B->cm != kEncrypt) {
    B->Status = -1;
    return;
  }

  if (B->w == 32) {
    RC6_32_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks)
  } else { /* w == 64 */
    RC6_64_ENCRYPT_ONE_BLOCK_XOR_WITH_INIT_OUTPUT(B,x,lastBlock,numBlocks)
  }
  B->Status = 0;
}

void RC6_final(RC6_info *B) { B->Status = 0; }
#endif /* STANSOFTSEC_INCLUDE_RC6 */


#endif /* STANSOFTSEC_INCLUDE_RC5 || STANSOFTSEC_INCLUDE_RC6 */



void RC5_encryptBlock(RC5_info *B, const BYTE *x) {
  RC5_cryptBlock(B, x);
}

void RC5_encryptBlock_xor_withInitOutput(RC5_info *B, const BYTE *x, BYTE *lastBlock, int numBlocks) {
  RC5_cryptBlock_xor_withInitOutput(B, x, lastBlock, numBlocks);
}

void RC6_encryptBlock(RC6_info *B, const BYTE *x) {
  RC6_cryptBlock(B, x);
}

void RC6_encryptBlock_xor_withInitOutput(RC6_info *B, const BYTE *x, BYTE *lastBlock, int numBlocks) {
  RC6_cryptBlock_xor_withInitOutput(B, x, lastBlock, numBlocks);
}

/******************************************************************************
 * Black box variants
 ******************************************************************************/
#define COPY_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d = *s; }
#define XOR_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d ^= *s; }
int RC5_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  RC5_info ctx;
  BYTE pt[RC5_BLOCK_LEN];
  BYTE ct[RC5_BLOCK_LEN];
  int i;
  const int numBlocks = numOutputBytes / RC5_BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes % RC5_BLOCK_LEN) != 0) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.w = RC5_W;
  ctx.r = RC5_R;
  ctx.b = RC5_B;
  ctx.key = (BYTE*)key;
  ctx.keyLengthInBits = RC5_B * 8;
  ctx.output = ct;

  RC5_init(&ctx);
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, i==0 ? iv : ct)
    XOR_BUF_8(pt, inBuf + i * RC5_BLOCK_LEN)
    RC5_encryptBlock(&ctx, pt);
    MEMXOR(outBuf + i * RC5_BLOCK_LEN, ct, 8);
  }
  return 0;
}

int RC5_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  RC5_info ctx;
  BYTE pt[RC5_BLOCK_LEN];
  BYTE ct[RC5_BLOCK_LEN];
  int i;
  int numBlocks = numOutputBytes / RC5_BLOCK_LEN;
  const int numSuppressedBytes = (RC5_ROUNDS - 1) * RC5_BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes % RC5_BLOCK_LEN) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.w = RC5_W;
  ctx.r = RC5_R;
  ctx.b = RC5_B;
  ctx.key = (BYTE*)key;
  ctx.keyLengthInBits = RC5_B * 8;
  ctx.output = outBuf;

  RC5_init(&ctx);

  /* first ROUNDS blocks ((ROUNDS - 1) suppressed + first) */
  COPY_BUF_8(pt, iv)
  XOR_BUF_8(pt, inBuf)
  {
    BYTE buf[RC5_ROUNDS * RC5_BLOCK_LEN];
    BYTE *p = ctx.output;
    ctx.output = buf;
    MEMSET(buf, 0, RC5_ROUNDS * RC5_BLOCK_LEN);
    RC5_encryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
    ctx.output = p;
    MEMXOR(ctx.output, buf, numBlocks < RC5_ROUNDS ? (numBlocks * RC5_BLOCK_LEN) : (RC5_ROUNDS * RC5_BLOCK_LEN));
  }

  if (numBlocks <= RC5_ROUNDS)
    return 0;
  ctx.output = ct;
  numBlocks -= RC5_ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, ct)
    XOR_BUF_8(pt, inBuf + (i + 1) * RC5_BLOCK_LEN)
    RC5_encryptBlock(&ctx, pt);
    MEMXOR(outBuf + (i + RC5_ROUNDS) * RC5_BLOCK_LEN, ct, 8);
  }
  return 0;
}

#define COPY_BUF_16(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d++ = *s++; *d = *s; }
#define XOR_BUF_16(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d++ ^= *s++; *d ^= *s; }
int RC6_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  RC6_info ctx;
  BYTE pt[RC6_BLOCK_LEN];
  BYTE ct[RC6_BLOCK_LEN];
  int i;
  const int numBlocks = numOutputBytes / RC6_BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes % RC6_BLOCK_LEN) != 0) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.w = RC6_W;
  ctx.r = RC6_R;
  ctx.b = RC6_B;
  ctx.key = (BYTE*)key;
  ctx.keyLengthInBits = RC6_B * 8;
  ctx.output = ct;

  RC6_init(&ctx);
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_16(pt, i==0 ? iv : ct)
    XOR_BUF_16(pt, inBuf + i * RC6_BLOCK_LEN)
    RC6_encryptBlock(&ctx, pt);
    MEMXOR(outBuf + i * RC6_BLOCK_LEN, ct, 16);
  }
  return 0;
}

int RC6_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  RC6_info ctx;
  BYTE pt[RC6_BLOCK_LEN];
  BYTE ct[RC6_BLOCK_LEN];
  int i;
  int numBlocks = numOutputBytes / RC6_BLOCK_LEN;
  const int numSuppressedBytes = (RC6_ROUNDS - 1) * RC6_BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes % RC6_BLOCK_LEN) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.w = RC6_W;
  ctx.r = RC6_R;
  ctx.b = RC6_B;
  ctx.key = (BYTE*)key;
  ctx.keyLengthInBits = RC6_B * 8;
  ctx.output = outBuf;

  RC6_init(&ctx);

  /* first ROUNDS blocks ((ROUNDS - 1) suppressed + first) */
  COPY_BUF_16(pt, iv)
  XOR_BUF_16(pt, inBuf)
  {
    BYTE buf[RC6_ROUNDS * RC6_BLOCK_LEN];
    BYTE *p = ctx.output;
    ctx.output = buf;
    MEMSET(buf, 0, RC6_ROUNDS * RC6_BLOCK_LEN);
    RC6_encryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
    ctx.output = p;
    MEMXOR(ctx.output, buf, numBlocks < RC6_ROUNDS ? (numBlocks * RC6_BLOCK_LEN) : (RC6_ROUNDS * RC6_BLOCK_LEN));
  }
  if (numBlocks <= RC6_ROUNDS)
    return 0;
  ctx.output = ct;
  numBlocks -= RC6_ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_16(pt, ct)
    XOR_BUF_16(pt, inBuf + (i + 1) * RC6_BLOCK_LEN)
    RC6_encryptBlock(&ctx, pt);
    MEMXOR(outBuf + (i + RC6_ROUNDS) * RC6_BLOCK_LEN, ct, 16);
  }
  return 0;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
int blackBoxRC5Encryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
      return RC5_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return RC5_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

int blackBoxRC6Encryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
      return RC6_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return RC6_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxRC5Info(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = RC5_B;
  if (ivSizeInBytes) *ivSizeInBytes = RC5_BLOCK_LEN;
  if (suppressedBytes) *suppressedBytes = (RC5_ROUNDS - 1) * RC5_BLOCK_LEN;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = RC5_BLOCK_LEN;
}

void getBlackBoxRC6Info(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = RC6_B;
  if (ivSizeInBytes) *ivSizeInBytes = RC6_BLOCK_LEN;
  if (suppressedBytes) *suppressedBytes = (RC6_ROUNDS - 1) * RC6_BLOCK_LEN;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = RC6_BLOCK_LEN;
}

