/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_present.h"

#define BLOCK_LEN 8
#define ROUNDS 31

#define REVERSE_BYTE_ORDER_32(x) \
  { \
    UINT32 v_rev = (((x) << 16) | ((x) >> 16)); \
    x = ( ((v_rev & 0xff00ff00UL) >> 8) | ((v_rev & 0x00ff00ffUL) << 8) ); \
  }

/* reverse byte order of one 64-bit word */
#define REVERSE_BYTE_ORDER_64(x) \
  { \
    UINT32 \
      a = (UINT32)((x) >> 32), \
      b = (UINT32)((x) & 0x00000000ffffffff); \
    REVERSE_BYTE_ORDER_32(a) \
    REVERSE_BYTE_ORDER_32(b) \
    x = (((UINT64)b)<<32) | (((UINT64)a)); \
  }

const UINT64 sbox[16] = {
  0xC000000000000000,
  0x5000000000000000,
  0x6000000000000000,
  0xB000000000000000,
  0x9000000000000000,
  0x0000000000000000,
  0xA000000000000000,
  0xD000000000000000,
  0x3000000000000000,
  0xE000000000000000,
  0xF000000000000000,
  0x8000000000000000,
  0x4000000000000000,
  0x7000000000000000,
  0x1000000000000000,
  0x2000000000000000
};

const UINT64 S[256] = {
  0xCC,0xC5,0xC6,0xCB,0xC9,0xC0,0xCA,0xCD,0xC3,0xCE,0xCF,0xC8,0xC4,0xC7,0xC1,0xC2,
  0x5C,0x55,0x56,0x5B,0x59,0x50,0x5A,0x5D,0x53,0x5E,0x5F,0x58,0x54,0x57,0x51,0x52,
  0x6C,0x65,0x66,0x6B,0x69,0x60,0x6A,0x6D,0x63,0x6E,0x6F,0x68,0x64,0x67,0x61,0x62,
  0xBC,0xB5,0xB6,0xBB,0xB9,0xB0,0xBA,0xBD,0xB3,0xBE,0xBF,0xB8,0xB4,0xB7,0xB1,0xB2,
  0x9C,0x95,0x96,0x9B,0x99,0x90,0x9A,0x9D,0x93,0x9E,0x9F,0x98,0x94,0x97,0x91,0x92,
  0x0C,0x05,0x06,0x0B,0x09,0x00,0x0A,0x0D,0x03,0x0E,0x0F,0x08,0x04,0x07,0x01,0x02,
  0xAC,0xA5,0xA6,0xAB,0xA9,0xA0,0xAA,0xAD,0xA3,0xAE,0xAF,0xA8,0xA4,0xA7,0xA1,0xA2,
  0xDC,0xD5,0xD6,0xDB,0xD9,0xD0,0xDA,0xDD,0xD3,0xDE,0xDF,0xD8,0xD4,0xD7,0xD1,0xD2,
  0x3C,0x35,0x36,0x3B,0x39,0x30,0x3A,0x3D,0x33,0x3E,0x3F,0x38,0x34,0x37,0x31,0x32,
  0xEC,0xE5,0xE6,0xEB,0xE9,0xE0,0xEA,0xED,0xE3,0xEE,0xEF,0xE8,0xE4,0xE7,0xE1,0xE2,
  0xFC,0xF5,0xF6,0xFB,0xF9,0xF0,0xFA,0xFD,0xF3,0xFE,0xFF,0xF8,0xF4,0xF7,0xF1,0xF2,
  0x8C,0x85,0x86,0x8B,0x89,0x80,0x8A,0x8D,0x83,0x8E,0x8F,0x88,0x84,0x87,0x81,0x82,
  0x4C,0x45,0x46,0x4B,0x49,0x40,0x4A,0x4D,0x43,0x4E,0x4F,0x48,0x44,0x47,0x41,0x42,
  0x7C,0x75,0x76,0x7B,0x79,0x70,0x7A,0x7D,0x73,0x7E,0x7F,0x78,0x74,0x77,0x71,0x72,
  0x1C,0x15,0x16,0x1B,0x19,0x10,0x1A,0x1D,0x13,0x1E,0x1F,0x18,0x14,0x17,0x11,0x12,
  0x2C,0x25,0x26,0x2B,0x29,0x20,0x2A,0x2D,0x23,0x2E,0x2F,0x28,0x24,0x27,0x21,0x22
};

typedef struct {
  const BYTE *key;
  const BYTE *iv;
  BYTE *out;
  UINT64 state;
  UINT64 K[2];
  UINT64 roundCounter;
} PRESENT_info;



void sBoxLayer(PRESENT_info *B) {
  UINT64 s = B->state;
  B->state =
     S[(int)( s        & 0xFF)] |
    (S[(int)((s >>  8) & 0xFF)] <<  8) |
    (S[(int)((s >> 16) & 0xFF)] << 16) |
    (S[(int)((s >> 24) & 0xFF)] << 24) |
    (S[(int)((s >> 32) & 0xFF)] << 32) |
    (S[(int)((s >> 40) & 0xFF)] << 40) |
    (S[(int)((s >> 48) & 0xFF)] << 48) |
    (S[(int)((s >> 56) & 0xFF)] << 56);
}

UINT64 p(UINT64 x) {
  UINT64 y = x & 0x8000040000200001;
  /* negative movement */
  y |= (x & 0x0000400002000010) >> 3;
  y |= (x & 0x0004000020000100) >> 6;
  y |= (x & 0x0040000200001000) >> 9;
  y |= (x & 0x0400002000010000) >> 12;
  y |= (x & 0x4000020000100000) >> 15;
  y |= (x & 0x0000200001000000) >> 18;
  y |= (x & 0x0002000010000000) >> 21;
  y |= (x & 0x0020000100000000) >> 24;
  y |= (x & 0x0200001000000000) >> 27;
  y |= (x & 0x2000010000000000) >> 30;
  y |= (x & 0x0000100000000000) >> 33;
  y |= (x & 0x0001000000000000) >> 36;
  y |= (x & 0x0010000000000000) >> 39;
  y |= (x & 0x0100000000000000) >> 42;
  y |= (x & 0x1000000000000000) >> 45;
  /* positive movement */
  y |= (x & 0x0800004000020000) << 3;
  y |= (x & 0x0080000400002000) << 6;
  y |= (x & 0x0008000040000200) << 9;
  y |= (x & 0x0000800004000020) << 12;
  y |= (x & 0x0000080000400002) << 15;
  y |= (x & 0x0000008000040000) << 18;
  y |= (x & 0x0000000800004000) << 21;
  y |= (x & 0x0000000080000400) << 24;
  y |= (x & 0x0000000008000040) << 27;
  y |= (x & 0x0000000000800004) << 30;
  y |= (x & 0x0000000000080000) << 33;
  y |= (x & 0x0000000000008000) << 36;
  y |= (x & 0x0000000000000800) << 39;
  y |= (x & 0x0000000000000080) << 42;
  y |= (x & 0x0000000000000008) << 45;
  return y;
}

void permutationLayer(PRESENT_info *B) {
  B->state = p(B->state);
}

void keySchedulePreparation(PRESENT_info *B) {
  UINT64 t = *((UINT64*)(B->key));

  REVERSE_BYTE_ORDER_64(t)
  B->K[0] = t;
  B->K[1] = ((UINT64)B->key[8] << 56) | ((UINT64)B->key[9] << 48);
}

void addRoundKey(PRESENT_info *B) {
  UINT64 t = B->K[0];

  /* add current key */
  B->state ^= t;

  /* update key register */

  /* rotate 61 bits left */
  B->K[0] = (t << 61) | (B->K[1] >> 3) | (t >> 19);
  B->K[1] = (t << 45) & U64C(0xFFFF000000000000);

  /* s-box top four bits */
  B->K[0] = (B->K[0] & U64C(0x0FFFFFFFFFFFFFFF)) | sbox[(int)(B->K[0] >> 60)];

  /* add round counter */
  t = B->roundCounter & 0x1F;
  B->K[0] ^= t >> 1;
  B->K[1] ^= t << 63;

  B->roundCounter++;
}

void PRESENT_init(PRESENT_info *B) {
  B->roundCounter = 1;
  keySchedulePreparation(B);
}

#define COPY_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d = *s; }
#define XOR_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d ^= *s; }
void PRESENT_encryptBlock(PRESENT_info *B, BYTE *x) {
  int i;
  UINT64 t;
  COPY_BUF_8(&t, x)
  REVERSE_BYTE_ORDER_64(t)
  B->state = t;

  for (i=0; i<ROUNDS; i++) {
    addRoundKey(B);
    sBoxLayer(B);
    permutationLayer(B);
  }
  addRoundKey(B);

  t = B->state;
  REVERSE_BYTE_ORDER_64(t)
  COPY_BUF_8(B->out, &t)
}

void PRESENT_encryptBlock_xor_withInitOutput(PRESENT_info *B, const BYTE *x, BYTE *lastBlock, int numBlocks) {
  int i;
  UINT64 t;
  BYTE *out = B->out;

  COPY_BUF_8(&t, x)
  REVERSE_BYTE_ORDER_64(t)
  B->state = t;

  addRoundKey(B);
  for (i=0; i<ROUNDS-1; i++) {
    sBoxLayer(B);
    permutationLayer(B);
    addRoundKey(B);
    t = B->state;
    REVERSE_BYTE_ORDER_64(t)
    XOR_BUF_8(out, &t)
    out += BLOCK_LEN;
    if (--numBlocks == 0)
      return;
  }
  sBoxLayer(B);
  permutationLayer(B);
  addRoundKey(B);
  t = B->state;
  REVERSE_BYTE_ORDER_64(t)
  XOR_BUF_8(out, &t)

  COPY_BUF_8(lastBlock, &t)
}

/******************************************************************************
 * Black box variants
 ******************************************************************************/
int PRESENT_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  PRESENT_info ctx;
  BYTE pt[BLOCK_LEN];
  BYTE ct[BLOCK_LEN];
  int i;
  const int numBlocks = numOutputBytes / BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes % BLOCK_LEN) != 0) return -1;

  ctx.key = (BYTE*)key;
  ctx.iv  = (BYTE*)iv;
  ctx.out = ct;

  PRESENT_init(&ctx);
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, i==0 ? iv : ct)
    XOR_BUF_8(pt, inBuf + i * BLOCK_LEN)
    PRESENT_init(&ctx);
    PRESENT_encryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + i * BLOCK_LEN, ct)
  }
  return 0;
}

int PRESENT_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  PRESENT_info ctx;
  BYTE pt[BLOCK_LEN];
  BYTE ct[BLOCK_LEN];
  int i;
  int numBlocks = numOutputBytes / BLOCK_LEN;
  const int numSuppressedBytes = (ROUNDS - 1) * BLOCK_LEN;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes % BLOCK_LEN) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.key = (BYTE*)key;
  ctx.iv  = (BYTE*)iv;
  ctx.out = outBuf;

  PRESENT_init(&ctx);

  /* first ROUNDS blocks ((ROUNDS - 1) suppressed + first) */
  COPY_BUF_8(pt, iv)
  XOR_BUF_8(pt, inBuf)
  PRESENT_encryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
  if (numBlocks <= ROUNDS)
    return 0;
  ctx.out = ct;
  numBlocks -= ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, ct)
    XOR_BUF_8(pt, inBuf + (i + 1) * BLOCK_LEN)
    PRESENT_init(&ctx);
    PRESENT_encryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + (i + ROUNDS) * BLOCK_LEN, ct)
  }
  return 0;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
int blackBoxPRESENTEncryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
      return PRESENT_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return PRESENT_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxPRESENTInfo(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 10;
  if (ivSizeInBytes) *ivSizeInBytes = BLOCK_LEN;
  if (suppressedBytes) *suppressedBytes = (ROUNDS - 1) * BLOCK_LEN;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = BLOCK_LEN;
}

