/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_optimal_maxterm.h"
#include "black_box_utils.h"
#include "black_box_bit_set_utils.h"
#include "bitslice_utils.h"
#include "bittwiddling.h"
#include "memory_utils.h"
#include "assert_utils.h"
#ifndef _WIN32
#include <unistd.h> /* CLK_TCK */
#endif

#define MAX_BIT_LEN 1024

/*******************************************************************************
 * Utilities
 ******************************************************************************/
static int bitCountByBlocks(bbCipher cipher) {
  if (blackBoxCipherType(cipher) == kBlockCipher) {
    return 1;
  }
  return 0;
}

void setBufBits(BYTE *buf, int *bufBit, int numBufBits) {
  int i;
  for (i=0; i<numBufBits; i++)
    setBufBit(buf, bufBit[i], 1);
}

/*******************************************************************************
 * Optimal bit sets
 ******************************************************************************/
#define MAX_BIT_SET_SIZE 10
  typedef struct {
    const int numBits;
    const int bit[MAX_BIT_SET_SIZE];
  } singleBitSetType;

  typedef struct {
    const singleBitSetType key;
    const singleBitSetType iv;
  } doubleBitSetType;

  typedef struct {
    const singleBitSetType key;
    const singleBitSetType iv;
    const doubleBitSetType keyIv;
  } optimalSetType;

  static const optimalSetType optimalSet[kNumBlackBoxCiphers] = {
    {{6, {33,42,45,57,72,78}} /* key bit set */,
     {7, {3,21,27,48,54,63,72}} /* iv bit set */,
     { {3, {34,43,76}} /* key bit set */,
       {2, {31,64}} /* iv bit set */ } }, /* Trivium */
    {{5, {12,40,81,96,112}} /* key bit set */,
     {7, {0,2,7,15,16,28,54}} /* iv bit set */,
     { {3, {19,77,113}} /* key bit set */,
       {2, {13,40}} /* iv bit set */ } }, /* Rabbit */
    {{5, {12,27,37,61,68}} /* key bit set */,
     {6, {9,15,21,28,34,36}} /* iv bit set */,
     { {4, {2,12,19,29}} /* key bit set */,
       {0, {0}} /* iv bit set */ } }, /* Edon80 */
    {{5, {19,24,38,70,80}} /* key bit set */,
     {5, {9,39,89,97,115}} /* iv bit set */,
     { {0, {0}} /* key bit set */,
       {4, {38,78,81,118}} /* iv bit set */ } }, /* AES-128 */
    {{4, {158,166,206,219}} /* key bit set */,
     {5, {59,67,83,101,107}} /* iv bit set */,
     { {2, {66,98}} /* key bit set */,
       {2, {66,98}} /* iv bit set */ } }, /* AES-256 */
    {{7, {14,19,25,34,42,44,49}} /* key bit set */,
     {6, {11,24,44,46,50,54}} /* iv bit set */,
     { {2, {42,47}} /* key bit set */,
       {3, {27,42,46}} /* iv bit set */ } }, /* DES */
    {{6, {14,113,118,120,123,125}} /* key bit set */,
     {6, {34,59,63,64,67,69}} /* iv bit set */,
     { {0, {0}} /* key bit set */,
       {5, {63,64,67,68,69}} /* iv bit set */ } }, /* Grain-128 */
    {{7, {1,27,31,33,58,71,77}} /* key bit set */,
     {7, {1,22,26,37,45,47,55}} /* iv bit set */,
     { {2, {24,33}} /* key bit set */,
       {3, {16,20,23}} /* iv bit set */ } }, /* Grain v1 */
    {{2, {7,39}} /* key bit set */,
     {7, {5,7,15,21,22,25,49}} /* iv bit set */,
     { {2, {71,103}} /* key bit set */,
       {0, {0}} /* iv bit set */ } }, /* TEA */
    {{6, {7,33,53,71,83,90}} /* key bit set */,
     {7, {7,25,28,37,44,46,51}} /* iv bit set */,
     { {3, {7,34,61}} /* key bit set */,
       {2, {7,38}} /* iv bit set */ } }, /* XTEA */
    {{5, {7,19,24,71,93}} /* key bit set */,
     {5, {10,37,38,59,113}} /* iv bit set */,
     { {3, {0,16,56}} /* key bit set */,
       {2, {31,54}} /* iv bit set */ } }, /* SEED */
    {{6, {14,48,49,68,71,77}} /* key bit set */,
     {7, {10,24,25,26,27,28,60}} /* iv bit set */,
     { {1, {62}} /* key bit set */,
       {4, {60,61,62,63}} /* iv bit set */ } }, /* PRESENT */
    {{6, {50,67,76,86,108,111}} /* key bit set */,
     {6, {13,17,45,53,77,92}} /* iv bit set */,
     { {2, {19,83}} /* key bit set */,
       {3, {7,39,71}} /* iv bit set */ } }, /* SMS4 */
    {{6, {29,36,66,67,95,125}} /* key bit set */,
     {6, {73,79,81,89,119,123}} /* iv bit set */,
     { {2, {3,121}} /* key bit set */,
       {3, {66,76,82}} /* iv bit set */ } }, /* Camellia */
    {{5, {45,56,78,86,88}} /* key bit set */,
     {7, {4,12,22,28,31,50,63}} /* iv bit set */,
     { {0, {0}} /* key bit set */,
       {5, {0,4,28,30,50}} /* iv bit set */ } }, /* RC5 */
    {{5, {51,103,107,116,121}} /* key bit set */,
     {5, {21,66,67,89,113}} /* iv bit set */,
     { {0, {0}} /* key bit set */,
       {4, {0,18,67,73}} /* iv bit set */ } }, /* RC6 */
    {{5, {44,53,63,85,89}} /* key bit set */,
     {7, {7,20,32,33,39,51,54}} /* iv bit set */,
     { {2, {103,111}} /* key bit set */,
       {3, {7,31,55}} /* iv bit set */ } }, /* HIGHT */
#if 0
    {{5, {15,30,35,59,104}} /* key bit set */,
     {5, {36,63,94,106,125}} /* iv bit set */,
     { {0, {0}} /* key bit set */,
       {4, {53,56,102,104}} /* iv bit set */ } }, /* Clefia */
#endif
    {{5, {3,12,91,113,120}} /* key bit set */,
     {5, {14,15,53,74,86}} /* iv bit set */,
     { {2, {37,110}} /* key bit set */,
       {2, {100,116}} /* iv bit set */ } }, /* HC-128 */
    {{4, {92,139,207,212}} /* key bit set */,
     {4, {51,161,216,227}} /* iv bit set */,
     { {1, {60}} /* key bit set */,
       {2, {43,84}} /* iv bit set */ } },  /* HC-256 */
    {{3, {44,56,79}} /* key bit set */,
     {3, {8,36,68}} /* iv bit set */,
     { {2, {6,34}} /* key bit set */,
       {1, {64}} /* iv bit set */ } },  /* MICKEY v2 */
    {{4, {77,84,89,95}} /* key bit set */,
     {5, {2,11,36,45,56}} /* iv bit set */,
     { {1, {48}} /* key bit set */,
       {3, {9,32,53}} /* iv bit set */ } },  /* Salsa20/12 */
    {{3, {35,73,81}} /* key bit set */,
     {3, {4,56,113}} /* iv bit set */,
     { {1, {19}} /* key bit set */,
       {2, {39,40}} /* iv bit set */ } }  /* SOSEMANUK */
  };

int optimalBitSetStored(bbCipher cipher, int keyBitsAllowed, int ivBitsAllowed) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Unknown cipher!\n");
  ASSERT(keyBitsAllowed || ivBitsAllowed, "Undefined bit set!\n");
  if (keyBitsAllowed && ivBitsAllowed) return optimalSet[cipher].keyIv.key.numBits + optimalSet[cipher].keyIv.iv.numBits;
  if (keyBitsAllowed) return optimalSet[cipher].key.numBits;
  if (ivBitsAllowed) return optimalSet[cipher].iv.numBits;
  return 0;
}

void getOptimalBitSet(bbCipher cipher, int *numKeyBits, int *keyBit, int *numIvBits, int *ivBit) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Unknown cipher!\n");
  ASSERT(keyBit || ivBit, "Undefined bit set!\n");
  if (keyBit && ivBit) {
    *numKeyBits = optimalSet[cipher].keyIv.key.numBits;
    MEMCPY(keyBit, optimalSet[cipher].keyIv.key.bit, *numKeyBits * sizeof(int));
    *numIvBits = optimalSet[cipher].keyIv.iv.numBits;
    MEMCPY(ivBit, optimalSet[cipher].keyIv.iv.bit, *numIvBits * sizeof(int));
    return;
  }
  if (keyBit) {
    *numKeyBits = optimalSet[cipher].key.numBits;
    MEMCPY(keyBit, optimalSet[cipher].key.bit, *numKeyBits * sizeof(int));
    return;
  }
  if (ivBit) {
    *numIvBits = optimalSet[cipher].iv.numBits;
    MEMCPY(ivBit, optimalSet[cipher].iv.bit, *numIvBits * sizeof(int));
    return;
  }
  ASSERT_ALWAYS("Parameter error!\n");
}

/*******************************************************************************
 * Optimal Maxtest Distinguisher
 ******************************************************************************/
static void printBestSet(FILE *logFile, int flags, int numKeyBits, int *keyBit, int numIvBits, int*ivBit, int z, int suppressedBytes, time_t start) {
  printf("[");
  logDateRaw(NULL, LOGSCREEN);
  printf(", ");
  logTimeRaw(NULL, LOGSCREEN, start);
  printf("], %d / %d zero bits", z, suppressedBytes * 8);
  if (keyBit) { printf(", Key bit set "); logBitSet(logFile, flags, keyBit, numKeyBits); }
  if (ivBit) { printf(", IV bit set "); logBitSet(logFile, flags, ivBit, numIvBits); }
  printf("\n");
}

void blackBoxOptimalMaxtermDistinguisher(bbCipher cipher, int numBits, int keyFill, int *numKeyBits, int *keyBit, int ivFill, int *numIvBits, int *ivBit, int output, int numParallellBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE key[MAX_BIT_LEN], iv[MAX_BIT_LEN];
  BYTE *in;
  BYTE *xor;
  int bestNumKeyBits, bestNumIvBits;
  int bestKeyBit[MAX_BIT_LEN], bestIvBit[MAX_BIT_LEN];
  int z, bestZ = -1;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  time_t start = time(NULL);

  if (output) {
    stars(nameLen + 45);
    printf("\n* Optimal %s Maxtest Distinguisher of Size %2d *\n", name, numBits);
    stars(nameLen + 45);
    printf("\n\n");
    if (blackBoxCipherType(cipher) == kBlockCipher)
      printf("The number of initial zero bits are counted %s\n\n", bitCountByBlocks(cipher) ? "blockwise" : "sequentially");
  }

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  ASSERT(ivSize <= MAX_BIT_LEN, "Insuffiecient iv container length!");
  ASSERT(keySize <= MAX_BIT_LEN, "Insuffiecient key container length!");

  /* initialize */
  in = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  xor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  bufFill(key, keyFill, keySize);
  bufFill(iv, ivFill, ivSize);
  MEMSET(in, 0, suppressedBytes);
  initializeConnectedBitSets(numBits, keyBit, numKeyBits, keySize*8, ivBit, numIvBits, ivSize*8);

  /* print key and iv */
  if (output) {
    printf("Key = "); logBuf(NULL, LOGSCREEN, key, keySize, 0, 0); printf("\n");
    printf("IV  = "); logBuf(NULL, LOGSCREEN, iv, ivSize, 0, 0); printf("\n\n");
  }

  do {

    /* compute xor */
    MEMSET(xor, 0, suppressedBytes);
    if (xorOverBitSet(cipher, key, *numKeyBits, keyBit, iv, *numIvBits, ivBit, in, implicitBlockSize, xor, numParallellBits)) { /* calculate xor of the initial bit set */
      printf("Error when calculating xor of initial bit set\n");
      FREE(in);
      FREE(xor);
      return;
    }

    /* output current bit set when user presses any key */
    if (output && kbhit()) {
      getch();
      printf("[");
      logDateRaw(NULL, LOGSCREEN);
      printf(", ");
      logTimeRaw(NULL, LOGSCREEN, start);
      printf("], Currently at Key bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, keyBit, *numKeyBits);
      printf(" and IV bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, ivBit, *numIvBits);
      printf("\n");
    }

    z = bitCountByBlocks(cipher) ?
        numInitialZeroBitsByBlocks(xor, suppressedBytes, implicitBlockSize) :
        numInitialZeroBits(xor, suppressedBytes);
    if (z > bestZ) { /* store current best */
      if (output)
        printBestSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, *numKeyBits, keyBit, *numIvBits, ivBit, z, suppressedBytes, start);
      bestZ = z;
      copyBitSet(bestKeyBit, &bestNumKeyBits, keyBit, *numKeyBits);
      copyBitSet(bestIvBit, &bestNumIvBits, ivBit, *numIvBits);
    }

  } while (nextConnectedBitSet(keyBit, numKeyBits, keySize*8, ivBit, numIvBits, ivSize*8));
  FREE(in);
  FREE(xor);

  /* restore best bit set */
  if (keyBit) copyBitSet(keyBit, numKeyBits, bestKeyBit, bestNumKeyBits);
  if (ivBit) copyBitSet(ivBit, numIvBits, bestIvBit, bestNumIvBits);

  if (output) {
    printf("\n[");
    logDateRaw(NULL, LOGSCREEN | LOGFILE | LOGFLUSH);
    printf(", ");
    logTimeRaw(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, start);
    if (blackBoxCipherType(cipher) == kBlockCipher) {
      printf("] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f blocks = %5.2f%%\n", *numKeyBits + *numIvBits, bestZ, suppressedBytes * 8, bestZ/(float)8/(float)implicitBlockSize, 100*bestZ/(float)(suppressedBytes*8));
    } else {
      printf("] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f%%\n", *numKeyBits + *numIvBits, bestZ, suppressedBytes * 8, 100*bestZ/(float)8/(float)suppressedBytes);
    }
  }
}

void printTime(double t) {
  int d, h, m;
  double s;
  double time;

#ifdef _WIN32
  time = t / CLK_TCK;
#else
  time = t / sysconf(_SC_CLK_TCK);
#endif
  d = (int)time / 3600 / 24;
  h = (time - d*3600*24) / 3600;
  m = (time - d*3600*24 - h*3600) / 60;
  s = time - d*3600*24 - h*3600 - m*60;
  printf("%d d %d h %d min %.2f sec", d, h, m, s);
}

void nextEstimate(double *next, double now, int n, int i) {
  *next = now * (n + i) / (i + 1) * 2;
}

void blackBoxOptimalMaxtermDistinguisherSurvey_ex(bbCipher cipher, int numBits, int okToUseKeyBits, int okToUseIvBits, int numParallellBits) {
  int _keyBit[80], _ivBit[80];
  int *keyBit = okToUseKeyBits ? _keyBit : NULL;
  int *ivBit = okToUseIvBits ? _ivBit : NULL;
  int numKeyBits, numIvBits;
  int n = 0;
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  time_t start = time(NULL);
  double time;
  clock_t t = clock();

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  if (keyBit) n += keySize * 8;
  if (ivBit) n += ivSize * 8;

  blackBoxOptimalMaxtermDistinguisher(cipher, numBits, FILL_ZEROS, &numKeyBits, keyBit, FILL_ZEROS, &numIvBits, ivBit, 0 /* output */, numParallellBits);

  time = clock() - t;
  printf("[");
  logDateRaw(NULL, LOGSCREEN);
  printf(", ");
  logTimeRaw(NULL, LOGSCREEN, start);
  printf("]\nTime: ");
  printTime(time);
  printf("\nOptimal %d-distinguisher: ", numBits);
  if (keyBit) { printf(" %d key bits ", numKeyBits); logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, keyBit, numKeyBits); }
  if (ivBit) { printf(" %d iv bits ", numIvBits); logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, ivBit, numIvBits); }

  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 1); nextEstimate(&time, time, n, numBits); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 2); nextEstimate(&time, time, n, numBits + 1); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 3); nextEstimate(&time, time, n, numBits + 2); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 4); nextEstimate(&time, time, n, numBits + 3); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 5); nextEstimate(&time, time, n, numBits + 4); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 6); nextEstimate(&time, time, n, numBits + 5); printTime(time);
  printf("\nEstimated computation time for optimal %d-distinguisher: ", numBits + 7); nextEstimate(&time, time, n, numBits + 6); printTime(time);
  printf("\n\n");
}

void blackBoxOptimalMaxtermDistinguisherSurvey(bbCipher cipher, int numBits, int numParallellBits) {
  const char *name = blackBoxCipherName(cipher);
  printf("\n\nOptimal %s Distinguishers\n\n", name);
  printf("Key Bits only\n");   blackBoxOptimalMaxtermDistinguisherSurvey_ex(cipher, numBits, 1, 0, numParallellBits);
  printf("IV Bits only\n");    blackBoxOptimalMaxtermDistinguisherSurvey_ex(cipher, numBits, 0, 1, numParallellBits);
  printf("Key and IV Bits\n"); blackBoxOptimalMaxtermDistinguisherSurvey_ex(cipher, numBits, 1, 1, numParallellBits);
}

void blackBoxOptimalMaxtermDistinguisherSurveyAll(int numBits, int numParallellBits) {
  int i;
  starHeader("Black Box - Optimal Maxterm Distinguisher Survey");
  for (i=0; i<kNumBlackBoxCiphers; i++)
    blackBoxOptimalMaxtermDistinguisherSurvey((bbCipher)i, numBits, numParallellBits);
}

/*******************************************************************************
 * Optimal bit flip set
 ******************************************************************************/
void blackBoxOptimalBitFlipSet(bbCipher cipher, int numBits, int okToUseKeyBits, int okToUseIvBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE key[MAX_BIT_LEN], iv[MAX_BIT_LEN];
  BYTE *in, *xor0, *xor;
  int z, bestZ = -1;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  time_t start = time(NULL);
  int _keyBit[MAX_BIT_LEN];
  int _ivBit[MAX_BIT_LEN];
  int *keyBit = okToUseKeyBits ? _keyBit : NULL;
  int *ivBit = okToUseIvBits ? _ivBit : NULL;
  int numKeyBits, numIvBits;

  ASSERT(okToUseKeyBits || okToUseIvBits, "Key or IV bits must be specified!");

  stars(nameLen + 36);
  printf("\n* Optimal %s Bit Flip Set of Size %2d *\n", name, numBits);
  stars(nameLen + 36);
  printf("\n\n");
  if (blackBoxCipherType(cipher) == kBlockCipher)
    printf("The number of initial zero bits are counted %s\n\n", bitCountByBlocks(cipher) ? "blockwise" : "sequentially");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  ASSERT(ivSize <= MAX_BIT_LEN, "Insuffiecient iv container length!");
  ASSERT(keySize <= MAX_BIT_LEN, "Insuffiecient key container length!");

  /* initialize */
  in = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  xor0 = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  xor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  MEMSET(key, 0, keySize);
  MEMSET(iv, 0, ivSize);
  MEMSET(in, 0, suppressedBytes);
  MEMSET(xor0, 0, suppressedBytes);
  initializeConnectedBitSets(numBits, keyBit, &numKeyBits, keySize*8, ivBit, &numIvBits, ivSize*8);

  /* calculate xor for empty bit set */
  if (blackBoxEncrypt(cipher, key, iv, in, suppressedBytes, xor0, suppressedBytes, 1 /* with init round output */)) {
    printf("Error when calculating xor of initial bit set\n");
    FREE(in);
    FREE(xor0);
    FREE(xor);
    return; /* cryption error */
  }

  /* iterate through all bit sets of the given size */
  do {

    /* reset xor buffer */
    MEMSET(xor, 0, suppressedBytes);

    /* set bit set */
    MEMSET(key, 0, keySize); setBufBits(key, keyBit, numKeyBits);
    MEMSET(iv, 0, ivSize); setBufBits(iv, ivBit, numIvBits);

    /* output current bit set when user presses any key */
    if (kbhit()) {
      getch();
      printf("[");
      logDateRaw(NULL, LOGSCREEN);
      printf(", ");
      logTimeRaw(NULL, LOGSCREEN, start);
      printf("], Currently at Key bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, keyBit, numKeyBits);
      printf(" and IV bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, ivBit, numIvBits);
      printf("\n");
    }

    if (blackBoxEncrypt(cipher, key, iv, in, suppressedBytes, xor, suppressedBytes, 1 /* with init round output */)) {
      printf("Error when calculating xor of Key bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, keyBit, numKeyBits);
      printf(" and IV bit set ");
      logBitSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, ivBit, numIvBits);
      printf("\n");
      FREE(in);
      FREE(xor0);
      FREE(xor);
      return; /* cryption error */
    }

    /* add initial xor */
    MEMXOR(xor, xor0, suppressedBytes);

    z = bitCountByBlocks(cipher) ?
        numInitialZeroBitsByBlocks(xor, suppressedBytes, implicitBlockSize) :
        numInitialZeroBits(xor, suppressedBytes);
    if (z > bestZ) { /* store current best */
      printBestSet(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, numKeyBits, keyBit, numIvBits, ivBit, z, suppressedBytes, start);
      bestZ = z;
    }

  } while (nextConnectedBitSet(keyBit, &numKeyBits, keySize*8, ivBit, &numIvBits, ivSize*8));
  FREE(in);
  FREE(xor0);
  FREE(xor);

  printf("\n[");
  logDateRaw(NULL, LOGSCREEN | LOGFILE | LOGFLUSH);
  printf(", ");
  logTimeRaw(NULL, LOGSCREEN | LOGFILE | LOGFLUSH, start);
  if (blackBoxCipherType(cipher) == kBlockCipher) {
    printf("] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f blocks = %5.2f%% ", numKeyBits + numIvBits, bestZ, suppressedBytes * 8, bestZ/(float)8/(float)implicitBlockSize, 100*bestZ/(float)(suppressedBytes*8));
  } else {
    printf("] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f%% ", numKeyBits + numIvBits, bestZ, suppressedBytes * 8, 100*bestZ/(float)8/(float)suppressedBytes);
  }
  printf("\n\n\n");
}

void blackBoxOptimalBitFlipSetAll(int numBits, int okToUseKeyBits, int okToUseIvBits) {
  int i;
  starHeader("Black Box - Optimal bit flip set");
  for (i=0; i<kNumBlackBoxCiphers; i++)
    blackBoxOptimalBitFlipSet((bbCipher)i, numBits, okToUseKeyBits, okToUseIvBits);
}
























/* Maxtest summation utilizing the complementary property of DES */
#if 0
void blackBoxGreedySkewDESDistinguisher(void) {
  const bbCipher cipher = kDES;
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE key[MAX_BIT_LEN], iv[MAX_BIT_LEN];
  int numKeyBits = 0;
  int keyBit[MAX_BIT_LEN];
  int numIvBits = 11;
  int ivBit[17] = { 0, 18, 52, 28, 44, 40,  6, 53, 59, 49, 29, 14, 22, 13, 33, 27, 34 };
#if 0
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  0  ->   45 / 960 zero bits =  0.70 blocks =  4.69% with IV bit set { }  0404015555015455 4D5947BFBA56EDEE CEF78B7E70BDDE99 99EF53E8E12EFD26 76CEA2D4960DAF49 FC8940FD685A5BD6 FC1691FFD1A0B3B8 AC3C22BAF7113361 18280474BB2223C2 740049A937411790 FC5193136FD37B74 BCF76622CFB2B2FC 3CEF9851DA2471AC 6C8B74B7A00CA21C CC53AC7E40581179
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  1  ->   63 / 960 zero bits =  0.98 blocks =  6.56% with IV bit set {  0 }  0100000000000000 0211000000010004 0062144000431108 14956D8111933354 693BDE46732236B9 8663F898A7102C73 4C82F1654B3159A2 C850A78BD762B344 D4B50E53AE9423CC FD3A08F25D281388 BB2004B5BA057251 73154D6B315BF1B7 A36F9A9726E7A33F 53DE716F08DF177B E2BCB79B10BE2AE3
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  2  ->  250 / 960 zero bits =  3.91 blocks = 26.04% with IV bit set {  0 18 }  0000000000000000 0000000000000000 0000000000000000 0140040044000100 57D41C5188454611 EEBD2CA2119ED836 983E0C446328B03D 24694D88D745642B 0C839E10AB8AC806 19422D654300900D 37D41E9FD341751A 3FED686FE382EE61 7BDED4CE9710CC92 E6E9ED983A70DD34 D8C2DB6021A4AA29
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  3  ->  253 / 960 zero bits =  3.95 blocks = 26.35% with IV bit set {  0 18 52 }  0000000000000000 0000000000000000 0000000000000000 0000000000101040 4104041541752080 921C1D6ED6FA4414 716C7AC9BCE4D93C A6CDE0963DC9F26C 199A946C2A87E1CD 672469DC015EC38F 9F0DD7ED56E8D21E 2B0FFA9FE9C0E139 071AF13B97918226 4E30A7322B364058 89241F70477991A0
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  4  ->  252 / 960 zero bits =  3.94 blocks = 26.25% with IV bit set {  0 18 52 28 }  0000000000000000 0000000000000000 0000000000000000 0000004540000000 5150519BD0555040 A2E0F732E5FFF595 40C1FA348FEBAA6B C5D2F0390A9311D7 DFF4B572117377AE EBEC3FE567B2BF0C 96D87A8B9B303A49 39E1F15662716193 72C3E3ED81F29276 A5C282CB57F065BC 0E905083FEA0CF68
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  5  ->  252 / 960 zero bits =  3.94 blocks = 26.25% with IV bit set {  0 18 52 28 44 }  0000000000000000 0000000000000000 0000000000000000 0010000100110000 1471554601725001 7CE2AB9816F0A513 AC95077468B01F77 082A5AE881316BBE 1000B4D156279779 305469F3E90A6AE7 65EDD7E7C214D48B CF8EFA9BD138BC03 9E49E523B3642853 78839B4632CC41A2 B01262DD75888710
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  6  ->  254 / 960 zero bits =  3.97 blocks = 26.46% with IV bit set {  0 18 52 28 44 40 }  0000000000000000 0000000000000000 0000000000000000 0000000000010001 0050454405474153 05F4CBD84ECFD3F2 1FB8C6B5D89EE2B0 2B65DC7EB169C520 578FFDF837D2DA50 FE0FBBF52BB1A1F5 BD1E77FB072242EA 2B2CAAE34F4181D5 06481483CE8217EA 4DC568139D003ED4 DA8B80662E4139E9
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  7  ->  250 / 960 zero bits =  3.91 blocks = 26.04% with IV bit set {  0 18 52 28 44 40  6 }  0000000000000000 0000000000000000 0000000000000000 4000040000014500 D5515C1151438F00 BFF7F876E6830B50 7AFBF5BDD81243B4 A1A3EE2EB434923C 12169C1D796D2428 303D3C7AB3DA0954 602B39E437E456B8 D15676893A9DBC21 E3F9ED53342A3C13 C6A6DFF229456C23 DD5DFBB5068BD907
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  8  ->  247 / 960 zero bits =  3.86 blocks = 25.73% with IV bit set {  0 18 52 28 44 40  6 53 }  0000000000000000 0000000000000000 0000000000000000 0044400001010541 058D811547425F96 1F1B027B8A95EF6C 2A6255A3106ADB8C 14D5FF162095B258 29EFAB68142B64B5 02DF56D57D12986B 55BEACEAAF7124D7 AF6D09944AE259AF 0FDA062C80C0B75E 5FA5495950956EB9 FF4A97F3A53F9D66
[Wed 10:22:45, 0 days  0 h  0 min  0 sec] Bit set size  9  ->  248 / 960 zero bits =  3.88 blocks = 25.83% with IV bit set {  0 18 52 28 44 40  6 53 59 }  0000000000000000 0000000000000000 0000000000000000 0151000000005410 17B315154440B970 6E626F2EDC8172A0 CCC19A4CAD02B450 8C8631994E407DE0 1C1C3726DC85FAC5 3D683B0DE91AF5CF 7BC0761E8660FF8B F2C1FD3D4C84BB57 A496BF3B891C72EB 49382B360639E0C3 D23547381C23C483
[Wed 10:22:46, 0 days  0 h  0 min  1 sec] Bit set size 10  ->  245 / 960 zero bits =  3.83 blocks = 25.52% with IV bit set {  0 18 52 28 44 40  6 53 59 49 }  0000000000000000 0000000000000000 0000000000000000 1000005105550040 615511F24BFE1181 96FF77A183E86316 29FABA4316D0D779 07B175C37DE5FEB2 5E33EED7AFCEE835 EC728CEE1BC9D53F 99A109982287EA7B 26561325041F91A2 48E9220A082A2655 80D74105440048EF 54EE874EC800D4CA
[Wed 10:22:46, 0 days  0 h  0 min  1 sec] Bit set size 11  ->  244 / 960 zero bits =  3.81 blocks = 25.42% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 }  0000000000000000 0000000000000000 0000000000000000 0044010445150404 14894658CB3A0D5C 2D02C8B1C2311BAC 5A04C1328062374D F459D63151952E9A FCF2FC67A73E5C70 E9B5FDDB1A68BCB4 963ABBA22491686D 296063114C27D4CE 06D5C7239D0AFC9C 19EF8F072E44FC28 628F5E1E5CD8B950
[Wed 10:22:47, 0 days  0 h  0 min  2 sec] Bit set size 12  ->  245 / 960 zero bits =  3.83 blocks = 25.52% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 }  0000000000000000 0000000000000000 0000000000000000 1410000405141014 6D2441190E79603C 8A09827618E7912D 115705EC759A220F 62AB1A8CAF31051B 8146700D0A235B63 52DCE41B5102B686 E0ED9833E3052C4D 858F7127864B4D8B 1E4AA31B5DD3CE16 29854232EFA2887D 461AC1749B5115EA
[Wed 10:22:49, 0 days  0 h  0 min  4 sec] Bit set size 13  ->  248 / 960 zero bits =  3.88 blocks = 25.83% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 22 }  0000000000000000 0000000000000000 0000000000000000 1000400504004440 6014D51B4854CC95 9538BB6284F98D7F 2E2432945CB64BEE 49182568E97DC6CD D2604A8597EEC8DB E5D0D00A2F9D85E2 CEA4F1141F2B4F81 CD58F66D7B46DF16 DBB1BCCEB398AF3C E7276DD923211A6C CA4BDAE6434325CC
[Wed 10:22:52, 0 days  0 h  0 min  7 sec] Bit set size 14  ->  250 / 960 zero bits =  3.91 blocks = 26.04% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 22 13 }  0000000000000000 0000000000000000 0000000000000000 1001001400004400 2012412805019D44 452492150B176E9C 8E0D752B423A9C3D 0D0BFF0291356D7B 0A42FE55232FCEB6 0185A8AF130BD96D 161E504A7747A2CB 7C28E5C1AA9A51C7 BD10CA971575B6CB 2B25C52E2FEA38C6 470F8B494E9135C8
[Wed 10:22:59, 0 days  0 h  0 min 14 sec] Bit set size 15  ->  249 / 960 zero bits =  3.89 blocks = 25.94% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 22 13 33 }  0000000000000000 0000000000000000 0000000000000000 0010000441104100 0071415D8234C315 40B6D2FB417D876E 802CA4E2D7AB5B88 000819C1BB56A351 5451238637F953E2 B9E2065D6FB2E295 279419BEDE64917B 4A7C373CADD837A3 C0EC6E6C0BB07F57 C1C989CC1630AAEB 86C6529C28655482
[Wed 10:23:13, 0 days  0 h  0 min 28 sec] Bit set size 16  ->  249 / 960 zero bits =  3.89 blocks = 25.94% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 22 13 33 27 }  0000000000000000 0000000000000000 0000000000000000 0001000140140044 055350539538159D 1AB7A5B33F346A6F 603F0A763E7DC0DA C57B54A97CEFC5B0 9EA3F956AD9F9F21 2C17A3AC1B3B2A42 096B025866761195 028254A488FC636B 1440FD0900FC96C3 3DD5EE5650F93882 7FAFDDACB5F23510
[Wed 10:23:39, 0 days  0 h  0 min 54 sec] Bit set size 17  ->  250 / 960 zero bits =  3.91 blocks = 26.04% with IV bit set {  0 18 52 28 44 40  6 53 59 49 29 14 22 13 33 27 34 }  0000000000000000 0000000000000000 0000000000000000 0405000000150000 4C1F4010112A4111 9C7FC02577119736 2CAB954BFF267A3C 48466F87FB5CE128 95D9CF4EA7E9C314 6EB29ACD5A868739 DC3474DFE11D1F33 F97DF9FE927B7A22 A7BEE7A820A7B411 4B398A11005B6823 D627156700A2C047
#endif
  BYTE in[MAX_SUPPRESSED_BYTES];
  BYTE xor[MAX_SUPPRESSED_BYTES];
  const char *name = blackBoxCipherName(cipher);
  UINT64 i;
  time_t start = time(NULL);

  printf("Skew DES summation\n\n");
  if (blackBoxCipherType(cipher) == kBlockCipher)
    printf("The number of initial zero bits are counted %s\n\n", bitCountByBlocks(cipher) ? "blockwise" : "sequentially");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  ASSERT(ivSize <= MAX_BIT_LEN, "Insuffiecient iv container length!");
  ASSERT(keySize <= MAX_BIT_LEN, "Insuffiecient key container length!");
  ASSERT(suppressedBytes <= MAX_SUPPRESSED_BYTES, "Insuffiecient output container length!");

  /* initialize */
  MEMSET(key, 0, keySize);
  MEMSET(iv, 0, ivSize);
  MEMSET(in, 0, MAX_SUPPRESSED_BYTES);
  MEMSET(xor, 0, MAX_SUPPRESSED_BYTES);

  printf("*********\n");
  printf("* Start *\n");
  printf("*********\n");

  if (iterateOverBitSet(cipher, key, numKeyBits, keyBit, iv, numIvBits, ivBit, in, implicitBlockSize, xor)) { /* calculate xor of the initial bit set */
    printf("Error when calculating xor of initial bit set\n");
    return;
  }
  /* print bit set */
  printStats(cipher, numKeyBits, keyBit, numIvBits, ivBit, suppressedBytes, implicitBlockSize, xor, 1, start);

  printf("*******\n");
  printf("* Mid *\n");
  printf("*******\n");

  /* sum over a linear combination of all key bits -> reverse key */
  for (i=0; i<keySize; i++)
    key[(int)i] ^= 0xFF;

  /* complete summation */
  if (iterateOverBitSet(cipher, key, numKeyBits, keyBit, iv, numIvBits, ivBit, in, implicitBlockSize, xor)) { /* calculate xor of the initial bit set */
    printf("Error when calculating xor of initial bit set\n");
    return;
  }

  /* print bit set */
  printStats(cipher, numKeyBits, keyBit, numIvBits, ivBit, suppressedBytes, implicitBlockSize, xor, 1, start);

  printf("*******\n");
  printf("* End *\n");
  printf("*******\n");

  printf("\n\n\n");
}
#endif




