/*
 * Adapted MICKEY v2 reference code.
 *
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_mickey.h"

/******************************************************************************
 * MICKEY v2
 ******************************************************************************/
static const int numKeyBytes = 10;
static const int numIvBytes = 10;
static const int numSuppressedBytes = 13;
static const int implicitBlockBytes = 1;

typedef struct {
  UINT32 R[4]; /* The 100 bit linear register R is stored in 4 32 bit unsigned integers. The l.s.b. of R[0] is r_0, and bit 3 of R[3] is r_99 */
  UINT32 S[4]; /* The 100 bit non-linear register S is stored in 4 32 bit unsigned integers. The l.s.b. of S[0] is s_0, and bit 3 of S[3] is s_99 */
  BYTE key[10];
  int ivsize;
} MickeyV2_ctx;


/* Feedback mask associated with the register R */
static const UINT32 R_Mask[4] = { U32C(0x1279327b), U32C(0xb5546660), U32C(0xdf87818f), U32C(0x00000003) };

/* Input mask associated with register S */
static const UINT32 Comp0[4] = { U32C(0x6aa97a30), U32C(0x7942a809), U32C(0x057ebfea), U32C(0x00000006) };

/* Second input mask associated with register S */
static const UINT32 Comp1[4] = { U32C(0xdd629e9a), U32C(0xe3a21d63), U32C(0x91c23dd7), U32C(0x00000001) };

/* Feedback mask associated with the register S for clock control bit = 0 */
static const UINT32 S_Mask0[4] = { U32C(0x9ffa7faf), U32C(0xaf4a9381), U32C(0x9cec5802), U32C(0x00000001) };

/* Feedback mask associated with the register S for clock control bit = 1 */
static const UINT32 S_Mask1[4] = { U32C(0x4c8cb877), U32C(0x4911b063), U32C(0x40fbc52b), U32C(0x00000008) };


/* The following routine clocks register R in ctx with given input and control bits */
void CLOCK_R(
  MickeyV2_ctx* ctx,
  int input_bit,
  int control_bit)
{
    int Feedback_bit;
        /* r_99 ^ input bit */
    int Carry0, Carry1, Carry2;
        /* Respectively, carry from R[0] into R[1], carry from R[1] into R[2] and carry from R[2] into R[3] */

    /* Initialise the variables */
    Feedback_bit = ((ctx->R[3] >> 3) & 1) ^ input_bit;
    Carry0 = (ctx->R[0] >> 31) & 1;
    Carry1 = (ctx->R[1] >> 31) & 1;
    Carry2 = (ctx->R[2] >> 31) & 1;

    if (control_bit)
    {
        /* Shift and xor */
        ctx->R[0] ^= (ctx->R[0] << 1);
        ctx->R[1] ^= (ctx->R[1] << 1) ^ Carry0;
        ctx->R[2] ^= (ctx->R[2] << 1) ^ Carry1;
        ctx->R[3] ^= (ctx->R[3] << 1) ^ Carry2;
    }
    else
    {
        /* Shift only */
        ctx->R[0] = (ctx->R[0] << 1);
        ctx->R[1] = (ctx->R[1] << 1) ^ Carry0;
        ctx->R[2] = (ctx->R[2] << 1) ^ Carry1;
        ctx->R[3] = (ctx->R[3] << 1) ^ Carry2;
    }

    /* Implement feedback into the various register stages */
    if (Feedback_bit)
    {
        ctx->R[0] ^= R_Mask[0];
        ctx->R[1] ^= R_Mask[1];
        ctx->R[2] ^= R_Mask[2];
        ctx->R[3] ^= R_Mask[3];
    }
}

/* The following routine clocks register S in ctx with given input and control bits */
void CLOCK_S(
  MickeyV2_ctx* ctx, 
  int input_bit,
  int control_bit)
{
    int Feedback_bit;
        /* s_99 ^ input bit */
    int Carry0, Carry1, Carry2;
        /* Respectively, carry from S[0] into S[1], carry from S[1] into S[2] and carry from S[2] into S[3] */

    /* Compute the feedback and carry bits */
    Feedback_bit = ((ctx->S[3] >> 3) & 1) ^ input_bit;
    Carry0 = (ctx->S[0] >> 31) & 1;
    Carry1 = (ctx->S[1] >> 31) & 1;
    Carry2 = (ctx->S[2] >> 31) & 1;

    /* Derive "s hat" according to the MICKEY v 2 specification */
    ctx->S[0] = (ctx->S[0] << 1) ^ ((ctx->S[0] ^ Comp0[0]) & ((ctx->S[0] >> 1) ^ (ctx->S[1] << 31) ^ Comp1[0]) & 0xfffffffe);
    ctx->S[1] = (ctx->S[1] << 1) ^ ((ctx->S[1] ^ Comp0[1]) & ((ctx->S[1] >> 1) ^ (ctx->S[2] << 31) ^ Comp1[1])) ^ Carry0;
    ctx->S[2] = (ctx->S[2] << 1) ^ ((ctx->S[2] ^ Comp0[2]) & ((ctx->S[2] >> 1) ^ (ctx->S[3] << 31) ^ Comp1[2])) ^ Carry1;
    ctx->S[3] = (ctx->S[3] << 1) ^ ((ctx->S[3] ^ Comp0[3]) & ((ctx->S[3] >> 1) ^ Comp1[3]) & 0x7) ^ Carry2;

    /* Apply suitable feedback from s_99 */
    if (Feedback_bit)
    {
        if (control_bit)
        {
            ctx->S[0] ^= S_Mask1[0];
            ctx->S[1] ^= S_Mask1[1];
            ctx->S[2] ^= S_Mask1[2];
            ctx->S[3] ^= S_Mask1[3];
        }
        else
        {
            ctx->S[0] ^= S_Mask0[0];
            ctx->S[1] ^= S_Mask0[1];
            ctx->S[2] ^= S_Mask0[2];
            ctx->S[3] ^= S_Mask0[3];
        }
    }
}

/* The following routine implements a clock of the keystream generator.  The parameter mixing is set to 0
   or a non-zero value to determine whether mixing (from s_50) is not/is applied; the parameter input_bit
   is used to specify any input bit to the generator */
int CLOCK_KG (
  MickeyV2_ctx* ctx,
  int mixing,
  int input_bit)
{
    int Keystream_bit; /* Keystream bit to be returned (only valid if mixing = 0 and input_bit = 0 */
    int control_bit_r; /* The control bit for register R */
    int control_bit_s; /* The control bit for register S */

    Keystream_bit = (ctx->R[0] ^ ctx->S[0]) & 1;
    control_bit_r = ((ctx->S[1] >> 2) ^ (ctx->R[2] >> 3)) & 1;
    control_bit_s = ((ctx->R[1] >> 1) ^ (ctx->S[2] >> 3)) & 1;

    if (mixing)
        CLOCK_R (ctx, ((ctx->S[1] >> 18) & 1) ^ input_bit, control_bit_r);
    else
        CLOCK_R (ctx, input_bit, control_bit_r);
    CLOCK_S (ctx, input_bit, control_bit_s);

    return Keystream_bit;
}


/* Key setup: simply save the key in ctx for use during IV setup */
void MickeyV2_keysetup(
  MickeyV2_ctx* ctx,
  const BYTE* key,
  UINT32 keysize,                /* Key size in bits. */
  UINT32 ivsize)                 /* IV size in bits. */
{
    int i; /* Indexing variable */

    (void)keysize; /* unused */

    /* Store the key in the algorithm context */
    for (i = 0; i<10; i++)
        ctx->key[i] = key[i];

    /* Store the iv size in the context too */
    ctx->ivsize = ivsize;
}

/*
 * IV setup. After having called ECRYPT_keysetup(), the user is
 * allowed to call ECRYPT_ivsetup() different times in order to
 * encrypt/decrypt different messages with the same key but different
 * IV's.
 */

/* This routine implements key loading according to the MICKEY v1 specification */
void MickeyV2_ivsetup(
  MickeyV2_ctx* ctx,
  const BYTE* iv)
{
    int i; /* Counting/indexing variable */
    int iv_or_key_bit; /* Bit being loaded */

    /* Initialise R and S to all zeros */
    for (i=0; i<4; i++)
    {
        ctx->R[i] = 0;
        ctx->S[i] = 0;
    }

    /* Load in IV */
    for (i=0; i<ctx->ivsize; i++)
    {
        iv_or_key_bit = (iv[i/8] >> (7-(i%8))) & 1; /* Adopt usual, perverse, labelling order */
        CLOCK_KG (ctx, 1, iv_or_key_bit);
    }

    /* Load in K */
    for (i=0; i<80; i++)
    {
        iv_or_key_bit = (ctx->key[i/8] >> (7-(i%8))) & 1; /* Adopt usual, perverse, labelling order */
        CLOCK_KG (ctx, 1, iv_or_key_bit);
    }

    /* Preclock */
    for (i=0; i<100; i++)
        CLOCK_KG (ctx, 1, 0);
}

/* Stream cipher a block of data */
void MickeyV2_encrypt_bytes(
  MickeyV2_ctx* ctx,
  const BYTE* input,
  BYTE* output,
  UINT32 msglen)                 /* length in bytes */
{
    UINT32 i, j;
        /* Counting variables */

    for (i=0; i<msglen; i++)
    {
        output[i] = input[i];

        for (j=0; j<8; j++)
            output [i] ^= (BYTE)(CLOCK_KG (ctx, 0, 0) << (7-j));
    }
}

/* Generate keystream data */
void MickeyV2_keystream_bytes(
  MickeyV2_ctx* ctx,
  BYTE* keystream,
  UINT32 length)                 /* Length of keystream in bytes. */
{
    UINT32 i, j;
        /* Counting variables */

    for (i=0; i<length; i++)
    {
        keystream[i] = 0;

        for (j=0; j<8; j++)
            keystream[i] ^= (BYTE)(CLOCK_KG (ctx, 0, 0) << (7-j));
    }
}

void MickeyV2_ivsetup_withInitOutput(MickeyV2_ctx* ctx, const BYTE* iv, BYTE *out) {
    int i; /* Counting/indexing variable */
    int iv_or_key_bit; /* Bit being loaded */

    /* Initialise R and S to all zeros */
    for (i=0; i<4; i++)
    {
        ctx->R[i] = 0;
        ctx->S[i] = 0;
    }

    /* Load in IV */
    for (i=0; i<ctx->ivsize; i++)
    {
        iv_or_key_bit = (iv[i/8] >> (7-(i%8))) & 1; /* Adopt usual, perverse, labelling order */
        CLOCK_KG (ctx, 1, iv_or_key_bit);
    }

    /* Load in K */
    for (i=0; i<80; i++)
    {
        iv_or_key_bit = (ctx->key[i/8] >> (7-(i%8))) & 1; /* Adopt usual, perverse, labelling order */
        CLOCK_KG (ctx, 1, iv_or_key_bit);
    }

#if 0
    /* Preclock */
    for (i=0; i<100; i++)
        CLOCK_KG (ctx, 1, 0);
#endif

    /* hidden output (100 bits, 12,5 bytes) */
    out[0] ^= (BYTE)(CLOCK_KG(ctx, 1, 0) << 3);
    out[0] ^= (BYTE)(CLOCK_KG(ctx, 1, 0) << 2);
    out[0] ^= (BYTE)(CLOCK_KG(ctx, 1, 0) << 1);
    out[0] ^= (BYTE)CLOCK_KG(ctx, 1, 0);
//    for (i=0; i<96; i++) out[1 + (i / 8)] ^= (BYTE)(CLOCK_KG(ctx, 1, 0) << (7-(i%8)));
#define INIT_BYTE(n) \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 7); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 6); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 5); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 4); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 3); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 2); \
    out[n] ^= (BYTE)(CLOCK_KG (ctx, 1, 0) << 1); \
    out[n] ^= (BYTE)CLOCK_KG (ctx, 1, 0);
  INIT_BYTE(1)  INIT_BYTE(2)  INIT_BYTE(3)  INIT_BYTE(4)
  INIT_BYTE(5)  INIT_BYTE(6)  INIT_BYTE(7)  INIT_BYTE(8)
  INIT_BYTE(9)  INIT_BYTE(10) INIT_BYTE(11) INIT_BYTE(12)
}

/******************************************************************************
 * Black box variants
 ******************************************************************************/
int mickey_v2_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  MickeyV2_ctx ctx;
  if (numInputBytes < numOutputBytes) return -1;
  MickeyV2_keysetup(&ctx, key, numKeyBytes * 8, numIvBytes * 8);
  MickeyV2_ivsetup(&ctx, iv);
  MickeyV2_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

int mickey_v2_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  MickeyV2_ctx ctx;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  MickeyV2_keysetup(&ctx, key, numKeyBytes * 8, numIvBytes * 8);
  MickeyV2_ivsetup_withInitOutput(&ctx, iv, outBuf); outBuf += numSuppressedBytes; numOutputBytes -= numSuppressedBytes;
  MickeyV2_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
int blackBoxMickeyV2Encryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
    return mickey_v2_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return mickey_v2_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxMickeyV2Info(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = numKeyBytes;
  if (ivSizeInBytes) *ivSizeInBytes = numIvBytes;
  if (suppressedBytes) *suppressedBytes = numSuppressedBytes;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = implicitBlockBytes;
}

