/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_latex_graph_writer.h"
#include "black_box_ciphers.h"
#include "log_utils.h"
#include "assert_utils.h"
#include <string.h>

#define MIN_NUM_X_TICKS 5
#define MIN_NUM_Y_TICKS 4
#define ZOFF 1.4 /* x offset for zoomed graph */

static int numTicks(int max, int minNumTicks) {

  static const int l[] = { 1, 2, 5, 10, 20, 25, 50, 100, 200, 250, 500, 1000, 2000, 2500, 5000, 10000, 20000, 25000, 50000, 100000 };
  static const int n = sizeof(l)/sizeof(int);
  int i;

  for (i=0; i<n; i++)
    if (max / l[n - i - 1] >= minNumTicks)
      return l[n - i - 1];
  return 1;
}


void writeLatexMaxtermPlot(FILE *logFile, bbCipher cipher, int numMaxtermCoords, MaxtermCoordinate *mc, int bitspaceSize) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  int i, maxx = -1, maxy = -1;
  int dx, dy;
  int closeX, closeY;
  int numLabelsX, numLabelsY;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  for (i=0; i<numMaxtermCoords; i++) {
    if (mc[i].bit > maxx) maxx = mc[i].bit;
    if (mc[i].numZeroRounds > maxy) maxy = mc[i].numZeroRounds;
  }
  dx = numTicks(maxx, MIN_NUM_X_TICKS);
  dy = numTicks(maxy, MIN_NUM_Y_TICKS);
  closeX = (maxx - maxx / dx * dx)/(float)maxx < 0.07;
  closeY = (maxy - maxy / dy * dy)/(float)maxy < 0.04;
  numLabelsX = closeX ? (maxx / dx - 1) : (maxx / dx);
  numLabelsY = closeY ? (maxy / dy - 1) : (maxy / dy);
  ASSERT(maxx > 0, "Unexpected x-coordinate values!");
  ASSERT(maxy > 0, "Unexpected y-coordinate values!");

/* Lunarc seems to have problems with logging to file.
 * After some time logging to file may simply stop.
 * The define below makes it better, it seems, but we
 * have no plausible explanation for this, unfortunately,
 * except possibly for excessive flushing. */
#define logger fprintf
  logger(logFile, "%% const MaxtermCoordinate mc[] = {{%d,%d}", mc[0].bit, mc[0].numZeroRounds);
  for (i=1; i<numMaxtermCoords; i++)
    logger(logFile, ",{%d,%d}", mc[i].bit, mc[i].numZeroRounds);
  logger(logFile, "};\n");
  logger(logFile, "\\begin{scope}[thick]\n");
  logger(logFile, "\n");
  logger(logFile, "%% Main plot\n");
  logger(logFile, "\n");
  logger(logFile, "%% help lines\n");
  logger(logFile, "\\draw[color=lightgray,thin] (1,0) -- (1,1) -- (0,1)(0,0) -- (1,1);\n");
  logger(logFile, "%%\\draw[step=0.5cm,color=lightgray,thin] (0,0) grid (1,1);\n");
  if ((mc[numMaxtermCoords - 1].numZeroRounds / (float)(suppressedBytes*8)) > (mc[numMaxtermCoords - 1].bit / (float)bitspaceSize)) { /* last coordinate above line */
    logger(logFile, "\\draw[color=lightgray,thin] (0,%d/%d) -- (%d/%d,%d/%d) -- (%d/%d,0);\n",
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8);
    logger(logFile, "\\draw[color=lightgray,thin] (0,%d/%d) -- (%d/%d,%d/%d) -- (%d/%d,0);\n",
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].bit, bitspaceSize,
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].bit, bitspaceSize);
  } else { /* last coordinate on or below line */
    logger(logFile, "\\draw[color=lightgray,thin] (0,%d/%d) -- (%d/%d,%d/%d) -- (%d/%d,0);\n",
      mc[numMaxtermCoords - 1].bit, bitspaceSize,
      mc[numMaxtermCoords - 1].bit, bitspaceSize,
      mc[numMaxtermCoords - 1].bit, bitspaceSize,
      mc[numMaxtermCoords - 1].bit, bitspaceSize);
    logger(logFile, "\\draw[color=lightgray,thin] (0,%d/%d) -- (%d/%d,%d/%d) -- (%d/%d,0);\n",
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].bit, bitspaceSize,
      mc[numMaxtermCoords - 1].numZeroRounds, suppressedBytes*8,
      mc[numMaxtermCoords - 1].bit, bitspaceSize);
  }
  logger(logFile, "\n");
  logger(logFile, "%% axes\n");
  logger(logFile, "\\draw[<->] (0,1.1) -- (0,0) -- (1.1,0);\n");
  logger(logFile, "\\draw (0,0.95)[anchor=west] node{\\scriptsize distinguisher};\n");
  logger(logFile, "\\draw (0,0.90)[anchor=west] node{\\scriptsize efficiency};\n");
  logger(logFile, "\n");
  logger(logFile, "%% ticks\n");
  logger(logFile, "\\foreach \\x in {0.25,0.5,...,1}{\\draw (\\x,-0.01) -- (\\x,0.01);}\n");
  logger(logFile, "\\foreach \\y in {0.25,0.5,...,1}{\\draw (-0.01,\\y) -- (0.01,\\y);}\n");
  logger(logFile, "\\draw (0.82,0.05) node{\\scriptsize bit set size};\n");
  logger(logFile, "\\draw (0.25,0)[anchor=north] node{\\tiny $%d$};\n", bitspaceSize/4);
  logger(logFile, "\\draw (0.50,0)[anchor=north] node{\\tiny $%d$};\n", bitspaceSize/2);
  logger(logFile, "\\draw (0.75,0)[anchor=north] node{\\tiny $%d$};\n", bitspaceSize*3/4);
  logger(logFile, "\\draw (1.00,0)[anchor=north] node{\\tiny $%d$};\n", bitspaceSize);
  logger(logFile, "\\draw (0,0.25)[anchor=east] node{\\tiny $25\\%%$};\n");
  logger(logFile, "\\draw (0,0.50)[anchor=east] node{\\tiny $50\\%%$};\n");
  logger(logFile, "\\draw (0,0.75)[anchor=east] node{\\tiny $75\\%%$};\n");
  logger(logFile, "\\draw (0,1.00)[anchor=east] node{\\tiny $100\\%%$};\n");
  logger(logFile, "\n");
  logger(logFile, "%% plot\n");
  logger(logFile, "\\draw (%d/%d,%d/%d)", mc[0].bit, bitspaceSize, mc[0].numZeroRounds, suppressedBytes*8);
  for (i=1; i<numMaxtermCoords; i++) {
    logger(logFile, " -- (%d/%d,%d/%d)", mc[i].bit, bitspaceSize, mc[i].numZeroRounds, suppressedBytes*8);
    if ((i & 3) == 0)
      logger(logFile, "\n");
  }
  logger(logFile, ";\n");
  logger(logFile, "\n");
  logger(logFile, "%% Zoomed plot\n");

  logger(logFile, "\n");
  logger(logFile, "%% help lines\n");
  logger(logFile, "\\draw[color=lightgray,thin] (%2.2f,0) -- (%2.2f,1) -- (%2.2f,1);\n", ZOFF + 1, ZOFF + 1, ZOFF);
  logger(logFile, "\n");
  logger(logFile, "%% axes\n");
  logger(logFile, "\\draw[<->] (%2.2f,1.1) -- (%2.2f,0) -- (%2.2f,0);\n", ZOFF, ZOFF, ZOFF + 1.1);
  logger(logFile, "\\draw (%2.2f,0.05) node{\\scriptsize bit set size};\n", ZOFF + 0.82);
  logger(logFile, "\\draw (%2.2f,0.95)[anchor=west] node{\\scriptsize number of};\n", ZOFF);
  logger(logFile, "\\draw (%2.2f,0.90)[anchor=west] node{\\scriptsize zero bits};\n", ZOFF);
  logger(logFile, "\n");
  logger(logFile, "%% ticks\n");
  if (closeX)
    logger(logFile, "\\foreach \\x in {%d,%d,...,%d,%d}{\\draw (%2.2f + \\x/%d,-0.01) -- (%2.2f + \\x/%d,0.01);}\n", dx, 2*dx, maxx / dx * dx - dx, maxx, ZOFF, maxx, ZOFF, maxx);
  else
    logger(logFile, "\\foreach \\x in {%d,%d,...,%d,%d}{\\draw (%2.2f + \\x/%d,-0.01) -- (%2.2f + \\x/%d,0.01);}\n", dx, 2*dx, maxx / dx * dx, maxx, ZOFF, maxx, ZOFF, maxx);
  if (closeY)
    logger(logFile, "\\foreach \\y in {%d,%d,...,%d,%d}{\\draw (%2.2f - 0.01,\\y/%d) -- (%2.2f + 0.01,\\y/%d);}\n", dy, 2*dy, maxy / dy * dy - dy, maxy, ZOFF, maxy, ZOFF, maxy);
  else
    logger(logFile, "\\foreach \\y in {%d,%d,...,%d,%d}{\\draw (%2.2f - 0.01,\\y/%d) -- (%2.2f + 0.01,\\y/%d);}\n", dy, 2*dy, maxy / dy * dy, maxy, ZOFF, maxy, ZOFF, maxy);
  for (i=0; i<numLabelsX; i++)
    logger(logFile, "\\draw (%2.2f + %d/%d,0)[anchor=north] node{\\tiny $%d$};\n", ZOFF, (i+1)*dx, maxx, (i+1)*dx);
  logger(logFile, "\\draw (%2.2f + %d/%d,0)[anchor=north] node{\\tiny $%d$};\n", ZOFF, maxx, maxx, maxx);
  for (i=0; i<numLabelsY; i++)
    logger(logFile, "\\draw (%2.2f,%d/%d)[anchor=east] node{\\tiny $%d$};\n", ZOFF, (i+1)*dy, maxy, (i+1)*dy);
  logger(logFile, "\\draw (%2.2f,%d/%d)[anchor=east] node{\\tiny $%d$};\n", ZOFF, maxy, maxy, maxy);
  logger(logFile, "\n");
  logger(logFile, "%% plot\n");
  logger(logFile, "\\draw (%2.2f+%d/%d,%d/%d)", ZOFF, mc[0].bit, maxx, mc[0].numZeroRounds, maxy);
  for (i=1; i<numMaxtermCoords; i++) {
    logger(logFile, " -- (%2.2f+%d/%d,%d/%d)", ZOFF, mc[i].bit, maxx, mc[i].numZeroRounds, maxy);
    if ((i & 3) == 0)
      logger(logFile, "\n");
  }
  logger(logFile, ";\n\n");
  logger(logFile, "\\end{scope}\n");
  fflush(logFile);
}

