/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_greedy_maxterm.h"
#include "black_box_optimal_maxterm.h"
#include "black_box_utils.h"
#include "black_box_bit_set_utils.h"
#include "bitslice_utils.h"
#include "bittwiddling.h"
#include "press_any_key.h"
#include "assert_utils.h"
#include "memory_utils.h"
#include "black_box_latex_graph_writer.h"

#define MAX_BIT_LEN 1024

/*******************************************************************************
 * Utilities
 ******************************************************************************/
static int bitCountByBlocks(bbCipher cipher) {
  if (blackBoxCipherType(cipher) == kBlockCipher) {
    return 1;
  }
  return 0;
}

/*******************************************************************************
 * Greedy bit sets
 ******************************************************************************/
#define MAX_BIT_SET_SIZE 40
  typedef struct {
    const int numBits;
    const int bit[MAX_BIT_SET_SIZE];
  } singleBitSetType;

  typedef struct {
    const singleBitSetType key;
    const singleBitSetType iv;
  } doubleBitSetType;

  typedef struct {
    const int zeroRoundsKey;
    const singleBitSetType key;
    const int zeroRoundsIv;
    const singleBitSetType iv;
    const int zeroRoundsKeyIv;
    const doubleBitSetType keyIv;
  } greedySetType;

  static const greedySetType greedySet[kNumBlackBoxCiphers] = {
    {806, {36, {33,42,45,57,72,78,48,6,66,63,51,36,24,54,9,27,13,0,15,69,3,60,75,30,39,18,21,12,74,47,76,49,73,71,55,10}} /* key bit set */,
     838, {37, {3,21,27,48,54,63,72,30,57,39,42,18,78,12,0,66,15,33,75,36,9,24,6,60,45,51,69,1,70,79,55,77,68,64,74,76,40}} /* iv bit set */,
     972, { {20, {34,43,76,7,49,61,16,52,25,46,70,55,22,19,12,10,31,73,64,1}} /* key bit set */,
            {14, {31,64,46,34,67,43,4,1,49,28,19,7,37,10}} /* iv bit set */ } }, /* Trivium */
    {8, {28, {12,40,81,96,112,23,52,24,65,79,41,82,85,73,57,70,99,14,7,58,76,118,9,48,98,16,28,42}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* Rabbit */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* Edon80 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* AES-128 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* AES-256 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* DES */
    {216, {31, {14,113,118,120,123,125,13,22,0,52,25,1,55,4,115,88,74,34,127,71,33,101,57,19,58,63,45,48,70,103,9}} /* key bit set */,
     244, {31, {34,59,63,64,67,69,29,37,15,65,94,26,1,78,80,10,42,7,2,23,32,18,12,68,92,21,28,91,20,45,43}} /* iv bit set */,
     238, { {17, {24,18,97,124,7,26,127,83,69,105,126,104,33,1,0,4,3}} /* key bit set */,
            {13, {63,64,67,68,69,58,36,35,0,1,59,7,33}} /* iv bit set */ } }, /* Grain-128 */
    {95, {7, {1,27,31,33,58,71,77}} /* key bit set */,
     96, {7, {1,22,26,37,45,47,55}} /* iv bit set */,
     83, { {20, {24,33,23,39,59,66,34,30,15,19,17,50,53,27,32,54,18,73,49,1}} /* key bit set */,
           {11, {16,20,23,28,7,30,25,32,13,60,15}} /* iv bit set */ } }, /* Grain v1 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* TEA */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* XTEA */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* SEED */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* PRESENT */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* SMS4 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* Camellia */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* RC5 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* RC6 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* HIGHT */
#if 0
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* Clefia */
#endif
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* HC-128 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* HC-256 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* MICKEY v2 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }, /* Salsa20/12 */
    {0, {0, {0}} /* key bit set */,
     0, {0, {0}} /* iv bit set */,
     0, { {0, {0}} /* key bit set */,
          {0, {0}} /* iv bit set */ } }  /* SOSEMANUK */
  };

int greedyBitSetStored(bbCipher cipher, int keyBitsAllowed, int ivBitsAllowed) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Unknown cipher!\n");
  ASSERT(keyBitsAllowed || ivBitsAllowed, "Undefined bit set!\n");
  if (keyBitsAllowed && ivBitsAllowed) return greedySet[cipher].keyIv.key.numBits + greedySet[cipher].keyIv.iv.numBits;
  if (keyBitsAllowed) return greedySet[cipher].key.numBits;
  if (ivBitsAllowed) return greedySet[cipher].iv.numBits;
  return 0;
}

void getGreedyBitSet(bbCipher cipher, int *numKeyBits, int *keyBit, int *numIvBits, int *ivBit) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Unknown cipher!\n");
  ASSERT(keyBit || ivBit, "Undefined bit set!\n");
  if (keyBit && ivBit) {
    *numKeyBits = greedySet[cipher].keyIv.key.numBits;
    MEMCPY(keyBit, greedySet[cipher].keyIv.key.bit, *numKeyBits * sizeof(int));
    *numIvBits = greedySet[cipher].keyIv.iv.numBits;
    MEMCPY(ivBit, greedySet[cipher].keyIv.iv.bit, *numIvBits * sizeof(int));
    return;
  }
  if (keyBit) {
    *numKeyBits = greedySet[cipher].key.numBits;
    MEMCPY(keyBit, greedySet[cipher].key.bit, *numKeyBits * sizeof(int));
    return;
  }
  if (ivBit) {
    *numIvBits = greedySet[cipher].iv.numBits;
    MEMCPY(ivBit, greedySet[cipher].iv.bit, *numIvBits * sizeof(int));
    return;
  }
  ASSERT_ALWAYS("Parameter error!\n");
}

/*******************************************************************************
 * Add one bit
 ******************************************************************************/
/*
 * xorBuf contains the xor of the summation over the given set
 *
 * returns 0 if a key bit was added
 * returns 1 if an iv bit was added
 * returns -1 on errors
 * the best bit is added to ivBitSet or keyBitSet
 * xorBuf contains the xor for the summation over the resulting bit set
 */
static int addOneBit(bbCipher cipher, BYTE *key, BYTE *iv, int *keyBitSet, int *numKeyBits, int okToUseKeyBits, int *ivBitSet, int *numIvBits, int okToUseIvBits, const BYTE *in, int inLen, BYTE *xor, int numParallellBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE *bestXor;
  BYTE *tempXor;
  int bestBit, bestBitType, maxZeroRounds = -1, i;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  bestXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  tempXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));

  /* iterate over key bits */
  if (okToUseKeyBits) {

    if (*numKeyBits == keySize * 8) return -1; /* key space exhausted */

    for (i=0; i<keySize*8; i++) {
      int prevBitValue, z;

      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;

      prevBitValue = getBufBit(key, i); /* get current key bit i */
      setBufBit(key, i, !prevBitValue); /* flip key bit */
      MEMCPY(tempXor, xor, suppressedBytes);
      if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) return -1;
      setBufBit(key, i, prevBitValue); /* restore key bit i */

      z = bitCountByBlocks(cipher) ?
          numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize) :
          numInitialZeroBits(tempXor, suppressedBytes);
      if (z > maxZeroRounds) {
        maxZeroRounds = z;
        bestBit = i;
        bestBitType = 0;
        MEMCPY(bestXor, tempXor, suppressedBytes);
      }
    }
  }

  /* iterate over iv bits */
  if (okToUseIvBits) {

    if (*numIvBits == ivSize * 8) return -1; /* iv space exhausted */

    for (i=0; i<ivSize*8; i++) {
      int prevBitValue, z;

      if (bitSetContainsValue(ivBitSet, *numIvBits, i)) continue;

      prevBitValue = getBufBit(iv, i); /* get current iv bit i */
      setBufBit(iv, i, !prevBitValue); /* flip iv bit */
      MEMCPY(tempXor, xor, suppressedBytes);
      if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) return -1;
      setBufBit(iv, i, prevBitValue); /* restore iv bit i */

      if (bitCountByBlocks(cipher))
        z = numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize);
      else
        z = numInitialZeroBits(tempXor, suppressedBytes);
      if (z > maxZeroRounds) {
        maxZeroRounds = z;
        bestBit = i;
        bestBitType = 1;
        MEMCPY(bestXor, tempXor, suppressedBytes);
      }
    }
  }

  /* update bit set */
  {
    int *numBits = bestBitType == 0 ? numKeyBits : numIvBits;
    int *bitSet = bestBitType == 0 ? keyBitSet : ivBitSet;
    bitSet[*numBits] = bestBit;
    *numBits = *numBits + 1;
    MEMCPY(xor, bestXor, suppressedBytes);
  }

  /* cleanup */
  FREE(bestXor);
  FREE(tempXor);

  return bestBitType;
}

/*******************************************************************************
 * Add two bits
 ******************************************************************************/
/*
 * xorBuf contains the xor of the summation over the given set
 *
 * returns 0 if two key bits were added
 * returns 1 if one key and one iv bit was added
 * returns 2 if two iv bits were added
 * returns -1 on errors
 * the best bits are added to ivBitSet and/or keyBitSet
 * xorBuf contains the xor for the summation over the resulting bit set
 */
static int addTwoBits(bbCipher cipher, BYTE *key, BYTE *iv, int *keyBitSet, int *numKeyBits, int okToUseKeyBits, int *ivBitSet, int *numIvBits, int okToUseIvBits, const BYTE *in, int inLen, BYTE *xor, int numParallellBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE *bestXor;
  BYTE *tempXor;
  int bestBit1, bestBit2, bestBitType, maxZeroRounds = -1, i, j;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  bestXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  tempXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));

  /* try two key bits */
  if (okToUseKeyBits) {

    if (*numKeyBits > keySize * 8 - 2) { FREE(bestXor); FREE(tempXor); return -1; } /* key space exhausted */

    for (i=0; i<keySize*8; i++) {
      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;

      for (j=i+1; j<keySize*8; j++) {
        int z;

        if (bitSetContainsValue(keyBitSet, *numKeyBits, j)) continue;

        MEMCPY(tempXor, xor, suppressedBytes);

        flipBufBit(key, i); /* flip key bit i */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* restore key bit i */
        flipBufBit(key, j); /* flip key bit j */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* flip key bit i */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* restore key bit i */
        flipBufBit(key, j); /* restore key bit j */

        z = bitCountByBlocks(cipher) ?
            numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize) :
            numInitialZeroBits(tempXor, suppressedBytes);
        if (z > maxZeroRounds) {
          maxZeroRounds = z;
          bestBit1 = i;
          bestBit2 = j;
          bestBitType = 0;
          MEMCPY(bestXor, tempXor, suppressedBytes);
        }
      }
    }
  }

  /* try two iv bits */
  if (okToUseIvBits) {

    if (*numIvBits > ivSize * 8 - 2) { FREE(bestXor); FREE(tempXor); return -1; } /* iv space exhausted */

    for (i=0; i<ivSize*8; i++) {

      if (bitSetContainsValue(ivBitSet, *numIvBits, i)) continue;

      for (j=i+1; j<ivSize*8; j++) {
        int z;

        if (bitSetContainsValue(ivBitSet, *numIvBits, j)) continue;

        MEMCPY(tempXor, xor, suppressedBytes);

        flipBufBit(iv, i); /* flip iv bit i */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(iv, i); /* restore iv bit i */
        flipBufBit(iv, j); /* flip iv bit j */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(iv, i); /* flip iv bit i */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(iv, i); /* restore iv bit i */
        flipBufBit(iv, j); /* restore iv bit j */

        if (bitCountByBlocks(cipher))
          z = numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize);
        else
          z = numInitialZeroBits(tempXor, suppressedBytes);
        if (z > maxZeroRounds) {
          maxZeroRounds = z;
          bestBit1 = i;
          bestBit2 = j;
          bestBitType = 2;
          MEMCPY(bestXor, tempXor, suppressedBytes);
        }
      }
    }
  }

  /* try one key and one iv bit */
  if (okToUseKeyBits && okToUseIvBits) {

    if (*numKeyBits == keySize * 8) { FREE(bestXor); FREE(tempXor); return -1; } /* key space exhausted */
    if (*numIvBits == ivSize * 8) { FREE(bestXor); FREE(tempXor); return -1; } /* iv space exhausted */

    for (i=0; i<keySize*8; i++) {

      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;

      for (j=0; j<ivSize*8; j++) {
        int z;

        if (bitSetContainsValue(ivBitSet, *numIvBits, j)) continue;

        MEMCPY(tempXor, xor, suppressedBytes);

        flipBufBit(key, i); /* flip key bit */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* restore key bit */
        flipBufBit(iv, j); /* flip iv bit */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* flip key bit */
        if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
        flipBufBit(key, i); /* restore key bit */
        flipBufBit(iv, j); /* restore iv bit */

        if (bitCountByBlocks(cipher))
          z = numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize);
        else
          z = numInitialZeroBits(tempXor, suppressedBytes);
        if (z > maxZeroRounds) {
          maxZeroRounds = z;
          bestBit1 = i;
          bestBit2 = j;
          bestBitType = 1;
          MEMCPY(bestXor, tempXor, suppressedBytes);
        }
      }
    }
  }

  /* update bit set(s) */
  if (bestBitType == 0) { /* two key bits */
    keyBitSet[*numKeyBits] = bestBit1;
    keyBitSet[*numKeyBits + 1] = bestBit2;
    *numKeyBits = *numKeyBits + 2;
  } else if (bestBitType == 1) { /* one key and one iv bit */
    keyBitSet[*numKeyBits] = bestBit1; *numKeyBits = *numKeyBits + 1;
    ivBitSet[*numIvBits] = bestBit2; *numIvBits = *numIvBits + 1;
  } else if (bestBitType == 2) { /* two iv bits */
    ivBitSet[*numIvBits] = bestBit1;
    ivBitSet[*numIvBits + 1] = bestBit2;
    *numIvBits = *numIvBits + 2;
  } else {
    FREE(bestXor);
    FREE(tempXor);
    return -1;
  }
  /* update xor */
  MEMCPY(xor, bestXor, suppressedBytes);

  /* cleanup */
  FREE(bestXor);
  FREE(tempXor);
  return bestBitType;
}

/*******************************************************************************
 * Add three bits
 ******************************************************************************/
/*
 * xorBuf contains the xor of the summation over the given set
 *
 * returns the number of iv bits added
 * returns -1 on errors
 * the best bits are added to ivBitSet and/or keyBitSet
 * xorBuf contains the xor for the summation over the resulting bit set
 */
static int addThreeBits(bbCipher cipher, BYTE *key, BYTE *iv, int *keyBitSet, int *numKeyBits, int okToUseKeyBits, int *ivBitSet, int *numIvBits, int okToUseIvBits, const BYTE *in, int inLen, BYTE *xor, int numParallellBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE *bestXor;
  BYTE *tempXor;
  int bestBit1, bestBit2, bestBit3, bestBitType, maxZeroRounds = -1, i, j, k;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  bestXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));
  tempXor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE));

  /* try three key bits */
  if (okToUseKeyBits) {

    if (*numKeyBits > keySize * 8 - 3) { FREE(bestXor); FREE(tempXor); return -1; } /* key space exhausted */

    for (i=0; i<keySize*8; i++) {
      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;
      for (j=i+1; j<keySize*8; j++) {
        if (bitSetContainsValue(keyBitSet, *numKeyBits, j)) continue;
        for (k=j+1; k<keySize*8; k++) {
          int z;

          if (bitSetContainsValue(keyBitSet, *numKeyBits, k)) continue;

          MEMCPY(tempXor, xor, suppressedBytes);

          flipBufBit(key, i); /* flip key bit i */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore key bit i */
          flipBufBit(key, j); /* flip key bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, j); /* restore key bit j */
          flipBufBit(key, k); /* flip key bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, k); /* restore key bit k */

          flipBufBit(key, i); /* flip key bit i */
          flipBufBit(key, j); /* flip key bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, j); /* restore key bit j */
//          flipBufBit(key, i); /* restore key bit i */
//          flipBufBit(key, i); /* flip key bit i */
          flipBufBit(key, k); /* flip key bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore key bit i */
//          flipBufBit(key, k); /* restore key bit k */
//          flipBufBit(key, k); /* flip key bit k */
          flipBufBit(key, j); /* flip key bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
//          flipBufBit(key, j); /* restore key bit j */
//          flipBufBit(key, k); /* restore key bit k */

          flipBufBit(key, i); /* flip key bit i */
//          flipBufBit(key, j); /* flip key bit j */
//          flipBufBit(key, k); /* flip key bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore key bit i */
          flipBufBit(key, j); /* restore key bit j */
          flipBufBit(key, k); /* restore key bit k */

          z = bitCountByBlocks(cipher) ?
              numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize) :
              numInitialZeroBits(tempXor, suppressedBytes);
          if (z > maxZeroRounds) {
            maxZeroRounds = z;
            bestBit1 = i;
            bestBit2 = j;
            bestBit3 = k;
            bestBitType = 0;
            MEMCPY(bestXor, tempXor, suppressedBytes);
          }
        }
      }
    }
  }

  /* try three iv bits */
  if (okToUseIvBits) {

    if (*numIvBits > ivSize * 8 - 3) { FREE(bestXor); FREE(tempXor); return -1; } /* iv space exhausted */

    for (i=0; i<ivSize*8; i++) {

      if (bitSetContainsValue(ivBitSet, *numIvBits, i)) continue;

      for (j=i+1; j<ivSize*8; j++) {
        if (bitSetContainsValue(ivBitSet, *numIvBits, j)) continue;

        for (k=j+1; k<ivSize*8; k++) {
          int z;

          if (bitSetContainsValue(ivBitSet, *numIvBits, k)) continue;

          MEMCPY(tempXor, xor, suppressedBytes);

          flipBufBit(iv, i); /* flip iv bit i */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, i); /* restore iv bit i */
          flipBufBit(iv, j); /* flip iv bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, j); /* restore iv bit j */
          flipBufBit(iv, k); /* flip iv bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, k); /* restore iv bit k */

          flipBufBit(iv, i); /* flip iv bit i */
          flipBufBit(iv, j); /* flip iv bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
//          flipBufBit(iv, i); /* restore iv bit i */
          flipBufBit(iv, j); /* restore iv bit j */
//          flipBufBit(iv, i); /* flip iv bit i */
          flipBufBit(iv, k); /* flip iv bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, i); /* restore iv bit i */
//          flipBufBit(iv, k); /* restore iv bit k */
          flipBufBit(iv, j); /* flip iv bit j */
//          flipBufBit(iv, k); /* flip iv bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
//          flipBufBit(iv, j); /* restore iv bit j */
//          flipBufBit(iv, k); /* restore iv bit k */

          flipBufBit(iv, i); /* flip iv bit i */
//          flipBufBit(iv, j); /* flip iv bit j */
//          flipBufBit(iv, k); /* flip iv bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, i); /* restore iv bit i */
          flipBufBit(iv, j); /* restore iv bit j */
          flipBufBit(iv, k); /* restore iv bit k */

          if (bitCountByBlocks(cipher))
            z = numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize);
          else
            z = numInitialZeroBits(tempXor, suppressedBytes);
          if (z > maxZeroRounds) {
            maxZeroRounds = z;
            bestBit1 = i;
            bestBit2 = j;
            bestBit3 = k;
            bestBitType = 3;
            MEMCPY(bestXor, tempXor, suppressedBytes);
            logDateRaw(NULL, LOGNOFLUSH);
            printf(", %d rounds for bits { %2d, %2d, %2d }, xor =", z, bestBit1, bestBit2, bestBit3);
            logBuf(NULL, LOGSCREEN, bestXor, suppressedBytes, 0, 0);
            printf("\n");
          }
        }
      }
    }
  }

  /* try two key and one iv bit */
  if (okToUseKeyBits && okToUseIvBits) {
    if (*numKeyBits > keySize * 8 - 2) { FREE(bestXor); FREE(tempXor); return -1; } /* key space exhausted */
    if (*numIvBits > ivSize * 8 - 1) { FREE(bestXor); FREE(tempXor); return -1; } /* iv space exhausted */

    for (i=0; i<keySize*8; i++) {
      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;
      for (j=i+1; j<keySize*8; j++) {
        if (bitSetContainsValue(keyBitSet, *numKeyBits, j)) continue;
        for (k=0; k<ivSize*8; k++) {
          int z;

          if (bitSetContainsValue(ivBitSet, *numIvBits, k)) continue;

          MEMCPY(tempXor, xor, suppressedBytes);

          flipBufBit(key, i); /* flip bit i */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
          flipBufBit(key, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, j); /* restore bit j */
          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, k); /* restore bit k */

          flipBufBit(key, i); /* flip bit i */
          flipBufBit(key, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, j); /* restore bit j */
//          flipBufBit(key, i); /* restore bit i */
//          flipBufBit(key, i); /* flip bit i */
          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
//          flipBufBit(iv, k); /* restore bit k */
//          flipBufBit(iv, k); /* flip bit k */
          flipBufBit(key, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
//          flipBufBit(key, j); /* restore bit j */
//          flipBufBit(iv, k); /* restore bit k */

          flipBufBit(key, i); /* flip bit i */
//          flipBufBit(key, j); /* flip bit j */
//          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
          flipBufBit(key, j); /* restore bit j */
          flipBufBit(iv, k); /* restore bit k */

          z = bitCountByBlocks(cipher) ?
              numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize) :
              numInitialZeroBits(tempXor, suppressedBytes);
          if (z > maxZeroRounds) {
            maxZeroRounds = z;
            bestBit1 = i;
            bestBit2 = j;
            bestBit3 = k;
            bestBitType = 1;
            MEMCPY(bestXor, tempXor, suppressedBytes);
          }
        }
      }
    }
  }

  /* try one key and two iv bits */
  if (okToUseKeyBits && okToUseIvBits) {
    if (*numKeyBits > keySize * 8 - 1) { FREE(bestXor); FREE(tempXor); return -1; } /* key space exhausted */
    if (*numIvBits > ivSize * 8 - 2) { FREE(bestXor); FREE(tempXor); return -1; } /* iv space exhausted */

    for (i=0; i<keySize*8; i++) {
      if (bitSetContainsValue(keyBitSet, *numKeyBits, i)) continue;
      for (j=0; j<ivSize*8; j++) {
        if (bitSetContainsValue(ivBitSet, *numIvBits, j)) continue;
        for (k=j+1; k<ivSize*8; k++) {
          int z;

          if (bitSetContainsValue(ivBitSet, *numIvBits, k)) continue;

          MEMCPY(tempXor, xor, suppressedBytes);

          flipBufBit(key, i); /* flip bit i */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
          flipBufBit(iv, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, j); /* restore bit j */
          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, k); /* restore bit k */

          flipBufBit(key, i); /* flip bit i */
          flipBufBit(key, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(iv, j); /* restore bit j */
//          flipBufBit(key, i); /* restore bit i */
//          flipBufBit(key, i); /* flip bit i */
          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
//          flipBufBit(iv, k); /* restore bit k */
//          flipBufBit(iv, k); /* flip bit k */
          flipBufBit(iv, j); /* flip bit j */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
//          flipBufBit(iv, j); /* restore bit j */
//          flipBufBit(iv, k); /* restore bit k */

          flipBufBit(key, i); /* flip bit i */
//          flipBufBit(iv, j); /* flip bit j */
//          flipBufBit(iv, k); /* flip bit k */
          if (xorOverBitSet(cipher, key, *numKeyBits, keyBitSet, iv, *numIvBits, ivBitSet, in, inLen, tempXor, numParallellBits)) { FREE(bestXor); FREE(tempXor); return -1; }
          flipBufBit(key, i); /* restore bit i */
          flipBufBit(iv, j); /* restore bit j */
          flipBufBit(iv, k); /* restore bit k */

          z = bitCountByBlocks(cipher) ?
              numInitialZeroBitsByBlocks(tempXor, suppressedBytes, implicitBlockSize) :
              numInitialZeroBits(tempXor, suppressedBytes);
          if (z > maxZeroRounds) {
            maxZeroRounds = z;
            bestBit1 = i;
            bestBit2 = j;
            bestBit3 = k;
            bestBitType = 2;
            MEMCPY(bestXor, tempXor, suppressedBytes);
          }
        }
      }
    }
  }

  /* update bit set(s) */
  switch (bestBitType) {
  case 0: /* three key bits */
    keyBitSet[*numKeyBits] = bestBit1;
    keyBitSet[*numKeyBits + 1] = bestBit2;
    keyBitSet[*numKeyBits + 2] = bestBit3;
    *numKeyBits = *numKeyBits + 3;
    break;
  case 1: /* two key and one iv bit */
    keyBitSet[*numKeyBits] = bestBit1;
    keyBitSet[*numKeyBits + 1] = bestBit2;
    ivBitSet[*numIvBits] = bestBit3;
    *numKeyBits = *numKeyBits + 2;
    *numIvBits = *numIvBits + 1;
    break;
  case 2: /* one key and two iv bits */
    keyBitSet[*numKeyBits] = bestBit1;
    ivBitSet[*numIvBits] = bestBit2;
    ivBitSet[*numIvBits + 1] = bestBit3;
    *numKeyBits = *numKeyBits + 1;
    *numIvBits = *numIvBits + 2;
    break;
  case 3: /* three iv bits */
    ivBitSet[*numIvBits] = bestBit1;
    ivBitSet[*numIvBits + 1] = bestBit2;
    ivBitSet[*numIvBits + 2] = bestBit3;
    *numIvBits = *numIvBits + 3;
    break;
  default:
    FREE(bestXor);
    FREE(tempXor);
    return -1;
  }

  /* update xor */
  MEMCPY(xor, bestXor, suppressedBytes);

  /* cleanup */
  FREE(bestXor);
  FREE(tempXor);
  return bestBitType;
}

/*******************************************************************************
 * Latex Graph
 ******************************************************************************/
void sendToLatexWriter(FILE *texFile, bbCipher cipher, int okToUseKeyBits, int okToUseIvBits, int numCoords, MaxtermCoordinate *mc) {
  int keySize, ivSize, bitspaceSize = 0;
  blackBoxInfo(cipher, &keySize, &ivSize, NULL, NULL);
  if (okToUseKeyBits) bitspaceSize += keySize * 8;
  if (okToUseIvBits)  bitspaceSize += ivSize * 8;
  ASSERT(bitspaceSize > 0, "Parameter error!\n");
  writeLatexMaxtermPlot(texFile, cipher, numCoords, mc, bitspaceSize);
}

/*******************************************************************************
 * Greedy Maxtest Distinguisher
 ******************************************************************************/
static int printStats(FILE *logFile, bbCipher cipher, int numKeyBits, const int *keyBit, int numIvBits, const int *ivBit, int suppressedBytes, int implicitBlockSize, const BYTE *xor, time_t start) {
  int z;
  logger(logFile, LOGSCREEN | LOGFILE, "[");
  logDateRaw(logFile, LOGSCREEN | LOGFILE);
  logger(logFile, LOGSCREEN | LOGFILE, ", ");
  logTimeRaw(logFile, LOGSCREEN | LOGFILE, start);
  z = bitCountByBlocks(cipher) ?
      numInitialZeroBitsByBlocks(xor, suppressedBytes, implicitBlockSize) :
      numInitialZeroBits(xor, suppressedBytes);
  if (blackBoxCipherType(cipher) == kBlockCipher) {
    logger(logFile, LOGSCREEN | LOGFILE, "] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f blocks = %5.2f%%", numKeyBits + numIvBits, z, suppressedBytes * 8, z/(float)8/(float)implicitBlockSize, 100*z/(float)(suppressedBytes*8));
  } else {
    logger(logFile, LOGSCREEN | LOGFILE, "] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f%%", numKeyBits + numIvBits, z, suppressedBytes * 8, 100*z/(float)8/(float)suppressedBytes);
  }
  logger(logFile, LOGSCREEN | LOGFILE, ", xor =");
  logBuf(logFile, LOGSCREEN | LOGFILE, xor, suppressedBytes, blackBoxCipherType(cipher) == kBlockCipher ? implicitBlockSize : 0, 0);
  if (keyBit) { logger(logFile, LOGSCREEN | LOGFILE, ", Key bit set = "); logBitSet(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, keyBit, numKeyBits); }
  if (ivBit) { logger(logFile, LOGSCREEN | LOGFILE, ", IV bit set = "); logBitSet(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, ivBit, numIvBits); }
  logger(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, "\n");
  return z;
}

void blackBoxGreedyMaxtermDistinguisher(bbCipher cipher,
                                        const char *logFileName,
                                        const char *texFileName,
                                        int numBits, int keyFill, int ivFill,
                                        int sizeOfStartingOptimalSet,
                                        int okToUseKeyBits, int okToUseIvBits,
                                        int strategy,
                                        int numParallellBits) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  BYTE key[MAX_BIT_LEN], iv[MAX_BIT_LEN];
  int _keyBit[MAX_BIT_LEN], _ivBit[MAX_BIT_LEN];
  int numKeyBits = 0, numIvBits = 0;
  int *keyBit = okToUseKeyBits ? _keyBit : NULL;
  int *ivBit = okToUseIvBits ? _ivBit : NULL;
  BYTE *in, *xor;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  UINT64 i;
  time_t start = time(NULL);
  int z;
  MaxtermCoordinate *mc;
//  FILE *logFile = openLogFile(cipher, keyFill, ivFill, okToUseKeyBits, okToUseIvBits);
  FILE *logFile = fopen(logFileName, "w");

  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Unknown cipher!");

  for (i=0; i<nameLen+11; i++) logger(logFile, LOGNOFLUSH, "*");
  logger(logFile, LOGNOFLUSH, "\n* Greedy %s *\n", name);
  for (i=0; i<nameLen+11; i++) logger(logFile, LOGNOFLUSH, "*");
  logger(logFile, LOGALL, "\n\n");
  if (blackBoxCipherType(cipher) == kBlockCipher)
    logger(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, "The number of initial zero bits are counted %s\n\n", bitCountByBlocks(cipher) ? "blockwise" : "sequentially");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    logger(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, "No standard implementation for %s\n\n\n", name);
    if (logFile) fclose(logFile);
    return;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);
  ASSERT(ivSize <= MAX_BIT_LEN, "Insuffiecient iv container length!");
  ASSERT(keySize <= MAX_BIT_LEN, "Insuffiecient key container length!");

  /* initialize */
  mc = (MaxtermCoordinate*)MALLOC(sizeof(MaxtermCoordinate) * (numBits - sizeOfStartingOptimalSet + 1));
  in = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE)); ASSERT(in, "Out of memory when allocating in buffer!");
  xor = (BYTE*)MALLOC(suppressedBytes * sizeof(BYTE)); ASSERT(xor, "Out of memory when allocating xor buffer!");
  bufFill(key, keyFill, keySize);
  bufFill(iv, ivFill, ivSize);
  MEMSET(in, 0, suppressedBytes);
  MEMSET(xor, 0, suppressedBytes);

  /* print key, iv and fill */
  logger(logFile, LOGNOFLUSH, "Key = ");
  logBuf(logFile, LOGNOFLUSH, key, keySize, 0, 0);
  logger(logFile, LOGNOFLUSH, "\n");
  logger(logFile, LOGNOFLUSH, "IV  = ");
  logBuf(logFile, LOGNOFLUSH, iv, ivSize, 0, 0);
  logger(logFile, LOGNOFLUSH, "\n\n");
  logger(logFile, LOGNOFLUSH, "Using key bits: %s\n", okToUseKeyBits ? "YES" : "NO");
  logger(logFile, LOGALL, "Using  IV bits: %s\n\n", okToUseIvBits ? "YES" : "NO");

  /* create (small) optimal initial bit set */
#if 1
  if (sizeOfStartingOptimalSet == -100) { /* Grain-128 greedy add2 */
    numKeyBits = 0;
    numIvBits = sizeOfStartingOptimalSet = 30;
    ivBit[ 0] = 34; ivBit[ 1] = 59; ivBit[ 2] = 63; ivBit[ 3] = 64; ivBit[ 4] = 67;
    ivBit[ 5] = 69; ivBit[ 6] = 55; ivBit[ 7] = 61; ivBit[ 8] = 25; ivBit[ 9] = 85;
    ivBit[10] = 35; ivBit[11] = 58; ivBit[12] =  2; ivBit[13] = 73; ivBit[14] = 30;
    ivBit[15] = 38; ivBit[16] =  5; ivBit[17] =  6; ivBit[18] = 10; ivBit[19] = 44;
    ivBit[20] = 24; ivBit[21] = 50; ivBit[22] =  3; ivBit[23] = 77; ivBit[24] = 91;
    ivBit[25] = 95; ivBit[26] = 12; ivBit[27] = 13; ivBit[28] = 41; ivBit[29] = 72;
    logger(logFile, LOGNOFLUSH, "Using hardcoded greedy bit set of size %d:", sizeOfStartingOptimalSet);
    //if (okToUseKeyBits)
    { logger(logFile, LOGNOFLUSH, " Key = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    //if (okToUseIvBits)
    { logger(logFile, LOGNOFLUSH, " IV = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGALL, "\n\n");
  } else if (sizeOfStartingOptimalSet == -101) { /* Trivium greedy add2 */
    numKeyBits = 12;
    numIvBits = 15;
    sizeOfStartingOptimalSet = numKeyBits + numIvBits;
    keyBit[ 0] = 34; keyBit[ 1] = 43; keyBit[ 2] = 76; keyBit[ 3] = 61; keyBit[ 4] = 58;
    keyBit[ 5] = 46; keyBit[ 6] = 25; keyBit[ 7] = 67; keyBit[ 8] = 52; keyBit[ 9] = 73;
    keyBit[10] =  7; keyBit[11] = 10;
    ivBit[ 0] = 31; ivBit[ 1] = 64; ivBit[ 2] = 55; ivBit[ 3] = 79; ivBit[ 4] = 67;
    ivBit[ 5] = 73; ivBit[ 6] =  1; ivBit[ 7] =  4; ivBit[ 8] = 61; ivBit[ 9] = 40;
    ivBit[10] = 49; ivBit[11] = 22; ivBit[12] = 37; ivBit[13] =  7; ivBit[14] = 34;
    logger(logFile, LOGNOFLUSH, "Using hardcoded greedy bit set of size %d:", sizeOfStartingOptimalSet);
    if (okToUseKeyBits) { logger(logFile, LOGNOFLUSH, " Key = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    if (okToUseIvBits) { logger(logFile, LOGNOFLUSH, " IV = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGALL, "\n\n");
  } else if (sizeOfStartingOptimalSet == -103) { /* Trivium greedy add3 */
    numKeyBits = 0;
    numIvBits = 15;
    sizeOfStartingOptimalSet = numKeyBits + numIvBits;
    ivBit[ 0] = 34; ivBit[ 1] = 59; ivBit[ 2] = 63; ivBit[ 3] = 64; ivBit[ 4] = 67;
    ivBit[ 5] = 69; ivBit[ 6] = 28; ivBit[ 7] = 31; ivBit[ 8] = 76; ivBit[ 9] =  5;
    ivBit[10] = 53; ivBit[11] = 68; ivBit[12] =  3; ivBit[13] =  7; ivBit[14] = 22;
    logger(logFile, LOGNOFLUSH, "Using hardcoded greedy bit set of size %d:", sizeOfStartingOptimalSet);
    if (okToUseKeyBits) { logger(logFile, LOGNOFLUSH, " Key = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    if (okToUseIvBits) { logger(logFile, LOGNOFLUSH, " IV = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGALL, "\n\n");
  } else
#endif
  if (sizeOfStartingOptimalSet > 0) {
    logger(logFile, LOGNOFLUSH, "Calculating optimal set of size %d...", sizeOfStartingOptimalSet);
    blackBoxOptimalMaxtermDistinguisher(cipher, sizeOfStartingOptimalSet, keyFill, &numKeyBits, keyBit, ivFill, &numIvBits, ivBit, 0 /* no output */, numParallellBits);
    if (keyBit) { logger(logFile, LOGNOFLUSH, ", Key bit set = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    if (ivBit) { logger(logFile, LOGNOFLUSH, ", IV bit set = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGNOFLUSH, "\n[");
    logDateRaw(logFile, LOGNOFLUSH);
    logger(logFile, LOGNOFLUSH, ", ");
    logTimeRaw(logFile, LOGNOFLUSH, start);
    logger(logFile, LOGALL, "]\n\n");
    start = time(NULL);
  } else if (sizeOfStartingOptimalSet == -1 && keyFill == 0 && optimalBitSetStored(cipher, okToUseKeyBits, okToUseIvBits)) { /* use stored optimal set */
    getOptimalBitSet(cipher, &numKeyBits, keyBit, &numIvBits, ivBit);
    sizeOfStartingOptimalSet = numKeyBits + numIvBits;
    logger(logFile, LOGNOFLUSH, "Using stored optimal bit set of size %d:", sizeOfStartingOptimalSet);
    if (okToUseKeyBits) { logger(logFile, LOGNOFLUSH, " Key = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    if (okToUseIvBits) { logger(logFile, LOGNOFLUSH, " IV = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGALL, "\n\n");
  } else if (sizeOfStartingOptimalSet == -2 && keyFill == 0 && greedyBitSetStored(cipher, okToUseKeyBits, okToUseIvBits)) { /* use stored greedy set */
    getGreedyBitSet(cipher, &numKeyBits, keyBit, &numIvBits, ivBit);
    sizeOfStartingOptimalSet = numKeyBits + numIvBits;
    logger(logFile, LOGNOFLUSH, "Using stored greedy bit set of size %d:", sizeOfStartingOptimalSet);
    if (okToUseKeyBits) { logger(logFile, LOGNOFLUSH, " Key = "); logBitSet(logFile, LOGNOFLUSH, keyBit, numKeyBits); }
    if (okToUseIvBits) { logger(logFile, LOGNOFLUSH, " IV = "); logBitSet(logFile, LOGNOFLUSH, ivBit, numIvBits); }
    logger(logFile, LOGALL, "\n\n");
  } else {
    logger(logFile, LOGALL, "Unhandled parameter sizeOfStartingOptimalSet=%d (keyFill=%d)!\n", sizeOfStartingOptimalSet, keyFill);
    fprintf(stderr, "Unhandled parameter sizeOfStartingOptimalSet=%d (keyFill=%d)!\n", sizeOfStartingOptimalSet, keyFill);
    ASSERT_ALWAYS("Unhandled parameter sizeOfStartingOptimalSet in Greedy!");
    return;
  }

  /* calculate xor for initial bit set */
  if (xorOverBitSet(cipher, key, numKeyBits, keyBit, iv, numIvBits, ivBit, in, implicitBlockSize, xor, numParallellBits)) { /* calculate xor of the initial bit set */
    logger(logFile, LOGALL, "Error when calculating xor of initial bit set\n");
    if (logFile) fclose(logFile);
    FREE(mc);
    FREE(in);
    FREE(xor);
    return;
  }
  mc[0].bit = sizeOfStartingOptimalSet;
  mc[0].numZeroRounds = bitCountByBlocks(cipher) ?
                        numInitialZeroBitsByBlocks(xor, suppressedBytes, implicitBlockSize) :
                        numInitialZeroBits(xor, suppressedBytes);

  /* print bit set */
  printStats(logFile, cipher, numKeyBits, keyBit, numIvBits, ivBit, suppressedBytes, implicitBlockSize, xor, start);

  for (i=0; i<numBits - sizeOfStartingOptimalSet; i++) {
    int r;

    switch (strategy) {
    case GREEDY_STRATEGY_ADD_3: r = addThreeBits(cipher, key, iv, keyBit, &numKeyBits, okToUseKeyBits, ivBit, &numIvBits, okToUseIvBits, in, implicitBlockSize, xor, numParallellBits); break;
    case GREEDY_STRATEGY_ADD_2: r = addTwoBits(cipher, key, iv, keyBit, &numKeyBits, okToUseKeyBits, ivBit, &numIvBits, okToUseIvBits, in, implicitBlockSize, xor, numParallellBits); break;
    default: r = addOneBit(cipher, key, iv, keyBit, &numKeyBits, okToUseKeyBits, ivBit, &numIvBits, okToUseIvBits, in, implicitBlockSize, xor, numParallellBits);
    }
    if (r == -1) {
      logger(logFile, LOGALL, "Error when adding a bit\n");
      if (logFile) fclose(logFile);
      FREE(mc);
      FREE(in);
      FREE(xor);
      return;
    }
    z = printStats(logFile, cipher, numKeyBits, keyBit, numIvBits, ivBit, suppressedBytes, implicitBlockSize, xor, start);
    mc[(int)i+1].bit = (int)i + 1 + sizeOfStartingOptimalSet;
    mc[(int)i+1].numZeroRounds = z;

    /* output latex graph */
#if 0
    if ((i >= 12 - 1 - sizeOfStartingOptimalSet) || (z == suppressedBytes * 8) || (i == numBits - sizeOfStartingOptimalSet - 1)) {
      FILE *texFile = openLatexFile(cipher, keyFill, ivFill, okToUseKeyBits, okToUseIvBits);
      if (texFile) {
        sendToLatexWriter(texFile, cipher, okToUseKeyBits, okToUseIvBits, (int)i + 2, mc);
        fclose(texFile);
      }
    }
#else
    if (texFileName) {
      if ((i >= 12 - 1 - sizeOfStartingOptimalSet) || (z == suppressedBytes * 8) || (i == numBits - sizeOfStartingOptimalSet - 1)) {
        FILE *texFile = fopen(texFileName, "w");
        if (texFile) {

          /* write bit set as LaTeX comment */
          if (okToUseKeyBits && okToUseIvBits) {
            logger(texFile, LOGFILE, "%% %d rounds with bit set size %d + %d = %d: Key = ", z, numKeyBits, numIvBits, numKeyBits + numIvBits);
            logBitSet(texFile, LOGFILE, keyBit, numKeyBits);
            logger(texFile, LOGFILE, ", IV = ");
            logBitSet(texFile, LOGFILE, ivBit, numIvBits);
            logger(texFile, LOGFILE, "\n");
          } else if (okToUseKeyBits) {
            logger(texFile, LOGFILE, "%% %d rounds with bit set size %d: Key = ", z, numKeyBits);
            logBitSet(texFile, LOGFILE, keyBit, numKeyBits);
            logger(texFile, LOGFILE, "\n");
          } else if (okToUseIvBits) {
            logger(texFile, LOGFILE, "%% %d rounds with bit set size %d: IV = ", z, numIvBits);
            logBitSet(texFile, LOGFILE, ivBit, numIvBits);
            logger(texFile, LOGFILE, "\n");
          }

          /* write LaTeX graph */
          sendToLatexWriter(texFile, cipher, okToUseKeyBits, okToUseIvBits, (int)i + 2, mc);
          fclose(texFile);
        }
      }
    }
#endif

    if (z == suppressedBytes * 8) break;
  }

  logger(logFile, LOGNOFLUSH, "\n[");
  logDateRaw(logFile, LOGNOFLUSH);
  logger(logFile, LOGNOFLUSH, ", ");
  logTimeRaw(logFile, LOGNOFLUSH, start);
  if (blackBoxCipherType(cipher) == kBlockCipher) {
    logger(logFile, LOGNOFLUSH, "] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f blocks = %5.2f%% ", numKeyBits + numIvBits, z, suppressedBytes * 8, z/(float)8/(float)implicitBlockSize, 100*z/(float)(suppressedBytes*8));
  } else {
    logger(logFile, LOGNOFLUSH, "] Bit set size %2d  ->  %3d / %3d zero bits = %5.2f%% ", numKeyBits + numIvBits, z, suppressedBytes * 8, 100*z/(float)8/(float)suppressedBytes);
  }
  logger(logFile, LOGALL, "\n\n\n");

  if (logFile) fclose(logFile);
  FREE(mc);
  FREE(in);
  FREE(xor);
}

void blackBoxGreedyMaxtermDistinguisherAll(int numBits, int keyFill, int ivFill, int optimalSetSize, int okToUseKeyBits, int okToUseIvBits, int strategy, int numParallellBits) {
  int i;
  starHeader("Black Box - Greedy");
  for (i=0; i<kNumBlackBoxCiphers; i++)
    blackBoxGreedyMaxtermDistinguisher((bbCipher)i, NULL, NULL, numBits, keyFill, ivFill, optimalSetSize, okToUseKeyBits, okToUseIvBits, strategy, numParallellBits);
}

