/*
 * Adapted reference code from Martin Hell and J.-P. Aumasson.
 *
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_grain.h"

//#define MAX_BITS 512
#define MAX_BITS 1024
//#define MAX_BITS 4096

/******************************************************************************
 * Grain v1
 ******************************************************************************/
typedef struct {
  UINT16 lfsr[1036];
  UINT16 nfsr[1036];
  UINT16 *nptr;
  UINT16 *lptr;
  UINT32 count;
  const BYTE *key;
  UINT32 keysize;
  UINT32 ivsize;
} GrainV1_ctx;

void GrainV1_Reinit_After_NIter(GrainV1_ctx* ctx)
{
  *(UINT64*)(ctx->lfsr)     = *(UINT64*)(ctx->lfsr+ctx->count-5);
  *(UINT16*)((ctx->lfsr)+4) = *(UINT16*)(ctx->lfsr+ctx->count-1);
  *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->nfsr+ctx->count-5);
  *(UINT16*)((ctx->nfsr)+4) = *(UINT16*)(ctx->nfsr+ctx->count-1);
  ctx->nptr=ctx->nfsr;
  ctx->lptr=ctx->lfsr;
  ctx->count=5;
}

UINT16 GrainV1_GetWord(GrainV1_ctx* ctx)
{UINT32 ln0=*(UINT32*)(ctx->lptr),
   nn0=*(UINT32*)(ctx->nptr),
   ln2=*(UINT32*)((ctx->lptr)+2),
   nn2=*(UINT32*)((ctx->nptr)+2),
   ln1=*(UINT32*)(++(ctx->lptr)),
   nn1=*(UINT32*)(++(ctx->nptr)),
   ln3=*(UINT32*)((ctx->lptr)+2),
   nn3=*(UINT32*)((ctx->nptr)+2),
   ln4=*(UINT32*)((ctx->lptr)+3);
 ctx->nfsr[ctx->count] = (UINT16)(ln0^nn0^((nn0^nn3)>>14)^(nn2>>5)^(nn3>>4)^
   ((nn3>>12)&~(nn3>>15))^
   ((nn0>>9)&(((nn1>>12)&(nn2>>13)&(nn3>>15))^(~(nn0>>15))))^
   ((nn2>>1)&~((nn2>>5)&~((nn3>>4)&(nn3>>12))))^
   ((nn1>>5)&~(((nn0&nn3)>>15)&(nn3>>12)))^
   ((nn2>>13)&~((nn3>>4)&((nn3>>12)&(~((nn3>>15)&(nn2>>5))))))^
   ((nn1>>12)&~(((nn1>>5)&(nn2>>1))&(((nn2>>5)&(nn2>>13)&(nn3>>4))^~((nn0>>9)&(nn0>>15))))));
 ctx->lfsr[ctx->count] = (UINT16)(ln0^(ln0>>13)^(ln1>>7)^(ln2>>6)^(ln3>>3)^(ln3>>14));
 ++(ctx->count);
 return (UINT16)((nn0>>1)^(nn0>>2)^(nn0>>4)^(nn0>>10)^(nn1>>15)^(nn2>>11)^(nn3>>8)^
   ((ln1>>9)&~((ln0>>3)&(ln2>>14)))^
   ((nn3>>15)&(((ln2>>14)&((ln0>>3)^(ln1>>9)^ln4))^(~ln4)))^
   (ln4&((ln0>>3)|(ln2>>14))));
}

void GrainV1_keystream_bytes(
                            GrainV1_ctx* ctx,
                            BYTE* keystream,
                            UINT32 msglen)
{
  UINT32 itr=msglen>>1,i,j=0;
  UINT32 rem=msglen&1;
  UINT16 remWord;
  for (i=0;i<itr;++i)
    {   *(UINT16*)(keystream+j) = GrainV1_GetWord(ctx);
    j+=2;
    if ((ctx->count)>=1024) GrainV1_Reinit_After_NIter(ctx);
    }
  if (rem)
    {  remWord=GrainV1_GetWord(ctx);
    for (i=0;i<rem;++i) *(keystream+j++) = (BYTE)(remWord>>(8*i));
    }
}

void GrainV1_encrypt_bytes(
                          GrainV1_ctx* ctx,
                          const BYTE* plaintext,
                          BYTE* ciphertext,
                          UINT32 msglen)
{
  UINT32 itr=msglen>>1,i,j=0;
  UINT32 rem=msglen&1;
  UINT16 remWord;
  for (i=0;i<itr;++i) {
    *((UINT16*)(ciphertext+j)) = GrainV1_GetWord(ctx) ^ *(UINT16*)(plaintext+j);
    j+=2;
    if ((ctx->count)>=1024) GrainV1_Reinit_After_NIter(ctx);
  }
  if (rem) {
    remWord = GrainV1_GetWord(ctx);
    for (i=0;i<rem;++i) {
      *((BYTE*)(ciphertext+j)) = (BYTE)(remWord>>(8*i)) ^ *(BYTE*)(plaintext+j);
      j++;
    }
  }
}


void GrainV1_dGrainV1_bytes(
                          GrainV1_ctx* ctx,
                          const BYTE* ciphertext,
                          BYTE* plaintext,
                          UINT32 msglen)
{
  UINT32 itr=msglen>>1,i,j=0;
  UINT32 rem=msglen&1;
  UINT16 remWord;
  for (i=0;i<itr;++i)
    {   *(UINT16*)(plaintext+j) = GrainV1_GetWord(ctx) ^ (*(UINT16*)(ciphertext+j));
    j+=2;
    if ((ctx->count)>=1024) GrainV1_Reinit_After_NIter(ctx);
    }
  if (rem)
    {  remWord=GrainV1_GetWord(ctx);
    for (i=0;i<rem;++i)
      {  *(plaintext+j) = ((BYTE)(remWord>>(8*i))) ^ *(ciphertext+j);
      j++;
      }
    }
}

void GrainV1_keysetup(
                     GrainV1_ctx* ctx,
                     const BYTE* key,
                     UINT32 keysize,                /* Key size in bits. */
                     UINT32 ivsize)  /* IV size in bits. */
{
  ctx->key=key;
  ctx->keysize=keysize;
  ctx->ivsize=ivsize;
}

void GrainV1_ivsetup(GrainV1_ctx* ctx, const BYTE* iv) {
 int i;
 register UINT16 pad;
 *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->key);
 *(UINT16*)((ctx->nfsr)+4) = *(UINT16*)((ctx->key)+8);
 *(UINT64*)(ctx->lfsr)     = *(UINT64*)(iv);
 *(UINT16*)((ctx->lfsr)+4) =  (UINT16)0xFFFF;
 ctx->nptr=ctx->nfsr;
 ctx->lptr=ctx->lfsr;
 ctx->count=5;
 for(i=0; i<10; ++i) {
   pad=GrainV1_GetWord(ctx);
   ctx->nfsr[i+5]^=pad;
   ctx->lfsr[i+5]^=pad;
 }
}

void GrainV1_ivsetup_withInitOutput(GrainV1_ctx* ctx, const BYTE* iv, BYTE *out) {
 int i;
 UINT16 *buf = (UINT16 *)out;
 register UINT16 pad;
 *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->key);
 *(UINT16*)((ctx->nfsr)+4) = *(UINT16*)((ctx->key)+8);
 *(UINT64*)(ctx->lfsr)     = *(UINT64*)(iv);
 *(UINT16*)((ctx->lfsr)+4) =  (UINT16)0xFFFF;
 ctx->nptr=ctx->nfsr;
 ctx->lptr=ctx->lfsr;
 ctx->count=5;
 for(i=0; i<10; ++i) {
   pad = GrainV1_GetWord(ctx);
   *buf++ ^= pad;
   ctx->nfsr[i+5]^=pad;
   ctx->lfsr[i+5]^=pad;
 }
}

/******************************************************************************
 * Grain-128
 ******************************************************************************/
typedef struct {
	UINT32 lfsr[1036];
	UINT32 nfsr[1036];
	UINT32 *nptr;
	UINT32 *lptr;
	UINT32 count;
	const BYTE *key;
	UINT32 keysize;
	UINT32 ivsize;
} Grain128_ctx;

void Grain128_Reinit_After_NIter(Grain128_ctx* ctx)
{
  *(UINT64*)(ctx->lfsr)     = *(UINT64*)(ctx->lfsr+ctx->count-4);
  *(UINT64*)((ctx->lfsr)+2) = *(UINT64*)(ctx->lfsr+ctx->count-2);
  *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->nfsr+ctx->count-4);
  *(UINT64*)((ctx->nfsr)+2) = *(UINT64*)(ctx->nfsr+ctx->count-2);
  ctx->nptr=ctx->nfsr;
  ctx->lptr=ctx->lfsr;
  ctx->count=4;
}

UINT32 Grain128_GetWord(Grain128_ctx* ctx)
{
	UINT64 ln0=*(UINT64*)(ctx->lptr),
	  nn0=*(UINT64*)(ctx->nptr),
	  ln2=*(UINT64*)((ctx->lptr)+2),
	  nn2=*(UINT64*)((ctx->nptr)+2),
	  ln1=*(UINT64*)(++(ctx->lptr)),
	  nn1=*(UINT64*)(++(ctx->nptr)),
	  ln3=*(UINT64*)((ctx->lptr)+2),
	  nn3=*(UINT64*)((ctx->nptr)+2);
	BYTE * pn = (BYTE*)(ctx->nptr);
    ctx->nfsr[ctx->count] = (UINT32)(nn0^nn3^ln0^(nn0>>26)^(*(UINT64*)(pn+3))^(((nn0&nn1)^nn2)>>27)^
			    ((*(UINT64*)(pn+1)) & (*(UINT64*)(pn+2)))^((nn0&nn2)>>3)^((nn1>>29)&(nn2>>1))^
			    ((nn0>>17)&(nn0>>18))^((nn2>>4)&(nn2>>20))^((nn0>>11)&(nn0>>13)));
	ctx->lfsr[ctx->count] = (UINT32)((ln0^ln3)^((ln1^ln2)>>6)^(ln0>>7)^(ln2>>17));
	ln3=ln1>>13;
	nn3=ln0>>7;
	ln3&=ln2;
    nn3&=ln0;
    ++(ctx->count);
	ln3^=nn0;
	nn3^=nn1;
	return  (UINT32)(nn2^(nn0>>2)^(nn2>>9)^((nn0>>12)&(ln0>>8))^((nn2>>31)&(ln1>>10))^
	      (nn3>>13)^(ln2>>29)^(ln3>>15)^(nn2>>25)^(nn1>>4)^((nn0>>12)&((nn2&ln2)>>31)));
}

void Grain128_keystream_bytes(
  Grain128_ctx* ctx,
  BYTE* keystream,
  UINT32 msglen)
{
	UINT32 itr=msglen/4,i,j=0;
	UINT32 rem=msglen%4;
	UINT32 remWord;
	for (i=0;i<itr;++i) {
     *(UINT32*)(keystream+j) = Grain128_GetWord(ctx);
		j+=4;
		if ((ctx->count)>=1024)
      Grain128_Reinit_After_NIter(ctx);
	}
	if (rem) {
    remWord=Grain128_GetWord(ctx);
    for (i=0;i<rem;++i)
      *(keystream+j++) = (BYTE)(remWord>>(8*i));
	}
}

void Grain128_encrypt_bytes(
  Grain128_ctx* ctx,
  const BYTE* plaintext,
  BYTE* ciphertext,
  UINT32 msglen)
{
	UINT32 itr=msglen/4,i,j=0;
	UINT32 rem=msglen%4;
	UINT32 remWord;
	for (i=0;i<itr;++i)	{
    *((UINT32*)(ciphertext+j)) = Grain128_GetWord(ctx) ^ *(UINT32*)(plaintext+j);
		j+=4;
		if ((ctx->count)>=1024)
      Grain128_Reinit_After_NIter(ctx);
	}
	if (rem) {
    remWord=Grain128_GetWord(ctx);
    for (i=0;i<rem;++i) {
      *((BYTE*)(ciphertext+j)) = (BYTE)(remWord>>(8*i)) ^ *(BYTE*)(plaintext+j);
      j++;
    }
	}
}

void Grain128_keysetup(
                     Grain128_ctx* ctx,
                     const BYTE* key,
                     UINT32 keysize,
                     UINT32 ivsize)
{
  ctx->key=key;
  ctx->keysize=keysize;
  ctx->ivsize=ivsize;
}

void Grain128_ivsetup(
                    Grain128_ctx* ctx,
                    const BYTE* iv)
{   int i;
 register UINT32 pad;
 *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->key);
 *(UINT64*)((ctx->nfsr)+2) = *(UINT64*)((ctx->key)+8);
 *(UINT64*)(ctx->lfsr)     = *(UINT64*)(iv);
 *(UINT32*)((ctx->lfsr)+2) = *(UINT32*)(iv+8);
 *(UINT32*)((ctx->lfsr)+3) =  (UINT32)0xFFFFFFFF;
 ctx->nptr=ctx->nfsr;
 ctx->lptr=ctx->lfsr;
 ctx->count=4;
 for(i=0; i<8; ++i)
   { pad=Grain128_GetWord(ctx);
   ctx->nfsr[i+4]^=pad;
   ctx->lfsr[i+4]^=pad;
   }
}

void Grain128_ivsetup_withInitOutput(
                    Grain128_ctx* ctx,
                    const BYTE* iv,
                    BYTE *out)
{
  int i;
  UINT32 *buf = (UINT32 *)out;
  register UINT32 pad;
  *(UINT64*)(ctx->nfsr)     = *(UINT64*)(ctx->key);
  *(UINT64*)((ctx->nfsr)+2) = *(UINT64*)((ctx->key)+8);
  *(UINT64*)(ctx->lfsr)     = *(UINT64*)(iv);
  *(UINT32*)((ctx->lfsr)+2) = *(UINT32*)(iv+8);
  *(UINT32*)((ctx->lfsr)+3) =  (UINT32)0xFFFFFFFF;
  ctx->nptr=ctx->nfsr;
  ctx->lptr=ctx->lfsr;
  ctx->count=4;
  for(i=0; i<8; ++i) {
    pad=Grain128_GetWord(ctx);
    *buf++ ^= pad;
    ctx->nfsr[i+4]^=pad;
    ctx->lfsr[i+4]^=pad;
  }
}


/******************************************************************************
 * Black box variants
 ******************************************************************************/
int grain_v1_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  GrainV1_ctx ctx;
  if (numInputBytes < numOutputBytes) return -1;
  GrainV1_keysetup(&ctx, key, 80, 64);
  GrainV1_ivsetup(&ctx, iv);
  GrainV1_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

int grain_v1_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  GrainV1_ctx ctx;
  if (numInputBytes + 20 < numOutputBytes) return -1;
  GrainV1_keysetup(&ctx, key, 80, 64);
  GrainV1_ivsetup_withInitOutput(&ctx, iv, outBuf); outBuf += 20; numOutputBytes -= 20;
  GrainV1_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

int grain128_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  Grain128_ctx ctx;
  if (numInputBytes < numOutputBytes) return -1;
  Grain128_keysetup(&ctx, key, 128, 96);
  Grain128_ivsetup(&ctx, iv);
  Grain128_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

int grain128_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  Grain128_ctx ctx;
  if (numInputBytes + 32 < numOutputBytes) return -1;
  Grain128_keysetup(&ctx, key, 128, 96);
  Grain128_ivsetup_withInitOutput(&ctx, iv, outBuf); outBuf += 32; numOutputBytes -= 32;
  Grain128_encrypt_bytes(&ctx, inBuf, outBuf, numOutputBytes);
  return 0;
}

int grain128Bitsliced_xor(const UINT64 *key, const UINT64 *iv, const UINT64 *inBuf, unsigned int numInputBits, UINT64 *outBuf, unsigned int numOutputBits) {
  UINT64 l[128 + 256 + MAX_BITS], n[128 + 256 + MAX_BITS];
  unsigned int i;

  if (numInputBits < numOutputBits) return -1;
  if (numOutputBits > MAX_BITS) return -1;

  /* load key and iv */
  for(i=0; i<96; i++){ n[i] = key[i]; l[i] = iv[i]; }
  for(i=96; i<128; i++){ n[i] = key[i]; l[i]= 0xFFFFFFFFFFFFFFFF; }

  /* suppressed bits */
  for(i=0; i<256; i++){
    UINT64 z;
    l[i+128] = l[i] ^ l[i+7] ^ l[i+38] ^ l[i+70] ^ l[i+81] ^ l[i+96];
    n[i+128] = l[i] ^ n[i] ^ n[i+26] ^ n[i+56] ^ n[i+91] ^ n[i+96] ^ (n[i+ 3] & n[i+67]) ^ (n[i+11] & n[i+13]) ^ (n[i+17] & n[i+18]) ^ (n[i+27] & n[i+59]) ^ (n[i+40] & n[i+48]) ^ (n[i+61] & n[i+65]) ^ (n[i+68] & n[i+84]);
    z = (n[i+12] & l[i+8]) ^ (l[i+13] & l[i+20]) ^ (n[i+95] & l[i+42]) ^ (l[i+60] & l[i+79]) ^ (n[i+12] & n[i+95] & l[i+95]) ^ n[i + 2] ^ n[i + 15] ^ n[i + 36] ^ n[i + 45] ^ n[i + 64] ^ n[i + 73] ^ n[i + 89] ^ l[i + 93];
    l[i+128] ^= z;
    n[i+128] ^= z;
  }

  /* keystream */
  for(i=256; i<256+numOutputBits; i++){
    l[i+128] = l[i] ^ l[i+7] ^ l[i+38] ^ l[i+70] ^ l[i+81] ^ l[i+96];
    n[i+128] = l[i] ^ n[i] ^ n[i+26] ^ n[i+56] ^ n[i+91] ^ n[i+96] ^ (n[i+ 3] & n[i+67]) ^ (n[i+11] & n[i+13]) ^ (n[i+17] & n[i+18]) ^ (n[i+27] & n[i+59]) ^ (n[i+40] & n[i+48]) ^ (n[i+61] & n[i+65]) ^ (n[i+68] & n[i+84]);
    *outBuf++ = *inBuf++ ^ (n[i+12] & l[i+8]) ^ (l[i+13] & l[i+20]) ^ (n[i+95] & l[i+42]) ^ (l[i+60] & l[i+79]) ^ (n[i+12] & n[i+95] & l[i+95]) ^ n[i + 2] ^ n[i + 15] ^ n[i + 36] ^ n[i + 45] ^ n[i + 64] ^ n[i + 73] ^ n[i + 89] ^ l[i + 93];
  }

  return 0;
}

int grain128Bitsliced_xor_withInitOutput(const UINT64 *key, const UINT64 *iv, const UINT64 *inBuf, unsigned int numInputBits, UINT64 *outBuf, unsigned int numOutputBits) {
  UINT64 l[128 + MAX_BITS], n[128 + MAX_BITS];
  unsigned int i;

  if (numInputBits + 256 < numOutputBits) return -1;
  if (numOutputBits > MAX_BITS) return -1;

  /* load key and iv */
  for(i=0; i<96; i++){ n[i] = key[i]; l[i] = iv[i]; }
  for(i=96; i<128; i++){ n[i] = key[i]; l[i]= 0xFFFFFFFFFFFFFFFF; }

  /* suppressed bits */
  for(i=0; i<256; i++){
    UINT64 z;
    l[i+128] = l[i] ^ l[i+7] ^ l[i+38] ^ l[i+70] ^ l[i+81] ^ l[i+96];
    n[i+128] = l[i] ^ n[i] ^ n[i+26] ^ n[i+56] ^ n[i+91] ^ n[i+96] ^ (n[i+ 3] & n[i+67]) ^ (n[i+11] & n[i+13]) ^ (n[i+17] & n[i+18]) ^ (n[i+27] & n[i+59]) ^ (n[i+40] & n[i+48]) ^ (n[i+61] & n[i+65]) ^ (n[i+68] & n[i+84]);
    z = (n[i+12] & l[i+8]) ^ (l[i+13] & l[i+20]) ^ (n[i+95] & l[i+42]) ^ (l[i+60] & l[i+79]) ^ (n[i+12] & n[i+95] & l[i+95]) ^ n[i + 2] ^ n[i + 15] ^ n[i + 36] ^ n[i + 45] ^ n[i + 64] ^ n[i + 73] ^ n[i + 89] ^ l[i + 93];
    *outBuf++ ^= z;
    l[i+128] ^= z;
    n[i+128] ^= z;
  }
  numOutputBits -= 256;

  /* keystream */
  for(i=256; i<256+numOutputBits; i++){
    l[i+128] = l[i] ^ l[i+7] ^ l[i+38] ^ l[i+70] ^ l[i+81] ^ l[i+96];
    n[i+128] = l[i] ^ n[i] ^ n[i+26] ^ n[i+56] ^ n[i+91] ^ n[i+96] ^ (n[i+ 3] & n[i+67]) ^ (n[i+11] & n[i+13]) ^ (n[i+17] & n[i+18]) ^ (n[i+27] & n[i+59]) ^ (n[i+40] & n[i+48]) ^ (n[i+61] & n[i+65]) ^ (n[i+68] & n[i+84]);
    *outBuf++ = *inBuf++ ^ (n[i+12] & l[i+8]) ^ (l[i+13] & l[i+20]) ^ (n[i+95] & l[i+42]) ^ (l[i+60] & l[i+79]) ^ (n[i+12] & n[i+95] & l[i+95]) ^ n[i + 2] ^ n[i + 15] ^ n[i + 36] ^ n[i + 45] ^ n[i + 64] ^ n[i + 73] ^ n[i + 89] ^ l[i + 93];
  }

  return 0;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
int blackBoxGrainV1Encryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  if (withInitRoundOutput)
    return grain_v1_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return grain_v1_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

int blackBoxGrain128Encryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  int res;

  if (withInitRoundOutput)
    res = grain128_xor_withInitOutput(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  else
    res = grain128_xor(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes);

  return res;
}

int blackBoxGrain128EncryptionBitsliced(const UINT64 *key, const UINT64 *iv, const UINT64 *inBuf, unsigned int numInputBits, UINT64 *outBuf, unsigned int numOutputBits, int withInitRoundOutput) {
  int res;

  if (withInitRoundOutput)
    res = grain128Bitsliced_xor_withInitOutput(key, iv, inBuf, numInputBits, outBuf, numOutputBits);
  else
    res = grain128Bitsliced_xor(key, iv, inBuf, numInputBits, outBuf, numOutputBits);

  return res;
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxGrainV1Info(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 10;
  if (ivSizeInBytes) *ivSizeInBytes = 8;
  if (suppressedBytes) *suppressedBytes = 20;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = 1;
}

void getBlackBoxGrain128Info(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 16;
  if (ivSizeInBytes) *ivSizeInBytes = 12;
  if (suppressedBytes) *suppressedBytes = 32;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = 1;
}

void getBlackBoxBitslicedGrain128Info(int *keySizeInBits, int *ivSizeInBits, int *suppressedBits, int *implicitBlockSizeInBits) {
  if (keySizeInBits) *keySizeInBits = 128;
  if (ivSizeInBits) *ivSizeInBits = 96;
  if (suppressedBits) *suppressedBits = 256;
  if (implicitBlockSizeInBits) *implicitBlockSizeInBits = 1;
}

