/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_first_test.h"
#include "black_box_utils.h"
#include "bitslice_utils.h"
#include "assert_utils.h"
#include "black_box_bit_set_utils.h"
#include "memory_utils.h"
#include <stdio.h> /* printf */

#define STANDARD  1
#define BITSLICED 2

typedef struct {
  bbCipher cipher;
  int applicability;
  const BYTE *key;
  const BYTE *iv;
  const BYTE *pt;
  int ptLen;
  const BYTE *ct;
  int ctLen;
} katType; /* Known Answer Tests */

static const katType kat[] = {
  { kTrivium, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 32, (const BYTE *)"\xFB\xE0\xBF\x26\x58\x59\x05\x1B\x51\x7A\x2E\x4E\x23\x9F\xC9\x7F\x56\x32\x03\x16\x19\x07\xCF\x2D\xE7\xA8\x79\x0F\xA1\xB2\xE9\xCD", 32 },
  { kTrivium, BITSLICED, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 36, (const BYTE *)"\xFB\xE0\xBF\x26\x58\x59\x05\x1B\x51\x7A\x2E\x4E\x23\x9F\xC9\x7F\x56\x32\x03\x16\x19\x07\xCF\x2D\xE7\xA8\x79\x0F\xA1\xB2\xE9\xCD\xF7\x52\x92\x03", 36 },
  { kTrivium, STANDARD, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9\x60\xC7", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 32, (const BYTE *)"\xA4\x38\x6C\x6D\x76\x24\x98\x3F\xEA\x8D\xBE\x73\x14\xE5\xFE\x1F\x9D\x10\x20\x04\xC2\xCE\xC9\x9A\xC3\xBF\xBF\x00\x3A\x66\x43\x3F", 32 },
  { kTrivium, BITSLICED, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9\x60\xC7", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 36, (const BYTE *)"\xA4\x38\x6C\x6D\x76\x24\x98\x3F\xEA\x8D\xBE\x73\x14\xE5\xFE\x1F\x9D\x10\x20\x04\xC2\xCE\xC9\x9A\xC3\xBF\xBF\x00\x3A\x66\x43\x3F\x30\x89\xA9\x8F", 36 },
  { kRabbit, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 32, (const BYTE *)"\xED\xB7\x05\x67\x37\x5D\xCD\x7C\xD8\x95\x54\xF8\x5E\x27\xA7\xC6\x8D\x4A\xDC\x70\x32\x29\x8F\x7B\xD4\xEF\xF5\x04\xAC\xA6\x29\x5F", 32 },
  { kEdon80, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 32, (const BYTE *)"\x45\xF3\xAD\x99\x37\x50\x00\xDC\xDB\xAD\x40\xC4\x55\xBD\x6B\xE4\x1E\x52\x7C\x68\x89\x6E\x16\xC8\xD9\x46\x73\xCA\x4A\xD4\x63\x01", 32 },
  { kAES128, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x66\xE9\x4B\xD4\xEF\x8A\x2C\x3B\x88\x4C\xFA\x59\xCA\x34\x2B\x2E", 16 },
  { kAES256, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 32, (const BYTE *)"\xDC\x95\xC0\x78\xA2\x40\x89\x89\xAD\x48\xA2\x14\x92\x84\x20\x87\x08\xC3\x74\x84\x8C\x22\x82\x33\xC2\xB3\x4F\x33\x2B\xD2\xE9\xD3", 32 },
  { kDES, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\x8C\xA6\x4D\xE9\xC1\xB1\x23\xA7", 8 },
  { kGrainV1, STANDARD, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x01\x7D\x13\xEC\xB2\x0A\xE0\xC9\xAC\xF7\x84\xCB\x06\x52\x5F\x72", 16 },
  { kGrain128, STANDARD | BITSLICED, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\xF0\x9B\x7B\xF7\xD7\xF6\xB5\xC2\xDE\x2F\xFC\x73\xAC\x21\x39\x7F", 16 },
  { kGrain128, STANDARD | BITSLICED, (const BYTE *)"\x01\x23\x45\x67\x89\xAB\xCD\xEF\x12\x34\x56\x78\x9A\xBC\xDE\xF0", (const BYTE *)"\x01\x23\x45\x67\x89\xAB\xCD\xEF\x12\x34\x56\x78", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\xAF\xB5\xBA\xBF\xA8\xDE\x89\x6B\x4B\x9C\x6A\xCA\xF7\xC4\xFB\xFD", 16 },
  { kTEA, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\x41\xea\x3a\x0a\x94\xba\xa9\x40", 8 },
  { kXTEA, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\xde\xe9\xd4\xd8\xf7\x13\x1e\xd9", 8 },
  { kSEED, STANDARD, (const BYTE *)"\x28\xDB\xC3\xBC\x49\xFF\xD8\x7D\xCF\xA5\x09\xB1\x1D\x42\x2B\xE7", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\xB4\x1E\x6B\xE2\xEB\xA8\x4A\x14\x8E\x2E\xED\x84\x59\x3C\x5E\xC7", 16, (const BYTE *)"\x9B\x9B\x7B\xFC\xD1\x81\x3C\xB9\x5D\x0B\x36\x18\xF4\x0F\x51\x22", 16 },
  { kPRESENT, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\x55\x79\xC1\x38\x7B\x22\x84\x45", 8 },
  { kSMS4, STANDARD, (const BYTE *)"\x01\x23\x45\x67\x89\xab\xcd\xef\xfe\xdc\xba\x98\x76\x54\x32\x10", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x01\x23\x45\x67\x89\xab\xcd\xef\xfe\xdc\xba\x98\x76\x54\x32\x10", 16, (const BYTE *)"\x68\x1e\xdf\x34\xd2\x06\x96\x5e\x86\xb3\xe9\x4f\x53\x6e\x42\x46", 16 },
  { kCamellia, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x07\x92\x3A\x39\xEB\x0A\x81\x7D\x1C\x4D\x87\xBD\xB8\x2D\x1F\x1C", 16 },
  { kRC5, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\x21\xA5\xDB\xEE\x15\x4B\x8F\x6D", 8 },
  { kRC6, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x8F\xC3\xA5\x36\x56\xB1\xF7\x78\xC1\x29\xDF\x4E\x98\x48\xA4\x1E", 16 },
#if 0
  /* There seems to be only one test vector in existence for HIGHT and this was
   * received directly from Jaechul Sung. By reverse calculation, this test
   * vector is derived from a slightly altered version of HIGHT, however.
   * To pass this test, define SUNG_TEST_VECTOR_VERSION in black_box_hight.c.
   * By not defining SUNG_TEST_VECTOR_VERSION, the algorithm is implemented
   * according to our interpretation of the specification (for which the test
   * vector fails).
   */
  { kHIGHT, STANDARD, (const BYTE *)"\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xaa\xbb\xcc\xdd\xee\xff", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x80\x00\x00\x00\x00\x00\x00\x00", 8, (const BYTE *)"\xf2\x03\x4f\xd9\xae\x18\xf4\x00", 8 },
#endif
//  { kCLEFIA, STANDARD, (const BYTE *)"\xff\xee\xdd\xcc\xbb\xaa\x99\x88\x77\x66\x55\x44\x33\x22\x11\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f", 16, (const BYTE *)"\xde\x2b\xf2\xfd\x9b\x74\xaa\xcd\xf1\x29\x85\x55\x45\x94\x94\xfd", 16 },
  { kHC128, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x82\x00\x15\x73\xA0\x03\xFD\x3B\x7F\xD7\x2F\xFB\x0E\xAF\x63\xAA", 16 },
  { kHC128, STANDARD, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA\x4D\xA0\xF3\x46\x99\xEC", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9\x60\xC7\x2E\x95\xFC\x63\xCA\x31", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x1C\xD8\xAE\xDD\xFE\x52\xE2\x17\xE8\x35\xD0\xB7\xE8\x4E\x29\x22", 16 },
  { kHC256, STANDARD, (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x5B\x07\x89\x85\xD8\xF6\xF3\x0D\x42\xC5\xC0\x2F\xA6\xB6\x79\x51", 16 },
  { kHC256, STANDARD, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA\x4D\xA0\xF3\x46\x99\xEC\x3F\x92\xE5\x38\x8B\xDE\x31\x84\xD7\x2A\x7D\xD0\x23\x76\xC9\x1C", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9\x60\xC7\x2E\x95\xFC\x63\xCA\x31\x98\xFF\x66\xCD\x34\x9B\x02\x69\xD0\x37\x9E\x05\x6C\xD3\x3A\xA1", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\xC8\x63\x20\x38\xDA\x61\x67\x9C\x46\x85\x28\x8B\x37\xD3\xE2\x32", 16 },
  { kMickeyV2, STANDARD, (const BYTE *)"\xf1\x1a\x56\x27\xce\x43\xb6\x1f\x89\x12", (const BYTE *)"\x9c\x53\x2f\x8a\xc3\xea\x4b\x2e\xa0\xf5", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\x21\xa0\x43\x66\x19\xcb\x9f\x3f\x6f\x1f\xb3\x03\xf5\x6a\x09\xa9", 16 },
  { kSalsa2012, STANDARD, (const BYTE *)"\x0F\x62\xB5\x08\x5B\xAE\x01\x54\xA7\xFA\x4D\xA0\xF3\x46\x99\xEC", (const BYTE *)"\x28\x8F\xF6\x5D\xC4\x2B\x92\xF9", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\xD2\x4D\x58\xB3\x3A\xFC\xCA\x12\x7A\xE5\x33\x29\xE4\x16\x61\xD3", 16 },
  { kSOSEMANUK, STANDARD, (const BYTE *)"\x00\x11\x22\x33\x44\x55\x66\x77\x88\x99\xAA\xBB\xCC\xDD\xEE\xFF", (const BYTE *)"\x88\x99\xAA\xBB\xCC\xDD\xEE\xFF\x00\x11\x22\x33\x44\x55\x66\x77", (const BYTE *)"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00", 16, (const BYTE *)"\xFA\x61\xDB\xEB\x71\x17\x81\x31\xA7\x7C\x71\x4B\xD2\xEA\xBF\x4E", 16 }
};

static void logInitBuf(const BYTE *buf, int numInitBytes, int totalBytes, int blockSize) {
  if (numInitBytes > 0) logBuf(NULL, LOGSCREEN, buf, numInitBytes, 0, 0);
  ASSERT(totalBytes > numInitBytes, "Unexpected cryption data length!\n");
  logBuf(NULL, LOGSCREEN, buf + numInitBytes, totalBytes - numInitBytes, blockSize, 0);
}

static void logBitslicedInitBuf(const UINT64 *buf, int numInitBits, int totalBits, int blockSize) {
  BYTE *buf8 = (BYTE*)MALLOC((totalBits + 7) / 8 * sizeof(BYTE));
  fromBitslicedBufByLSB(buf8, buf, totalBits);
  logInitBuf(buf8, (numInitBits + 7) / 8, (totalBits + 7) / 8, (blockSize + 7) / 8);
  FREE(buf8);
}

static int performKatTestStandard(bbCipher cipher, const char *name, int keySize, int ivSize, int suppressedBytes, int implicitBlockSize, const katType *kat) {
  BYTE *out;
  int outLen;
  int res, test1ok = 0, test2ok = 0, test3ok = 0;
  const UINT64 marker = U64C(0xABBAACDCABBAACDC);

  printf("Key:"); logBuf(NULL, LOGSCREEN, kat->key, keySize, 0, 0); printf("\n");
  printf("IV :"); logBuf(NULL, LOGSCREEN, kat->iv, ivSize, 0, 0); printf("\n");

  outLen = suppressedBytes + kat->ctLen;
  out = (BYTE*)MALLOC((outLen + sizeof(marker)) * sizeof(BYTE));
  MEMSET(out, 0, outLen);
  *((UINT64*)(out + suppressedBytes + kat->ctLen)) = marker;

  /* encrypt with initial round output */
  res = blackBoxEncrypt(cipher, kat->key, kat->iv, kat->pt, kat->ptLen, out, outLen, 1 /* with init round output */);
  printf("%s encryption with hidden output (%d + %d bytes):", name, suppressedBytes, kat->ctLen);
  if (!res) {
    test1ok = !memcmp(out + suppressedBytes, kat->ct, kat->ctLen);
    logInitBuf(out, suppressedBytes, outLen, implicitBlockSize);
  } else {
    printf(" *** Error (returned %d) ***", res);
  }
  printf("\n");

  /* encrypt without initial round output */
  MEMSET(out, 0, outLen);
  outLen = kat->ctLen;
  res = blackBoxEncrypt(cipher, kat->key, kat->iv, kat->pt, kat->ptLen, out, outLen, 0 /* with init round output */);
  printf("%s encryption without hidden output (%d bytes):", name, kat->ctLen);
  if (!res) {
    test2ok = !memcmp(out, kat->ct, kat->ctLen);
    logBuf(NULL, LOGSCREEN, out, kat->ctLen, implicitBlockSize, 0);
  } else {
    printf(" *** Error (returned %d) ***", res);
  }
  printf("\n");

  { /* print kat cipher text */
    int j;

    printf("Test vector: ");
    for (j=0; j<kat->ctLen; j++) {
      printf("%02X", kat->ct[j]);
      if ((j+1) % implicitBlockSize == 0)
        printf(" ");
    }
    printf("\n");
  }
  test3ok = *((UINT64*)(out + suppressedBytes + kat->ctLen)) == marker;
  printf("Output buffer overwrite test: %s\n", test3ok ? "OK" : "FAILED!!!"); /* buffer overwrite test */
  printf("KAT Test %s\n\n", (test1ok && test2ok && test3ok) ? "OK" : "FAILED!!!");

  FREE(out);
  return !(test1ok && test2ok && test3ok);
}

static int performBufferXorTestStandard(bbCipher cipher) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  int inLen, outLen;
  BYTE *key, *iv, *in, *out, *out1, *out2;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);

  printf("XOR encryption data into non-empty output buffer test: ");

  key = (BYTE*)MALLOC(keySize); MEMRND(key, keySize);
  iv  = (BYTE*)MALLOC(ivSize);  MEMRND(iv, ivSize);
//  printf("Key:"); logBuf(NULL, key, keySize, 0, 0); printf("\n");
//  printf("IV :"); logBuf(NULL, iv, ivSize, 0, 0); printf("\n");

  outLen = suppressedBytes + 2 * implicitBlockSize;
  out  = (BYTE*)MALLOC(outLen); MEMRND(out, outLen);
  out1 = (BYTE*)MALLOC(outLen); MEMSET(out1, 0, outLen);
  out2 = (BYTE*)MALLOC(outLen); MEMCPY(out2, out, outLen);

  inLen = 2 * implicitBlockSize;
  in  = (BYTE*)MALLOC(inLen); MEMRND(in, inLen);

  /* encrypt with initial round output into zeroized output buffer */
  if (blackBoxEncrypt(cipher, key, iv, in, inLen, out1, outLen, 1 /* with init round output */)) {
    FREE(key); FREE(iv); FREE(in); FREE(out); FREE(out1); FREE(out2);
    printf("FAILED (at first encryption)!!!\n\n");
    return 1;
  }

  /* encrypt with initial round output into non-empty output buffer */
  if (blackBoxEncrypt(cipher, key, iv, in, inLen, out2, outLen, 1 /* with init round output */)) {
    FREE(key); FREE(iv); FREE(in); FREE(out); FREE(out1); FREE(out2);
    printf("FAILED (at second encryption)!!!\n\n");
    return 1;
  }

  MEMXOR(out1, out, outLen); /* xor original output buffer afterwards */

  if (memcmp(out1, out2, inLen)) {
    FREE(key); FREE(iv); FREE(in); FREE(out); FREE(out1); FREE(out2);
    printf("FAILED!!!\n\n");
    return 1;
  }

  printf("OK\n\n");
  FREE(key); FREE(iv); FREE(in); FREE(out); FREE(out1); FREE(out2);
  return 0;
}

#if 0
#define NUM_KEY_BITS 5
#define NUM_IV_BITS  5
static int threadingTest(bbCipher cipher) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  const char *name = blackBoxCipherName(cipher);
#if (NUM_KEY_BITS > 0)
  int keyBit[NUM_KEY_BITS];
#else
  int *keyBit = NULL;
#endif
#if (NUM_IV_BITS > 0)
  int ivBit[NUM_IV_BITS];
#else
  int *ivBit = NULL;
#endif
  BYTE *key, *iv, *in;
  BYTE *xor1, *xor2;
  int i;

  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);

  printf("Threading test: ");

  /* initialize bit sets */
  ASSERT(keySize * 8 >= NUM_KEY_BITS, "Unexpected key size!\n");
  ASSERT(ivSize * 8 >= NUM_IV_BITS, "Unexpected iv size!\n");
  for (i=0; i<NUM_KEY_BITS; i++) keyBit[i] = i;
  for (i=0; i<NUM_IV_BITS; i++) ivBit[i] = i;

  /* allocate and initialize buffers */
  key  = (BYTE*)MALLOC(keySize);           MEMRND(key, keySize);
  iv   = (BYTE*)MALLOC(ivSize);            MEMRND(iv, ivSize);
  in   = (BYTE*)MALLOC(implicitBlockSize); MEMSET(in, 0, implicitBlockSize);
  xor1 = (BYTE*)MALLOC(suppressedBytes);   MEMSET(xor1, 0, suppressedBytes);
  xor2 = (BYTE*)MALLOC(suppressedBytes);   MEMSET(xor2, 0, suppressedBytes);

  /* compute xor single threaded */
  xorOverBitSet(cipher, key, NUM_KEY_BITS, keyBit, iv, NUM_IV_BITS, ivBit, in, implicitBlockSize, xor1, 0);

  /* compute xor multi-threaded */
  xorOverBitSet(cipher, key, NUM_KEY_BITS, keyBit, iv, NUM_IV_BITS, ivBit, in, implicitBlockSize, xor2, 2);

  /* results should match */
  if (memcmp(xor1, xor2, suppressedBytes)) {
    FREE(key); FREE(iv); FREE(in); FREE(xor1); FREE(xor2);
    printf("FAILED!!!\n");
    printf("This means that the implementation of %s is not thread safe.\n", name);
    printf("A cipher implementation may be called by multiple threads, which\n"
           "means that the implementation must take into account that writes\n"
           "to the output buffer may need special care by, for example, using\n"
           "tMEMXOR (thread-safe memory xor from tMALLOC.h).\n\n");
    return 1;
  }

  FREE(key); FREE(iv); FREE(in); FREE(xor1); FREE(xor2);
  printf("OK\n\n");
  return 0;
}
#endif

int blackBoxFirstTestStandard(bbCipher cipher) {
  int keySize, ivSize, suppressedBytes, implicitBlockSize;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  int i;
  int foundError = 0;

  stars(nameLen + 12);
  printf("\n* Testing %s *\n", name);
  stars(nameLen + 12);
  printf("\n\n");

  if (!blackBoxCipherProvidesStandardImplementation(cipher)) {
    printf("No standard implementation for %s\n\n\n", name);
    return 0;
  }
  blackBoxInfo(cipher, &keySize, &ivSize, &suppressedBytes, &implicitBlockSize);

  printf("Key size                          : %4d bits / %3d bytes\n", keySize * 8, keySize);
  printf("IV size                           : %4d bits / %3d bytes\n", ivSize * 8, ivSize);
  printf("Implicit block size               : %4d bits / %3d byte%s\n", implicitBlockSize * 8, implicitBlockSize, implicitBlockSize == 1 ? " " : "s");
  printf("Suppressed during Key and IV setup: %4d bits / %3d bytes / %d blocks\n\n", suppressedBytes * 8, suppressedBytes, (suppressedBytes + implicitBlockSize - 1) / implicitBlockSize);

  /* kat tests */
  for (i=0; i<sizeof(kat)/sizeof(katType); i++)
    if ((kat[i].cipher == cipher) && (kat[i].applicability & STANDARD))
      foundError = performKatTestStandard(cipher, name, keySize, ivSize, suppressedBytes, implicitBlockSize, &kat[i]) || foundError;

  /* xor test */
  foundError = performBufferXorTestStandard(cipher) || foundError;

  /* threading test */
//  threadingTest(cipher);

  /* negative tests */
  for (i=0; i<sizeof(kat)/sizeof(katType); i++) {
    if (kat[i].cipher == cipher) { /* if cipher has a test case */
      const UINT64 marker = U64C(0xABBAACDCABBAACDC);
      BYTE *out = (BYTE*)MALLOC((suppressedBytes + implicitBlockSize + sizeof(marker)) * sizeof(BYTE));
      int res;

      printf("Negative tests:\n");

      printf("Error returned when insufficient plaintext (with initial round output): ");
      res = blackBoxEncrypt(cipher, kat[i].key, kat[i].iv, kat[i].pt, 0, out, suppressedBytes + implicitBlockSize, 1 /* with init round output */);
      if (!res) foundError = 1;
      printf("%s\n", res ? "OK" : "NOT OK!!!");

      printf("Error returned when insufficient plaintext (no initial round output): ");
      res = blackBoxEncrypt(cipher, kat[i].key, kat[i].iv, kat[i].pt, 0, out, implicitBlockSize, 0 /* no init round output */);
      if (!res) foundError = 1;
      printf("%s\n", res ? "OK" : "NOT OK!!!");

      MEMSET(out, 0, suppressedBytes);
      *((UINT64*)(out + suppressedBytes)) = marker;
      printf("Buffer overwrite check with hidden bytes only: ");
      res = blackBoxEncrypt(cipher, kat[i].key, kat[i].iv, kat[i].pt, kat[i].ptLen, out, suppressedBytes, 1 /* with init round output */);
      if (!res) {
        res = *((UINT64*)(out + suppressedBytes)) != marker;
      }
      if (res) foundError = 1;
      FREE(out);
      printf("%s\n\n\n", res == 0 ? "OK" : "NOT OK!!!");

      break; /* do this only once */
    }
  }

  return foundError;
}

static int performKatTestBitsliced(bbCipher cipher, const char *name, int keySize, int ivSize, int suppressedBits, int implicitBlockSize, const katType *kat) {
  UINT64 *key, *iv, *pt, *ct, *out;
  int outLen, res, test1ok = 0, test2ok = 0, test3ok = 0;
  const UINT64 marker = U64C(0xABBAACDCABBAACDC);

  printf("Key:"); logBuf(NULL, LOGSCREEN, kat->key, keySize / 8, 0, 0); printf("\n");
  printf("IV :"); logBuf(NULL, LOGSCREEN, kat->iv, ivSize / 8, 0, 0); printf("\n");

  key = (UINT64*)MALLOC(keySize * sizeof(UINT64)); ASSERT(key, "Allocation failed!\n");
  iv = (UINT64*)MALLOC(ivSize * sizeof(UINT64)); ASSERT(iv, "Allocation failed!\n");
  pt = (UINT64*)MALLOC(kat->ptLen * 8 * sizeof(UINT64)); ASSERT(pt, "Allocation failed!\n");
  ct = (UINT64*)MALLOC(kat->ctLen * 8 * sizeof(UINT64)); ASSERT(ct, "Allocation failed!\n");
  outLen = suppressedBits + kat->ctLen * 8;
  out = (UINT64*)MALLOC((outLen + 1) * sizeof(UINT64)); ASSERT(out, "Allocation failed!\n");
  MEMSET(out, 0, outLen * sizeof(UINT64));
  out[outLen] = marker;

  toBitslicedBuf(key, kat->key, keySize);
  toBitslicedBuf(iv, kat->iv, ivSize);
  toBitslicedBuf(pt, kat->pt, kat->ptLen * 8);
  toBitslicedBuf(ct, kat->ct, kat->ctLen * 8);

  /* encrypt with initial round output */
  res = blackBoxEncryptBitsliced(cipher, key, iv, pt, kat->ptLen * 8, out, outLen, 1 /* with init round output */);
  if (out[outLen] != marker)
    printf("*** Garbaging at end of out buffer!!! ***\n");
  else
    printf("No garbaging at end of out buffer\n");

  printf("%s encryption with hidden output (%d + %d bits):", name, suppressedBits, kat->ctLen * 8);
  if (!res) {
    test1ok = !memcmp(out + suppressedBits, ct, kat->ctLen * 8 * sizeof(UINT64));
    logBitslicedInitBuf(out, suppressedBits, outLen, implicitBlockSize);
  } else {
    printf(" *** Error (returned %d) ***", res);
  }
  printf("\n");

  /* encrypt without initial round output */
  MEMSET(out, 0, outLen * sizeof(UINT64));
  outLen = kat->ctLen * 8;
  res = blackBoxEncryptBitsliced(cipher, key, iv, pt, kat->ptLen * 8, out, outLen, 0 /* with init round output */);
  printf("%s encryption without hidden output (%d bytes):", name, kat->ctLen * 8);
  if (!res) {
    test2ok = !memcmp(out, ct, kat->ctLen * 8 * sizeof(UINT64));
    logBitslicedInitBuf(out, 0, outLen, implicitBlockSize);
  } else {
    printf(" *** Error (returned %d) ***", res);
  }
  printf("\n");

  { /* print kat cipher text */
    int j;

    printf("Test vector: ");
    for (j=0; j<kat->ctLen; j++) {
      printf("%02X", kat->ct[j]);
      if ((j+1) % implicitBlockSize == 0)
        printf(" ");
    }
    printf("\n");
  }
  test3ok = out[suppressedBits + kat->ctLen * 8] == marker;
  printf("Output buffer overwrite test: %s\n", test3ok ? "OK" : "FAILED!!!"); /* buffer overwrite test */
  printf("KAT Test %s\n\n", (test1ok && test2ok && test3ok) ? "OK" : "FAILED!!!");

  FREE(out);
  FREE(ct);
  FREE(pt);
  FREE(iv);
  FREE(key);

  return !(test1ok && test2ok && test3ok);
}

int blackBoxFirstTestBitsliced(bbCipher cipher) {
  int keySize, ivSize, suppressedBits, implicitBlockSize;
  const char *name = blackBoxCipherName(cipher);
  int nameLen = strlen(name);
  int i, foundError = 0;

  stars(nameLen + 22);
  printf("\n* Testing Bitsliced %s *\n", name);
  stars(nameLen + 22);
  printf("\n\n");

  if (!blackBoxCipherProvidesBitslicedImplementation(cipher)) {
    printf("No bitsliced implementation for %s\n\n\n", name);
    return 0;
  }
  blackBoxInfoBitsliced(cipher, &keySize, &ivSize, &suppressedBits, &implicitBlockSize);

  printf("Key size                          : %4d bits / %3d bytes\n", keySize, (keySize + 7) / 8);
  printf("IV size                           : %4d bits / %3d bytes\n", ivSize, (ivSize + 7) / 8);
  printf("Implicit block size               : %4d bits / %3d byte%s\n", implicitBlockSize, (implicitBlockSize + 7) / 8, ((implicitBlockSize + 7) / 8)==1 ? " " : "s");
  printf("Suppressed during Key and IV setup: %4d bits / %3d bytes / %d blocks\n\n", suppressedBits, (suppressedBits + 7) / 8, (suppressedBits + implicitBlockSize - 1) / implicitBlockSize);

  /* kat tests */
  for (i=0; i<sizeof(kat)/sizeof(katType); i++)
    if ((kat[i].cipher == cipher) && (kat[i].applicability & BITSLICED))
      foundError = performKatTestBitsliced(cipher, name, keySize, ivSize, suppressedBits, implicitBlockSize, &kat[i]) || foundError;

  /* negative tests */
  for (i=0; i<sizeof(kat)/sizeof(katType); i++) {
    if (kat[i].cipher == cipher) { /* if cipher has a test case */
      UINT64 *key = (UINT64*)MALLOC(keySize * sizeof(UINT64));
      UINT64 *iv = (UINT64*)MALLOC(ivSize * sizeof(UINT64));
      UINT64 *out = (UINT64*)MALLOC((suppressedBits + implicitBlockSize + 1) * sizeof(UINT64));
      int ptLen = implicitBlockSize;
      UINT64 *pt = (UINT64*)MALLOC(ptLen * sizeof(UINT64));
      const UINT64 marker = U64C(0xABBAACDCABBAACDC);
      int res;

      toBitslicedBuf(key, kat[i].key, keySize);
      toBitslicedBuf(iv, kat[i].iv, ivSize);
      printf("Negative tests:\n");
      printf("Error returned when insufficient plaintext (with initial round output): ");
      res = blackBoxEncryptBitsliced(cipher, key, iv, NULL, 0, out, suppressedBits + implicitBlockSize, 1 /* with init round output */);
      if (!res) foundError = 1;
      printf("%s\n", res ? "OK" : "NOT OK!!!");
      printf("Error returned when insufficient plaintext (no initial round output): ");
      res = blackBoxEncryptBitsliced(cipher, key, iv, NULL, 0, out, implicitBlockSize, 0 /* no init round output */);
      if (!res) foundError = 1;
      printf("%s\n", res ? "OK" : "NOT OK!!!");

      MEMSET(pt, 0, ptLen * sizeof(UINT64));
      MEMSET(out, 0, suppressedBits * sizeof(UINT64));
      out[suppressedBits] = marker;
      printf("Buffer overwrite check with hidden bytes only: ");
      res = blackBoxEncryptBitsliced(cipher, key, iv, pt, ptLen, out, suppressedBits, 1 /* with init round output */);
      if (!res) {
        printf("%s\n\n\n", out[suppressedBits] == marker ? "OK" : "NOT OK!!!");
        if (out[suppressedBits] != marker)
          foundError = 1;
      } else {
        printf("ENCRYPTION FAILED (%d)!!!\n\n\n", res);
        foundError = 1;
      }

      FREE(pt);
      FREE(out);
      FREE(iv);
      FREE(key);
      break; /* do this only once */
    }
  }

  return foundError;
}

void blackBoxFirstTest(bbCipher cipher) {
  int err = blackBoxFirstTestStandard(cipher);
  err = blackBoxFirstTestBitsliced(cipher) || err;
  if (!err)
    printf("All tests OK for %s!\n\n", blackBoxCipherName(cipher));
  else
    printf("Error(s) detected for %s!\n\n", blackBoxCipherName(cipher));
}

void blackBoxFirstTestAll(void) {
  int i, err = 0;
  starHeader("Black Box - First Test");
  for (i=0; i<kNumBlackBoxCiphers; i++) {
    err = blackBoxFirstTestStandard((bbCipher)i) || err;
    err = blackBoxFirstTestBitsliced((bbCipher)i) || err;
  }
  if (!err)
    printf("\nAll tests OK!\n\n");
  else
    printf("\nError(s) detected!\n\n");
}

