/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_des.h"
#include <string.h>
#include <malloc.h>

#ifdef ENDIANNESS_UNDETERMINED
#undef ENDIANNESS_UNDETERMINED
#endif
#ifdef BIG_ENDIAN
#undef BIG_ENDIAN
#endif
#ifndef LITTLE_ENDIAN
#define LITTLE_ENDIAN
#endif

/* Macros: */
/* Tweak number of rounds: define DES_ROUNDS 16 */
/* Don't use initial permution and its inverse: define DES_NO_IP */

#ifndef DES_ROUNDS
#define DES_ROUNDS 16 /* if specifying, use 0 < DES_ROUNDS <= 16 */
#endif

typedef enum {
  kEncrypt, /* Encryption */
  kDecrypt  /* Decryption */
} CryptionMode;

typedef enum {
  kECB, /* Electronic-codebook */
  kCBC, /* Cipher-block chaining */
  kCFB, /* Cipher feedback */
  kOFB, /* Output feedback */
  kCTR  /* Counter mode */
} BlockMode;

typedef struct {
  CryptionMode cm; /* kEncrypt or kDecrypt */
  BlockMode bm; /* kECB (Electronic Code Book) or kCBC (Cipher Block Chaining) */
  BYTE *key; /* Cryption key */
  UINT32 keyLengthInBits; /* Length of cryption key in bits */
  BYTE *IV; /* Initialization vector must be provided if block mode kCBC is used (set to NULL when using kECB) */
  UINT32 IVLengthInBytes; /* IV length in bytes (set to 0 when using block mode kECB) (IV length must be same as the block length of the cipher) */
  BYTE *output; /* Output buffer (this is where the cryption result, the cipher text, is placed) */
  BYTE Status;
  UINT32 expandedKeyProvided;
  UINT32 KL[DES_ROUNDS], KR[DES_ROUNDS]; /* container for the expanded key */
} DES_info;

#define REVERSE_BYTE_ORDER_32(x) \
  { \
    UINT32 v_rev = (((x) << 16) | ((x) >> 16)); \
    x = ( ((v_rev & 0xff00ff00UL) >> 8) | ((v_rev & 0x00ff00ffUL) << 8) ); \
  }

#define COPY_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d = *s; }
#define XOR_BUF_8(dst, src) { UINT64 *d = (UINT64*)(dst); UINT64 *s = (UINT64*)(src); *d ^= *s; }

#define DES_BLOCK_SIZE 8

#define RET(H,err) \
  { \
    H->Status = err; \
    return; \
  }

#define CHECK_DES_PARAMS(B,cm,bm) \
    if (!(B->output)) /* No provided output buffer */ \
      RET(B,-1) /* Invalid output buffer */ \
    if ((cm != kEncrypt) && (cm != kDecrypt)) \
      RET(B,-1) /* invalid cryption mode */ \
    if ((bm != kECB) && (bm != kCBC)) \
      RET(B,-1) /* invalid block mode */ \
    if (!(B->expandedKeyProvided)) \
    { \
      if (!(B->key)) \
        RET(B,-1) /* invalid key buffer */ \
      if (B->keyLengthInBits != 64) \
        RET(B,-1) /* invalid key length */ \
    } \
    if (bm == kCBC) \
    { \
      if (!(B->IV)) \
        RET(B,-1) /* invalid IV buffer */ \
      if (B->IV && (B->IVLengthInBytes != 8)) \
        RET(B,-1) /* invalid IV length */ \
    }

/* PC1 */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
#define maskd (maskc >> 4)
#define PC1(C,D,KC,KD) \
  { \
    UINT32 j, mask = 0x00000080L; /* bit 57/25 */ \
    C = D = 0x00000000L; \
    for (j=0; j<3; j++, mask >>= 1) \
    { \
      UINT32 k, maskc = mask; \
      for (k=0; k<4; k++) \
      { \
        UINT32 shc = 56 - (7*j) - (9*k); \
        UINT32 shd = shc + (j << 4) - 12; \
        C |= ((shc>32)?((KD & maskc) << (shc-32)):((KD & maskc) >> (32-shc))); \
        C |= ((shc>36)?((KC & maskc) << (shc-36)):((KC & maskc) >> (36-shc))); \
        D |= ((shd>32)?((KD & maskd) << (shd-32)):((KD & maskd) >> (32-shd))); \
        D |= ((shd>36)?((KC & maskd) << (shd-36)):((KC & maskd) >> (36-shd))); \
        maskc <<= 8; \
      } \
    } \
    C |= (KD & mask) << 3; \
    D |= (KC & mask) << 3; \
    for (j=0; j<3; j++) \
    { \
      mask <<= 8; \
      C |= (KD & mask) >> (6 + (9*j)); \
      D |= (KC & mask) >> (6 + (9*j)); \
    } \
  }
#else
#define PC1(C,D,KC,KD) \
  { \
    UINT32 mask = 0x00000080L; /* bit 57/25 */ \
    C = (KD & mask) << 24; \
    C |= (KC & mask) << 20; mask <<= 8; \
    C |= (KD & mask) << 15; \
    C |= (KC & mask) << 11; mask <<= 8; \
    C |= (KD & mask) << 6; \
    C |= (KC & mask) << 2;  mask <<= 8; \
    C |= (KD & mask) >> 3; \
    C |= (KC & mask) >> 7; \
    mask >>= 25; /* bit 58/26 */ \
    C |= (KD & mask) << 17; \
    C |= (KC & mask) << 13; mask <<= 8; \
    C |= (KD & mask) << 8; \
    C |= (KC & mask) << 4;  mask <<= 8; \
    C |= (KD & mask) >> 1; \
    C |= (KC & mask) >> 5;  mask <<= 8; \
    C |= (KD & mask) >> 10; \
    C |= (KC & mask) >> 14; \
    mask >>= 25; /* bit 59/27 */ \
    C |= (KD & mask) << 10; \
    C |= (KC & mask) << 6;  mask <<= 8; \
    C |= (KD & mask) << 1; \
    C |= (KC & mask) >> 3;  mask <<= 8; \
    C |= (KD & mask) >> 8; \
    C |= (KC & mask) >> 12; mask <<= 8; \
    C |= (KD & mask) >> 17; \
    C |= (KC & mask) >> 21; \
    mask >>= 25; /* bit 60/28 */ \
    C |= (KD & mask) << 3; \
    D = (KC & mask) << 3;  mask <<= 8; \
    C |= (KD & mask) >> 6; \
    D |= (KC & mask) >> 6;  mask <<= 8; \
    C |= (KD & mask) >> 15; \
    D |= (KC & mask) >> 15; mask <<= 8; \
    C |= (KD & mask) >> 24; \
    D |= (KC & mask) >> 24; \
    mask >>= 25; /* bit 61/29 */ \
    D |= (KD & mask) << 12; \
    D |= (KC & mask) << 8;  mask <<= 8; \
    D |= (KD & mask) << 3; \
    D |= (KC & mask) >> 1;  mask <<= 8; \
    D |= (KD & mask) >> 6; \
    D |= (KC & mask) >> 10; mask <<= 8; \
    D |= (KD & mask) >> 15; \
    D |= (KC & mask) >> 19; \
    mask >>= 25; /* bit 62/30 */ \
    D |= (KD & mask) << 21; \
    D |= (KC & mask) << 17; mask <<= 8; \
    D |= (KD & mask) << 12; \
    D |= (KC & mask) << 8;  mask <<= 8; \
    D |= (KD & mask) << 3; \
    D |= (KC & mask) >> 1;  mask <<= 8; \
    D |= (KD & mask) >> 6; \
    D |= (KC & mask) >> 10; \
    mask >>= 25; /* bit 63/31 */ \
    D |= (KD & mask) << 30; \
    D |= (KC & mask) << 26; mask <<= 8; \
    D |= (KD & mask) << 21; \
    D |= (KC & mask) << 17; mask <<= 8; \
    D |= (KD & mask) << 12; \
    D |= (KC & mask) << 8;  mask <<= 8; \
    D |= (KD & mask) << 3; \
    D |= (KC & mask) >> 1; \
  }
#endif

/* PC2 */
#define PC2(KL,KR,i,C,D) \
  { \
    UINT32 CT = (C & 0x00000110L) << 20; /* bit 24 in C -> bit 4 in KL[i] and bit 28 in C -> bit 8 in KL[i] */ \
    UINT32 DT = (D & 0x00000100L) << 22; /* bit 24 in D -> bit 2 in KR[i] */ \
    CT |= (C & 0x00008000L) << 15; /* 17->2 */ \
    CT |= (C & 0x00040000L) << 13; /* 14->1 */ \
    CT |= (C & 0x00000a00L) << 10; /* 21->11, 23->13 */ \
    CT |= (C & 0x00200000L) << 8;  /* 11->3 */ \
    CT |= (C & 0x00000040L) << 9;  /* 26->17 */ \
    CT |= (C & 0x00020020L) << 6;  /* 15->9, 27->21 */ \
    CT |= (C & 0x00002000L) << 5;  /* 19->14 */ \
    CT |= (C & 0x08000000L) >> 1;  /* 5->6 */ \
    CT |= (C & 0x00401000L) >> 2;  /* 10->12, 20->22 */ \
    CT |= (C & 0x00110000L) >> 3;  /* 12->15, 16->19 */ \
    CT |= (C & 0xa4000000L) >> 4;  /* 1->5, 3->7, 6->10 */ \
    CT |= (C & 0x01080000L) >> 10; /* 8->18, 13->23 */ \
    CT |= (C & 0x10000000L) >> 12; /* 4->16 */ \
    CT |= (C & 0x02000000L) >> 13; /* 7->20 */ \
    CT |= (C & 0x40000000L) >> 22; /* 2->24 */ \
    DT |= (D & 0x00000020L) << 21; /* 27->6 */ \
    DT |= (D & 0x00002200L) << 14; /* 19->5, 23->9 */ \
    DT |= (D & 0x00080010L) << 12; /* 13->1, 28->16 */ \
    DT |= (D & 0x00001000L) << 8;  /* 20->12 */ \
    DT |= (D & 0x00008880L) << 7;  /* 17->10, 21->14, 25->18 */ \
    DT |= (D & 0x00800000L) << 5;  /* 9->4 */ \
    DT |= (D & 0x00100000L) << 4;  /* 12->8 */ \
    DT |= (D & 0x00010000L) << 3;  /* 16->13 */ \
    DT |= (D & 0x00000400L) << 1;  /* 22->21 */ \
    DT |= (D & 0x20000000L);       /* 3->3 */ \
    DT |= (D & 0x00004000L) >> 1;  /* 18->19 */ \
    DT |= (D & 0x00200000L) >> 4;  /* 11->15 */ \
    DT |= (D & 0x40000000L) >> 5;  /* 2->7 */ \
    DT |= (D & 0x08040000L) >> 6;  /* 5->11, 14->20 */ \
    DT |= (D & 0x04000000L) >> 11; /* 6->17 */ \
    DT |= (D & 0x01000000L) >> 14; /* 8->22 */ \
    DT |= (D & 0x10000000L) >> 20; /* 4->24 */ \
    DT |= (D & 0x80000000L) >> 22; /* 1->23 */ \
    KL[i] = CT; \
    KR[i] = DT; \
  }

/* 28-bit left rotation */
#define ROTL28(X,n) (((X) << (n)) | (((X) >> (28 - (n))) & 0xfffffff0L))

/* Key schedule */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
#define KEY_SCHEDULE(KL,KR,KC,KD) \
  { \
    UINT32 C, D; \
    PC1(C,D,KC,KD) \
    for (i=0; i<DES_ROUNDS; i++) \
    { \
      UINT32 vi = ((i == 0 || i == 1 || i == 8 || i == 15)?(1):(2)); \
      C = ROTL28(C,vi); \
      D = ROTL28(D,vi); \
      PC2(KL,KR,i,C,D) \
    } \
  }
#else
#define KEY_SCHEDULE(KL,KR,KC,KD) \
  { \
    UINT32 C, D; \
    PC1(C,D,KC,KD) \
    for (i=0; i<DES_ROUNDS; i++) \
    { \
      static const BYTE v[16] = {0x01,0x01,0x02,0x02,0x02,0x02,0x02,0x02, \
                                 0x01,0x02,0x02,0x02,0x02,0x02,0x02,0x01}; \
      UINT32 vi = (UINT32)v[i]; \
      C = ROTL28(C,vi); \
      D = ROTL28(D,vi); \
      PC2(KL,KR,i,C,D) \
    } \
  }
#endif

/* Initial permutation - support function */
void IP_one_side(UINT32 *B, UINT32 xL, UINT32 xR, UINT32 mask)
{
  UINT32 mask1, mask2, A;
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
  UINT32 i;
#endif

  A = 0x00000000L;
  mask2 = 0x80000000L; /* bit 1 */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
  for (i=0; i<4; i++)
  {
    mask1 = mask;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 4); mask2 >>= 5;
    mask >>= 2;
  }
#else
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask2 >>= 5;
  mask >>= 2;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask2 >>= 5;
  mask >>= 2;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask2 >>= 5;
  mask >>= 2;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4); mask1 <<= 8; mask2 >>= 1;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 4);
#endif
  *B = A;
}

/* Initial permutation */
/*  (L,R) = IP(xL,xR)  */
#ifdef DES_NO_IP
#define IP(L, R, xL, xR) \
  *L = xL; \
  *R = xR;
#else
#define IP(L, R, xL, xR) \
{ \
  IP_one_side((L), xL, xR, 0x00000040L); /* start at bit 58 */ \
  IP_one_side((R), xL, xR, 0x00000080L); /* start at bit 57 */ \
}
#endif

/* Inverse initial permutation - support function */
void IPinv_one_side(UINT32 *B, UINT32 xL, UINT32 xR, UINT32 mask)
{
  UINT32 mask1, mask2, A;
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
  UINT32 i;
#endif

  A = 0x00000000L;
  mask2 = 0x80000000L; /* bit 1 */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
  for (i=0; i<4; i++)
  {
    mask1 = mask;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
    if (xR & mask1) A |= mask2;
    if (xL & mask1) A |= (mask2 >> 1); mask2 >>= 2;
    mask <<= 1;
  }
#else
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask2 >>= 2;
  mask <<= 1;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask2 >>= 2;
  mask <<= 1;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask2 >>= 2;
  mask <<= 1;
  mask1 = mask;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1); mask1 >>= 8; mask2 >>= 2;
  if (xR & mask1) A |= mask2;
  if (xL & mask1) A |= (mask2 >> 1);
#endif
  *B = A;
}

/* Inverse initial permutation */
/*  (L,R) = IPinv(xL,xR)  */
#ifdef DES_NO_IP
#define IPinv(L, R, xL, xR) \
  *L = xL; \
  *R = xR;
#else
#define IPinv(L, R, xL, xR) \
    IPinv_one_side((L), xL, xR, 0x01000000L); /* start at bit 40 */ \
    IPinv_one_side((R), xL, xR, 0x10000000L); /* start at bit 36 */
#endif

/* mapping E */
#define E_MAP(EL,ER,R) \
  { \
    UINT32 mask = 0x1f800000L; /* bits 4 through 9 */ \
    EL = 0x00000000L; \
    if (R & 0x00000001L) \
      EL = 0x80000000L; \
    EL |= (R & 0xf8000000L) >> 1; \
    EL |= (R & mask) >> 3; \
    mask >>= 4; /* bits 8 through 13 */ \
    EL |= (R & mask) >> 5; \
    mask >>= 4; /* bits 12 through 17 */ \
    EL |= (R & mask) >> 7; \
    ER = 0x00000000L; \
    if (R & 0x80000000L) \
      ER = 0x00000100L; \
    mask >>= 4; /* bits 16 through 21 */ \
    ER |= (R & mask) << 15; \
    mask >>= 4; /* bits 20 through 25 */ \
    ER |= (R & mask) << 13; \
    mask >>= 4; /* bits 24 through 29 */ \
    ER |= (R & mask) << 11; \
    mask >>= 4; /* bits 28 through 32 */ \
    ER |= (R & mask) << 9; \
  }

/* xor key */
#define XOR_KEY(cm,EL,ER,KL,KR,round) \
  { \
    if (cm == kEncrypt) \
    { \
      EL ^= KL[round];  /* (EL,ER)<-(EL^KL[round],ER^KR[round]) for encryption */ \
      ER ^= KR[round]; \
    } \
    else /* if (cm == kDecrypt) */ \
    { \
      EL ^= KL[(DES_ROUNDS-1)-round];  /* (EL,ER)<-(EL^KL[15-round],ER^KR[15-round]) for decryption */ \
      ER ^= KR[(DES_ROUNDS-1)-round]; \
    } \
  }

/* S-Boxes and S-Box substitution */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
static const BYTE DES_SBox_nBcompact[8][32] = { /* for smaller code size */
 {0xe0,0x4f,0xd7,0x14,0x2e,0xf2,0xbd,0x81,0x3a,0xa6,0x6c,0xcb,0x59,0x95,0x03,0x78,0x4f,0x1c,0xe8,0x82,0xd4,0x69,0x21,0xb7,0xf5,0xcb,0x93,0x7e,0x3a,0xa0,0x56,0x0d}, /* S1 */
 {0xf3,0x1d,0x84,0xe7,0x6f,0xb2,0x38,0x4e,0x9c,0x70,0x21,0xda,0xc6,0x09,0x5b,0xa5,0x0d,0xe8,0x7a,0xb1,0xa3,0x4f,0xd4,0x12,0x5b,0x86,0xc7,0x6c,0x90,0x35,0x2e,0xf9}, /* S2 */
 {0xad,0x07,0x90,0xe9,0x63,0x34,0xf6,0x5a,0x12,0xd8,0xc5,0x7e,0xbc,0x4b,0x2f,0x81,0xd1,0x6a,0x4d,0x90,0x86,0xf9,0x38,0x07,0xb4,0x1f,0x2e,0xc3,0x5b,0xa5,0xe2,0x7c}, /* S3 */
 {0x7d,0xd8,0xeb,0x35,0x06,0x6f,0x90,0xa3,0x14,0x27,0x82,0x5c,0xb1,0xca,0x4e,0xf9,0xa3,0x6f,0x90,0x06,0xca,0xb1,0x7d,0xd8,0xf9,0x14,0x35,0xeb,0x5c,0x27,0x82,0x4e}, /* S4 */
 {0x2e,0xcb,0x42,0x1c,0x74,0xa7,0xbd,0x61,0x85,0x50,0x3f,0xfa,0xd3,0x09,0xe8,0x96,0x4b,0x28,0x1c,0xb7,0xa1,0xde,0x72,0x8d,0xf6,0x9f,0xc0,0x59,0x6a,0x34,0x05,0xe3}, /* S5 */
 {0xca,0x1f,0xa4,0xf2,0x97,0x2c,0x69,0x85,0x06,0xd1,0x3d,0x4e,0xe0,0x7b,0x53,0xb8,0x94,0xe3,0xf2,0x5c,0x29,0x85,0xcf,0x3a,0x7b,0x0e,0x41,0xa7,0x16,0xd0,0xb8,0x6d}, /* S6 */
 {0x4d,0xb0,0x2b,0xe7,0xf4,0x09,0x81,0xda,0x3e,0xc3,0x95,0x7c,0x52,0xaf,0x68,0x16,0x16,0x4b,0xbd,0xd8,0xc1,0x34,0x7a,0xe7,0xa9,0xf5,0x60,0x8f,0x0e,0x52,0x93,0x2c}, /* S7 */
 {0xd1,0x2f,0x8d,0x48,0x6a,0xf3,0xb7,0x14,0xac,0x95,0x36,0xeb,0x50,0x0e,0xc9,0x72,0x72,0xb1,0x4e,0x17,0x94,0xca,0xe8,0x2d,0x0f,0x6c,0xa9,0xd0,0xf3,0x35,0x56,0x8b}  /* S8 */
};
#define SBox(n,B) ((((B)&0x01)?((DES_SBox_nBcompact[n][(B) >> 1])&0x0f):((DES_SBox_nBcompact[n][(B) >> 1])>>4)))
#else
static const BYTE DES_SBox_nB[8][64] = { /* for faster look-ups */
 {0x0e,0x00,0x04,0x0f,0x0d,0x07,0x01,0x04,0x02,0x0e,0x0f,0x02,0x0b,0x0d,0x08,0x01,0x03,0x0a,0x0a,0x06,0x06,0x0c,0x0c,0x0b,0x05,0x09,0x09,0x05,0x00,0x03,0x07,0x08,0x04,0x0f,0x01,0x0c,0x0e,0x08,0x08,0x02,0x0d,0x04,0x06,0x09,0x02,0x01,0x0b,0x07,0x0f,0x05,0x0c,0x0b,0x09,0x03,0x07,0x0e,0x03,0x0a,0x0a,0x00,0x05,0x06,0x00,0x0d}, /* S1 */
 {0x0f,0x03,0x01,0x0d,0x08,0x04,0x0e,0x07,0x06,0x0f,0x0b,0x02,0x03,0x08,0x04,0x0e,0x09,0x0c,0x07,0x00,0x02,0x01,0x0d,0x0a,0x0c,0x06,0x00,0x09,0x05,0x0b,0x0a,0x05,0x00,0x0d,0x0e,0x08,0x07,0x0a,0x0b,0x01,0x0a,0x03,0x04,0x0f,0x0d,0x04,0x01,0x02,0x05,0x0b,0x08,0x06,0x0c,0x07,0x06,0x0c,0x09,0x00,0x03,0x05,0x02,0x0e,0x0f,0x09}, /* S2 */
 {0x0a,0x0d,0x00,0x07,0x09,0x00,0x0e,0x09,0x06,0x03,0x03,0x04,0x0f,0x06,0x05,0x0a,0x01,0x02,0x0d,0x08,0x0c,0x05,0x07,0x0e,0x0b,0x0c,0x04,0x0b,0x02,0x0f,0x08,0x01,0x0d,0x01,0x06,0x0a,0x04,0x0d,0x09,0x00,0x08,0x06,0x0f,0x09,0x03,0x08,0x00,0x07,0x0b,0x04,0x01,0x0f,0x02,0x0e,0x0c,0x03,0x05,0x0b,0x0a,0x05,0x0e,0x02,0x07,0x0c}, /* S3 */
 {0x07,0x0d,0x0d,0x08,0x0e,0x0b,0x03,0x05,0x00,0x06,0x06,0x0f,0x09,0x00,0x0a,0x03,0x01,0x04,0x02,0x07,0x08,0x02,0x05,0x0c,0x0b,0x01,0x0c,0x0a,0x04,0x0e,0x0f,0x09,0x0a,0x03,0x06,0x0f,0x09,0x00,0x00,0x06,0x0c,0x0a,0x0b,0x01,0x07,0x0d,0x0d,0x08,0x0f,0x09,0x01,0x04,0x03,0x05,0x0e,0x0b,0x05,0x0c,0x02,0x07,0x08,0x02,0x04,0x0e}, /* S4 */
 {0x02,0x0e,0x0c,0x0b,0x04,0x02,0x01,0x0c,0x07,0x04,0x0a,0x07,0x0b,0x0d,0x06,0x01,0x08,0x05,0x05,0x00,0x03,0x0f,0x0f,0x0a,0x0d,0x03,0x00,0x09,0x0e,0x08,0x09,0x06,0x04,0x0b,0x02,0x08,0x01,0x0c,0x0b,0x07,0x0a,0x01,0x0d,0x0e,0x07,0x02,0x08,0x0d,0x0f,0x06,0x09,0x0f,0x0c,0x00,0x05,0x09,0x06,0x0a,0x03,0x04,0x00,0x05,0x0e,0x03}, /* S5 */
 {0x0c,0x0a,0x01,0x0f,0x0a,0x04,0x0f,0x02,0x09,0x07,0x02,0x0c,0x06,0x09,0x08,0x05,0x00,0x06,0x0d,0x01,0x03,0x0d,0x04,0x0e,0x0e,0x00,0x07,0x0b,0x05,0x03,0x0b,0x08,0x09,0x04,0x0e,0x03,0x0f,0x02,0x05,0x0c,0x02,0x09,0x08,0x05,0x0c,0x0f,0x03,0x0a,0x07,0x0b,0x00,0x0e,0x04,0x01,0x0a,0x07,0x01,0x06,0x0d,0x00,0x0b,0x08,0x06,0x0d}, /* S6 */
 {0x04,0x0d,0x0b,0x00,0x02,0x0b,0x0e,0x07,0x0f,0x04,0x00,0x09,0x08,0x01,0x0d,0x0a,0x03,0x0e,0x0c,0x03,0x09,0x05,0x07,0x0c,0x05,0x02,0x0a,0x0f,0x06,0x08,0x01,0x06,0x01,0x06,0x04,0x0b,0x0b,0x0d,0x0d,0x08,0x0c,0x01,0x03,0x04,0x07,0x0a,0x0e,0x07,0x0a,0x09,0x0f,0x05,0x06,0x00,0x08,0x0f,0x00,0x0e,0x05,0x02,0x09,0x03,0x02,0x0c}, /* S7 */
 {0x0d,0x01,0x02,0x0f,0x08,0x0d,0x04,0x08,0x06,0x0a,0x0f,0x03,0x0b,0x07,0x01,0x04,0x0a,0x0c,0x09,0x05,0x03,0x06,0x0e,0x0b,0x05,0x00,0x00,0x0e,0x0c,0x09,0x07,0x02,0x07,0x02,0x0b,0x01,0x04,0x0e,0x01,0x07,0x09,0x04,0x0c,0x0a,0x0e,0x08,0x02,0x0d,0x00,0x0f,0x06,0x0c,0x0a,0x09,0x0d,0x00,0x0f,0x03,0x03,0x05,0x05,0x06,0x08,0x0b}  /* S8 */
};
#define SBox(n,B) (DES_SBox_nB[n][B])
#endif

/* mapping S - big endian version */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
#define Smap_BE(S,EL,ER) \
  { \
    UINT32 is, mask = 0xfc000000L; /* bits 1 through 6 */ \
    UINT32 D = (EL & mask) >> 2; /* B1 */ \
    S = 0x00000000L; \
    S |= ((UINT32)(SBox(0,D))) << 4; \
    D = (ER & mask) >> 2; /* B5 */ \
    S |= ((UINT32)(SBox(4,D))) >> 12; \
    for (is=0; is<3; is++) \
    { \
      mask >>= 6; \
      D = (EL & mask) << (4+(is*6)); \
      S |= ((UINT32)(SBox((is+1),D))) >> (is<<2); \
      D = (ER & mask) << (4+(is*6)); \
      S |= ((UINT32)(SBox((is+5),D))) >> (16+(is<<2)); \
    } \
  }
#else
#define Smap_BE(S,EL,ER) \
  { \
    UINT32 mask = 0xfc000000L; /* bits 1 through 6 */ \
    UINT32 D = (EL & mask) >> 2; /* B1 */ \
    S = 0x00000000L; \
    S |= ((UINT32)(SBox(0,D))) << 4; \
    D = (ER & mask) >> 2; /* B5 */ \
    S |= ((UINT32)(SBox(4,D))) >> 12; \
    \
    mask >>= 6; /* bits 7 through 12 */ \
    D = (EL & mask) << 4; /* B2 */ \
    S |= ((UINT32)(SBox(1,D))); \
    D = (ER & mask) << 4; /* B6 */ \
    S |= ((UINT32)(SBox(5,D))) >> 16; \
    \
    mask >>= 6; /* bits 13 through 18 */ \
    D = (EL & mask) << 10; /* B3 */ \
    S |= ((UINT32)(SBox(2,D))) >> 4; \
    D = (ER & mask) << 10; /* B7 */ \
    S |= ((UINT32)(SBox(6,D))) >> 20; \
    \
    mask >>= 6; /* bits 19 through 24 */ \
    D = (EL & mask) << 16; /* B4 */ \
    S |= ((UINT32)(SBox(3,D))) >> 8; \
    D = (ER & mask) << 16; /* B8 */ \
    S |= ((UINT32)(SBox(7,D))) >> 24; \
  }
#endif

/* mapping S - little endian version */
#ifdef STANSOFTSEC_OPTIMIZE_SIZE_DES
#define Smap_LE(S,EL,ER) \
  { \
    UINT32 is, mask = 0xfc000000L; /* bits 1 through 6 */ \
    S = 0x00000000L; \
    for (is=0; is<4; is++) \
    { \
      UINT32 D = (EL & mask) >> (26-(is*6)); \
      S |= ((UINT32)(SBox(is,D))) << (28-(is<<2)); \
      D = (ER & mask) >> (26-(is*6)); \
      S |= ((UINT32)(SBox((is+4),D))) << (12-(is<<2)); \
      mask >>= 6; \
    } \
  }
#else
#define Smap_LE(S,EL,ER) \
  { \
    UINT32 mask = 0xfc000000L; /* bits 1 through 6 */ \
    UINT32 D = (EL & mask) >> 26; /* B1 */ \
    S = 0x00000000L; \
    S |= ((UINT32)(SBox(0,D))) << 28; \
    D = (ER & mask) >> 26; /* B5 */ \
    S |= ((UINT32)(SBox(4,D))) << 12; \
    \
    mask >>= 6; /* bits 7 through 12 */ \
    D = (EL & mask) >> 20; /* B2 */ \
    S |= ((UINT32)(SBox(1,D))) << 24; \
    D = (ER & mask) >> 20; /* B6 */ \
    S |= ((UINT32)(SBox(5,D))) << 8; \
    \
    mask >>= 6; /* bits 13 through 18 */ \
    D = (EL & mask) >> 14; /* B3 */ \
    S |= ((UINT32)(SBox(2,D))) << 20; \
    D = (ER & mask) >> 14; /* B7 */ \
    S |= ((UINT32)(SBox(6,D))) << 4; \
    \
    mask >>= 6; /* bits 19 through 24 */ \
    D = (EL & mask) >> 8; /* B4 */ \
    S |= ((UINT32)(SBox(3,D))) << 16; \
    D = (ER & mask) >> 8; /* B8 */ \
    S |= ((UINT32)(SBox(7,D))); \
  }
#endif

/* mapping S */
/* (EL,ER) = ((B1,B2,B3,B4),(B5,B6,B7,B8)) */ \
/* S<-(S1(B1),S2(B2),...,S8(B8)) */ \
#if defined(LITTLE_ENDIAN)
#define S_MAP(S,EL,ER) Smap_LE(S,EL,ER)
#elif defined(BIG_ENDIAN)
#define S_MAP(S,EL,ER) Smap_BE(S,EL,ER)
#else /* ENDIANNESS_UNDETERMINED */
#define S_MAP(S,EL,ER) \
  { \
    if (LITTLE_ENDIAN_PLATFORM) \
      Smap_LE(S,EL,ER) \
    else \
      Smap_BE(S,EL,ER) \
  }
#endif

/* mapping P */
#define P_MAP(F,S) \
    F = (S & 0x00000004L) << 3; /* bit 30 in S -> bit 27 in F */ \
    F |= (S & 0x00004000L) << 4; /* 18->14 */ \
    F |= (S & 0x02020120L) << 5; /* 7->2, 15->10, 24->19, 27->22 */ \
    F |= (S & 0x00100000L) << 6; /* 12->6 */ \
    F |= (S & 0x00442000L) >> 6; /* 10->16, 14->20, 19->25 */ \
    F |= (S & 0x00000480L) >> 7; /* 22->29, 25->32 */ \
    F |= (S & 0x88000000L) >> 8; /* 1->9, 5->13 */ \
    F |= (S & 0x00008000L) << 9; /* 17->8 */ \
    F |= (S & 0x01000000L) >> 10; /* 8->18 */ \
    F |= (S & 0x00000001L) << 11; /* 32->21 */ \
    F |= (S & 0x00000200L) << 12; /* 23->11 */ \
    F |= (S & 0x00080000L) >> 13; /* 13->26 */ \
    F |= (S & 0x00000040L) << 14; /* 26->12 */ \
    F |= (S & 0x00010000L) << 15; /* 16->1 */ \
    F |= (S & 0x40800000L) >> 15; /* 2->17, 9->24 */ \
    F |= (S & 0x00000002L) << 16; /* 31->15 */ \
    F |= (S & 0x00001800L) << 17; /* 20->3, 21->4 */ \
    F |= (S & 0x00200000L) >> 19; /* 11->30 */ \
    F |= (S & 0x20000000L) >> 20; /* 3->23 */ \
    F |= (S & 0x00000010L) << 21; /* 28->7 */ \
    F |= (S & 0x04000000L) >> 22; /* 6->28 */ \
    F |= (S & 0x00000008L) << 24; /* 29->5 */ \
    F |= (S & 0x10000000L) >> 27; /* 4->31 */


#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define REVERSE_L_R(L,R) \
  if (LITTLE_ENDIAN_PLATFORM) \
  { \
    REVERSE_BYTE_ORDER_32(L) \
    REVERSE_BYTE_ORDER_32(R) \
  }
#elif defined(LITTLE_ENDIAN)
#define REVERSE_L_R(L,R) \
  REVERSE_BYTE_ORDER_32(L) \
  REVERSE_BYTE_ORDER_32(R)
#else
#define REVERSE_L_R(L,R) /* Do nothing */
#endif

#define INITIALIZE_L_R(bm,IV,L,R,Lx,Rx) \
  if (bm == kCBC) \
  { \
    L = Lx = *IV; \
    R = Rx = *(IV+1); \
  }

#define GET_NEXT_INPUT_BLOCK(bm,cm,L,R,x) \
    if ((bm == kCBC) && (cm == kEncrypt)) \
    { \
      L ^= *((UINT32*)(x)); /* Get next input block */ \
      R ^= *((UINT32*)(x+4)); \
    } \
    else \
    { \
      L = *((UINT32*)(x)); /* Get next input block */ \
      R = *((UINT32*)(x+4)); \
    }

#ifdef ENDIANNESS_UNDETERMINED /* Reverse byte order if little endian platform */
#define UPDATE_OUTPUT_BUFFER(B,L,R) \
    { \
      UINT32 *output32Ptr = (UINT32*)(B->output); /* Get output buffer */ \
      UINT32 tmp1 = L, tmp2 = R; \
      if (LITTLE_ENDIAN_PLATFORM) \
      { \
        REVERSE_BYTE_ORDER_32(tmp1) \
        REVERSE_BYTE_ORDER_32(tmp2) \
      } \
      *output32Ptr++ = tmp1; /* crypted block to output buffer */ \
      *output32Ptr = tmp2; \
    }
#elif defined(LITTLE_ENDIAN)
#define UPDATE_OUTPUT_BUFFER(B,L,R) \
    { \
      UINT32 *output32Ptr = (UINT32*)(B->output); /* Get output buffer */ \
      UINT32 tmp1 = L, tmp2 = R; \
      REVERSE_BYTE_ORDER_32(tmp1) \
      REVERSE_BYTE_ORDER_32(tmp2) \
      *output32Ptr++ = tmp1; /* crypted block to output buffer */ \
      *output32Ptr = tmp2; \
    }
#else
#define UPDATE_OUTPUT_BUFFER(B,L,R) \
  { \
    UINT32 *output32Ptr = (UINT32*)(B->output); /* Get output buffer */ \
    *output32Ptr++ = L; /* crypted block to output buffer */ \
    *output32Ptr = R; \
  }
#endif

#define XOR_TO_OUTPUT_BUFFER(L,R, out) \
    { \
      UINT32 *output32Ptr = (UINT32*)(out); \
      UINT32 tmp1 = L, tmp2 = R; \
      REVERSE_BYTE_ORDER_32(tmp1) \
      REVERSE_BYTE_ORDER_32(tmp2) \
      *output32Ptr++ ^= tmp1; /* crypted block to output buffer */ \
      *output32Ptr ^= tmp2; \
    }

/*
  Function DES_crypt

  Description: DES cryption function
               as specified in 'Handbook of Applied Cryptography' (ISBN 0-8493-8523-7)

  Input:
    'cm' is the cryption mode (kEncrypt or kDecrypt)
    'H->key' is the DES key to be used in the cryption
    'H->output' is a buffer where the crypted message is placed
    'H->outputBufferLengthInBytes' is the length of the output buffer in bytes
    'x' is the message to be crypted
    'b' is the length of 'x' (in bytes)

  Output:
    'H->Status' is the Status of the calculation (SS_OK if computed)
    'H->output' contains the DES cryption of 'x'
    'H->outputLengthInBytes' is the length of the output in bytes

  Note:
    Encryption and decryption algorithms differ only in the order subkeys are applied.
*/

void DES_init(DES_info *B)
{
  CryptionMode cm = B->cm;
  BlockMode bm = B->bm;
  UINT32 L, R, *KL = B->KL, *KR = B->KR;

  CHECK_DES_PARAMS(B,cm,bm) /* Check parameters */
  if (!(B->expandedKeyProvided))
  {
    UINT32 i;
    L = *((UINT32*)(B->key)); /* Get key */
    R = *((UINT32*)((B->key) + 4));
    REVERSE_L_R(L,R) /* Reverse byte order if little endian platform */
    KEY_SCHEDULE(KL,KR,L,R) /* Key schedule */
  }
  B->Status = 0;
}

void DES_encryptBlock(DES_info *B, BYTE *x)
{
  UINT32 L, R, Li, Ri, round;
  UINT32 *KL = B->KL, *KR = B->KR;
  CryptionMode cm = B->cm;

  L = *((UINT32*)(x)); /* Get next input block */ \
  R = *((UINT32*)(x+4)); \
  REVERSE_L_R(L,R) /* Reverse byte order if little endian platform */
  IP(&Li, &Ri, L, R) /* Initial permutatuion */
  for (round=0; round<DES_ROUNDS; round++) /* 16 Rounds */
  {
    UINT32 S, F;
    UINT32 EL, ER;
    /* round */
    /* Li = R   (eq. 7.4) */
    /* Ri = L ^ f(R,Ki), where f(R,Ki)=P(S(E(R)^Ki))   (eq. 7.5) */
         L = Li;
    Li = R = Ri;
    E_MAP(EL,ER,R)
    XOR_KEY(cm,EL,ER,KL,KR,round)
    S_MAP(S,EL,ER)
    P_MAP(F,S)
    Ri = L ^ F;
  }
  IPinv(&L, &R, Ri, Li) /* Exchange final blocks and transpose */
  UPDATE_OUTPUT_BUFFER(B,L,R)
  B->Status = 0;
}

void DES_encryptBlock_xor_withInitOutput(DES_info *B, BYTE *x, BYTE *lastBlock, int numBlocks) {
  UINT32 L, R, Li, Ri, round;
  UINT32 *KL = B->KL, *KR = B->KR;
  CryptionMode cm = B->cm;
  BYTE *out = B->output;

  B->Status = 0;
  L = *((UINT32*)(x)); /* Get next input block */ \
  R = *((UINT32*)(x+4)); \
  REVERSE_L_R(L,R) /* Reverse byte order if little endian platform */
  IP(&Li, &Ri, L, R) /* Initial permutatuion */
  for (round=0; round<DES_ROUNDS; round++) /* 16 Rounds */
  {
    UINT32 S, F;
    UINT32 EL, ER;
    UINT32 Ltemp, Rtemp;
    /* round */
    /* Li = R   (eq. 7.4) */
    /* Ri = L ^ f(R,Ki), where f(R,Ki)=P(S(E(R)^Ki))   (eq. 7.5) */
         L = Li;
    Li = R = Ri;
    E_MAP(EL,ER,R)
    XOR_KEY(cm,EL,ER,KL,KR,round)
    S_MAP(S,EL,ER)
    P_MAP(F,S)
    Ri = L ^ F;

    Ltemp = L;
    Rtemp = R;
    IPinv(&Ltemp, &Rtemp, Ri, Li) /* Exchange final blocks and transpose */
    XOR_TO_OUTPUT_BUFFER(Ltemp,Rtemp,out)
    if (--numBlocks == 0) return;
    out += DES_BLOCK_SIZE;
  }
  COPY_BUF_8(lastBlock, out - DES_BLOCK_SIZE)
}

void DES_final(DES_info *B) { B->Status = 0; }

/******************************************************************************
 * Black box variants
 ******************************************************************************/
int des_xor(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  DES_info ctx;
  BYTE pt[DES_BLOCK_SIZE];
  BYTE ct[DES_BLOCK_SIZE];
  int i;
  const int numBlocks = numOutputBytes / DES_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 64;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = 8;
  ctx.output = ct;
  ctx.expandedKeyProvided = 0;

  DES_init(&ctx);
  if (ctx.Status) return -1;
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, i==0 ? iv : ct)
    XOR_BUF_8(pt, inBuf + i * DES_BLOCK_SIZE)
    DES_encryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + i * DES_BLOCK_SIZE, ct)
  }
  DES_final(&ctx);
  return ctx.Status;
}

int des_xor_withInitOutput(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes) {
  DES_info ctx;
  BYTE pt[DES_BLOCK_SIZE];
  BYTE ct[DES_BLOCK_SIZE];
  int i;
  int numBlocks = numOutputBytes / DES_BLOCK_SIZE;
  const int numSuppressedBytes = (DES_ROUNDS-1) * DES_BLOCK_SIZE;

  if (numOutputBytes == 0) return 0;
  if (numInputBytes + numSuppressedBytes < numOutputBytes) return -1;
  if ((numOutputBytes & 7) != 0) return -1;
  if (inBuf == outBuf) return -1;

  ctx.cm = kEncrypt;
  ctx.bm = kECB;
  ctx.key = (BYTE*)key; ctx.keyLengthInBits = 64;
  ctx.IV  = (BYTE*)iv;  ctx.IVLengthInBytes = 8;
  ctx.output = outBuf;
  ctx.expandedKeyProvided = 0;

  DES_init(&ctx);
  if (ctx.Status) return -1;

  /* first DES_ROUNDS (16) blocks (DES_ROUNDS-1 (15) suppressed + first) */
  COPY_BUF_8(pt, iv)
  XOR_BUF_8(pt, inBuf)
  DES_encryptBlock_xor_withInitOutput(&ctx, pt, ct, numBlocks);
  if (numBlocks <= DES_ROUNDS) {
    DES_final(&ctx);
    return ctx.Status;
  }
  ctx.output = ct;
  numBlocks -= DES_ROUNDS;

  /* remaining blocks */
  for (i=0; i<numBlocks; i++) {
    COPY_BUF_8(pt, ct)
    XOR_BUF_8(pt, inBuf + (i + 1) * DES_BLOCK_SIZE)
    DES_encryptBlock(&ctx, pt);
    XOR_BUF_8(outBuf + (i + DES_ROUNDS) * DES_BLOCK_SIZE, ct)
  }

  DES_final(&ctx);
  return ctx.Status;
}

/******************************************************************************
 * Black box API
 ******************************************************************************/
/* the black box api assumes contiguous key bytes, so the passed 56-bit key is
   transformed to the corresponding 64-bit version internally */
void insertParitySpaces(BYTE *dst, const BYTE *src) {
  dst[0] = src[0];
  dst[1] = (BYTE)((src[0]<<7) | (src[1]>>1));
  dst[2] = (BYTE)((src[1]<<6) | (src[2]>>2));
  dst[3] = (BYTE)((src[2]<<5) | (src[3]>>3));
  dst[4] = (BYTE)((src[3]<<4) | (src[4]>>4));
  dst[5] = (BYTE)((src[4]<<3) | (src[5]>>5));
  dst[6] = (BYTE)((src[5]<<2) | (src[6]>>6));
  dst[7] = (BYTE)((src[6]<<1) | (src[7]>>7));
}

int blackBoxDESEncryption(const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  BYTE keyWithParitySpaces[8];
  insertParitySpaces(keyWithParitySpaces, key);
  if (withInitRoundOutput)
      return des_xor_withInitOutput(keyWithParitySpaces, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
  return des_xor(keyWithParitySpaces, iv, inBuf, numInputBytes, outBuf, numOutputBytes);
}

/******************************************************************************
 * Basic cipher information
 ******************************************************************************/
void getBlackBoxDESInfo(int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  if (keySizeInBytes) *keySizeInBytes = 7; /* parity are not to be included in key */
  if (ivSizeInBytes) *ivSizeInBytes = DES_BLOCK_SIZE;
  if (suppressedBytes) *suppressedBytes = (DES_ROUNDS-1) * DES_BLOCK_SIZE;
  if (implicitBlockSizeInBytes) *implicitBlockSizeInBytes = DES_BLOCK_SIZE;
}

