/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "black_box_ciphers.h"
#include <stddef.h> /* NULL */
#include <pthread.h>
#include "assert_utils.h"
#include "black_box_trivium.h"
#include "black_box_rabbit.h"
#include "black_box_edon80.h"
#include "black_box_aes.h"
#include "black_box_des.h"
#include "black_box_grain.h"
#include "black_box_xtea.h"
#include "black_box_seed.h"
#include "black_box_present.h"
#include "black_box_sms4.h"
#include "black_box_camellia.h"
#include "black_box_rc6.h"
#include "black_box_hight.h"
//#include "black_box_clefia.h"
#include "black_box_hc128.h"
#include "black_box_hc256.h"
#include "black_box_mickey.h"
#include "black_box_salsa.h"
#include "black_box_sosemanuk.h"

typedef struct {
  const char *name;
  const bbCipherType type;
  const bbInfoFunction infoFunction;
  const bbEncryptionFunction encryptionFunction;
  const bbInfoFunctionBitsliced infoFunctionBitsliced;
  const bbEncryptionFunctionBitsliced encryptionFunctionBitsliced;
} bbInfo;

/*
 * The info function provides basic information for using the corresponding cryption function:
 *   key length
 *   iv length
 *   the number of bytes suppressed during initialization (S)
 *   the implicit block length of the implementation (B)
 *
 * Rules:
 *   NULL info function indicates absence of the corresponding cryption function
 *   An info function must be accompanied by a cryption function (do not add the info function only)
 *   A cipher may or may not provide a standard implementation
 *   A cipher may or may not provide a bitsliced implementation
 *   Supported output lengths:
 *     S + k * B, k = 0, 1, 2,... when including initialization rounds output
 *     k * B, k = 0, 1, 2,... when not including initialization rounds output
 *     A cipher implementation may support other output lengths, but only the ones specified above are guaranteed
 *   Output from initialization rounds should preferably correspond to reduced round versions of the algorithm
 *   Little endian machine may be assumed
 *   A CBC-implementation is required for block ciphers
 */
static const bbInfo info[kNumBlackBoxCiphers] = {
  { "Trivium", kStreamCipher, getBlackBoxTriviumInfo, blackBoxTriviumEncryption, getBlackBoxBitslicedTriviumInfo, blackBoxTriviumEncryptionBitsliced },
  { "Rabbit", kStreamCipher, getBlackBoxRabbitInfo, blackBoxRabbitEncryption, NULL, NULL },
  { "Edon80", kStreamCipher, getBlackBoxEdon80Info, blackBoxEdon80Encryption, NULL, NULL },
  { "AES-128", kBlockCipher, getBlackBoxAES128Info, blackBoxAES128Encryption, NULL, NULL },
  { "AES-256", kBlockCipher, getBlackBoxAES256Info, blackBoxAES256Encryption, NULL, NULL },
  { "DES", kBlockCipher, getBlackBoxDESInfo, blackBoxDESEncryption, NULL, NULL },
  { "Grain-128", kStreamCipher, getBlackBoxGrain128Info, blackBoxGrain128Encryption, getBlackBoxBitslicedGrain128Info, blackBoxGrain128EncryptionBitsliced },
  { "Grain v1", kStreamCipher, getBlackBoxGrainV1Info, blackBoxGrainV1Encryption, NULL, NULL },
  { "TEA", kBlockCipher, getBlackBoxTEAInfo, blackBoxTEAEncryption, NULL, NULL },
  { "XTEA", kBlockCipher, getBlackBoxXTEAInfo, blackBoxXTEAEncryption, NULL, NULL },
  { "SEED", kBlockCipher, getBlackBoxSEEDInfo, blackBoxSEEDEncryption, NULL, NULL },
  { "PRESENT", kBlockCipher, getBlackBoxPRESENTInfo, blackBoxPRESENTEncryption, NULL, NULL },
  { "SMS4", kBlockCipher, getBlackBoxSMS4Info, blackBoxSMS4Encryption, NULL, NULL },
  { "Camellia", kBlockCipher, getBlackBoxCamelliaInfo, blackBoxCamelliaEncryption, NULL, NULL },
  { "RC5", kBlockCipher, getBlackBoxRC5Info, blackBoxRC5Encryption, NULL, NULL },
  { "RC6", kBlockCipher, getBlackBoxRC6Info, blackBoxRC6Encryption, NULL, NULL },
  { "HIGHT", kBlockCipher, getBlackBoxHIGHTInfo, blackBoxHIGHTEncryption, NULL, NULL },
//  { "CLEFIA", kBlockCipher, getBlackBoxCLEFIAInfo, blackBoxCLEFIAEncryption, NULL, NULL },
  { "HC-128", kStreamCipher, getBlackBoxHC128Info, blackBoxHC128Encryption, NULL, NULL },
  { "HC-256", kStreamCipher, getBlackBoxHC256Info, blackBoxHC256Encryption, NULL, NULL },
  { "MICKEY v2", kStreamCipher, getBlackBoxMickeyV2Info, blackBoxMickeyV2Encryption, NULL, NULL },
  { "Salsa20/12", kStreamCipher, getBlackBoxSalsa2012Info, blackBoxSalsa2012Encryption, NULL, NULL },
  { "SOSEMANUK", kStreamCipher, getBlackBoxSosemanukInfo, blackBoxSosemanukEncryption, NULL, NULL }
};

/* cipher name */
const char *blackBoxCipherName(bbCipher cipher) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in getCipherName!");
  return info[cipher].name;
}

/* cipher type */
const bbCipherType blackBoxCipherType(bbCipher cipher) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in getCipherType!");
  return info[cipher].type;
}

/* implementation variants */
int blackBoxCipherProvidesStandardImplementation(bbCipher cipher) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in cipherProvidesStandardImplementation!");
  return info[cipher].encryptionFunction != NULL;
}

int blackBoxCipherProvidesBitslicedImplementation(bbCipher cipher) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in cipherProvidesStandardImplementation!");
  return info[cipher].encryptionFunctionBitsliced != NULL;
}

/* info function (standard implementation) */
void blackBoxInfo(bbCipher cipher, int *keySizeInBytes, int *ivSizeInBytes, int *suppressedBytes, int *implicitBlockSizeInBytes) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in blackBoxInfo!");
  if (blackBoxCipherProvidesStandardImplementation(cipher))
    info[cipher].infoFunction(keySizeInBytes, ivSizeInBytes, suppressedBytes, implicitBlockSizeInBytes);
}

/* info function (bitsliced implementation) */
void blackBoxInfoBitsliced(bbCipher cipher, int *keySizeInBits, int *ivSizeInBits, int *suppressedBits, int *implicitBlockSizeInBits) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in blackBoxInfoBitsliced!");
  if (blackBoxCipherProvidesBitslicedImplementation(cipher))
    info[cipher].infoFunctionBitsliced(keySizeInBits, ivSizeInBits, suppressedBits, implicitBlockSizeInBits);
}

/* keystream (standard implementation) */
int blackBoxEncrypt(bbCipher cipher, const BYTE *key, const BYTE *iv, const BYTE *inBuf, unsigned int numInputBytes, BYTE *outBuf, unsigned int numOutputBytes, int withInitRoundOutput) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in blackBoxEncrypt!");
  if (!blackBoxCipherProvidesStandardImplementation(cipher))
    return -1;
  return info[cipher].encryptionFunction(key, iv, inBuf, numInputBytes, outBuf, numOutputBytes, withInitRoundOutput);
}

/* keystream (bitsliced implementation) */
int blackBoxEncryptBitsliced(bbCipher cipher, const UINT64 *key, const UINT64 *iv, const UINT64 *inBuf, unsigned int numInputBits, UINT64 *outBuf, unsigned int numOutputBits, int withInitRoundOutput) {
  ASSERT(cipher >= 0 && cipher < kNumBlackBoxCiphers, "Invalid parameter in blackBoxEncryptBitsliced!");
  if (!blackBoxCipherProvidesBitslicedImplementation(cipher))
    return -1;
  return info[cipher].encryptionFunctionBitsliced(key, iv, inBuf, numInputBits, outBuf, numOutputBits, withInitRoundOutput);
}

