/*
 * Copyright (c) Paul Stankovski
 * Free for all non-commercial use unless this directive conflicts with
 * other applicable copyright statement(s), patent holders, laws or such.
 */
#include "bitslice_utils.h"
#include "bittwiddling.h"
#include "assert_utils.h"
#include "tmalloc.h"

void logBitslicedBufByLSB(FILE *logFile, const UINT64 *buf, int len, int gap) {
  BYTE *buf8 = (BYTE*)tmalloc((len + 7) / 8 * sizeof(BYTE));
  fromBitslicedBufByLSB(buf8, buf, len);
  logBuf(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, buf8, (len + 7) / 8 * sizeof(BYTE), gap / 8, 0);
  tfree(buf8);
}

void logBitslicedBufByWeight(FILE *logFile, const UINT64 *buf, int len, int gap) {
  BYTE *buf8 = (BYTE*)tmalloc((len + 7) / 8 * sizeof(BYTE));
  fromBitslicedBufByWeight(buf8, buf, len);
  logBuf(logFile, LOGSCREEN | LOGFILE | LOGFLUSH, buf8, (len + 7) / 8 * sizeof(BYTE), gap / 8, 0);
  tfree(buf8);
}

void fromBitslicedBufByWeight(BYTE *dst, const UINT64 *src, int numBits) {
  int k;

  tmemset(dst, 0, (numBits+7)/8*sizeof(BYTE));
  for (k=0; k<numBits; k++)
    if (weight(src[k]) & 1)
      setBufBit(dst, k, 1);
}

void fromBitslicedBufByLSB(BYTE *dst, const UINT64 *src, int numBits) {
  int k;

  tmemset(dst, 0, (numBits+7)/8*sizeof(BYTE));
  for (k=0; k<numBits; k++)
    if (src[k] & 1)
      setBufBit(dst, k, 1);
}

#define BUFBIT(buf, n) (buf[(n) / 8] & (1 << ((n) & 7)))
void toBitslicedBuf(UINT64 *dst, const BYTE *src, int numBits) {
  int k;

  for (k=0; k<numBits; k++)
    dst[k] = BUFBIT(src, k) ? U64C(0xFFFFFFFFFFFFFFFF) : 0;
}

void initializeBitslicedKeyValues(UINT64 *key, int keyLengthInBits, int fill, const BYTE *explicitKeyFill) {
  int k;
  if (explicitKeyFill) {
    toBitslicedBuf(key, explicitKeyFill, keyLengthInBits);
  } else {
    switch (fill) {
    case  FILL_ZEROS: for (k=0; k<keyLengthInBits; k++) key[k] = 0; break;
    case   FILL_ONES: for (k=0; k<keyLengthInBits; k++) key[k] = U64C(0xFFFFFFFFFFFFFFFF); break;
    case FILL_RANDOM: for (k=0; k<keyLengthInBits; k++) key[k] = rand() & 1 ? U64C(0xFFFFFFFFFFFFFFFF) : 0; break;
    default: ASSERT_ALWAYS("Unexpected fill value!");
    }
  }
}

void initializeBitslicedIvValues(UINT64 *iv, int ivLengthInBits, const int *ivBit, int ivFill, const BYTE *explicitIvFill, const UINT64 *key) {
  int k;
  if (explicitIvFill) {
    toBitslicedBuf(iv, explicitIvFill, ivLengthInBits);
  } else {
    switch (ivFill) {
    case  FILL_ZEROS: for (k=0; k<ivLengthInBits; k++) iv[k] = 0; break;
    case   FILL_ONES: for (k=0; k<ivLengthInBits; k++) iv[k] = U64C(0xFFFFFFFFFFFFFFFF); break;
    case FILL_RANDOM: for (k=0; k<ivLengthInBits; k++) iv[k] = rand() & 1 ? U64C(0xFFFFFFFFFFFFFFFF) : 0; break;
    case   FILL_COPY: for (k=0; k<ivLengthInBits; k++) iv[k] = key[k]; break; /* it is up to the called to make sure that the key buffer is large enough */
    default: ASSERT_ALWAYS("Unexpected fill value!");
    }
  }
  iv[ivBit[0]] = U64C(0xAAAAAAAAAAAAAAAA); /* bitslice over 6 iv bits */
  iv[ivBit[1]] = U64C(0xCCCCCCCCCCCCCCCC);
  iv[ivBit[2]] = U64C(0xF0F0F0F0F0F0F0F0);
  iv[ivBit[3]] = U64C(0xFF00FF00FF00FF00);
  iv[ivBit[4]] = U64C(0xFFFF0000FFFF0000);
  iv[ivBit[5]] = U64C(0xFFFFFFFF00000000);
}

void toBitslicedKeyIv(UINT64 *key, int keySizeInBits, const BYTE *key8,
                      UINT64 *iv, int ivSizeInBits, const BYTE *iv8,
                      int *keyBit, int numKeyBits,
                      int *ivBit, int numIvBits,
                      int *numBitsclicedKeyBits,
                      int *numBitsclicedIvBits) {

  toBitslicedBuf(key, key8, keySizeInBits);
  toBitslicedBuf(iv, iv8, ivSizeInBits);

  ASSERT(numKeyBits + numIvBits >= 6, "Unexpected bit lengths!");
  *numBitsclicedIvBits = numIvBits < 6 ? numIvBits : 6;
  *numBitsclicedKeyBits = 6 - *numBitsclicedIvBits;
  ASSERT(*numBitsclicedKeyBits + *numBitsclicedIvBits == 6, "Bitslice bit error!");

  /* set iv bitslice bits */
  switch (*numBitsclicedIvBits) {
  case 6: iv[ivBit[5]] = U64C(0xAAAAAAAAAAAAAAAA);
  case 5: iv[ivBit[4]] = U64C(0xCCCCCCCCCCCCCCCC);
  case 4: iv[ivBit[3]] = U64C(0xF0F0F0F0F0F0F0F0);
  case 3: iv[ivBit[2]] = U64C(0xFF00FF00FF00FF00);
  case 2: iv[ivBit[1]] = U64C(0xFFFF0000FFFF0000);
  case 1: iv[ivBit[0]] = U64C(0xFFFFFFFF00000000);
  }

  /* set key bitslice bits */
  switch (*numBitsclicedKeyBits) {
  case 6: key[keyBit[5]] = U64C(0xFFFFFFFF00000000);
  case 5: key[keyBit[4]] = U64C(0xFFFF0000FFFF0000);
  case 4: key[keyBit[3]] = U64C(0xFF00FF00FF00FF00);
  case 3: key[keyBit[2]] = U64C(0xF0F0F0F0F0F0F0F0);
  case 2: key[keyBit[1]] = U64C(0xCCCCCCCCCCCCCCCC);
  case 1: key[keyBit[0]] = U64C(0xAAAAAAAAAAAAAAAA);
  }
}

void setKeyBits(UINT64 *key, UINT64 j, int numKeyBits, const int *keyBit, int numBitsliceBits) {
  int k;
  for (k=0; k<numKeyBits-numBitsliceBits; k++)
    key[keyBit[k+numBitsliceBits]] = (j & ((UINT64)1 << k)) ? U64C(0xFFFFFFFFFFFFFFFF) : 0;
}

void setIvBits(UINT64 *iv, UINT64 i, int numIvBits, const int *ivBit, int numBitsliceBits) {
  int k;
  for (k=0; k<numIvBits-numBitsliceBits; k++)
    iv[ivBit[k+numBitsliceBits]] = (i & ((UINT64)1 << k)) ? U64C(0xFFFFFFFFFFFFFFFF) : 0;
}

